/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.s3accessgrants.plugin;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.regex.Pattern;
import software.amazon.awssdk.annotations.NotNull;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.identity.spi.AwsCredentialsIdentity;
import software.amazon.awssdk.identity.spi.IdentityProvider;
import software.amazon.awssdk.identity.spi.ResolveIdentityRequest;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.s3accessgrants.cache.S3AccessGrantsCachedCredentialsProvider;
import software.amazon.awssdk.s3accessgrants.plugin.internal.S3AccessGrantsStaticOperationToPermissionMapper;
import software.amazon.awssdk.s3accessgrants.plugin.internal.S3AccessGrantsUtils;
import software.amazon.awssdk.services.s3control.S3ControlAsyncClient;
import software.amazon.awssdk.services.s3control.model.Credentials;
import software.amazon.awssdk.services.s3control.model.GetDataAccessRequest;
import software.amazon.awssdk.services.s3control.model.Permission;
import software.amazon.awssdk.services.s3control.model.Privilege;
import software.amazon.awssdk.services.s3control.model.S3ControlException;
import software.amazon.awssdk.services.sts.StsAsyncClient;
import software.amazon.awssdk.services.sts.model.GetCallerIdentityResponse;
import software.amazon.awssdk.utils.Validate;

public class S3AccessGrantsIdentityProvider
implements IdentityProvider<AwsCredentialsIdentity> {
    private final IdentityProvider<? extends AwsCredentialsIdentity> credentialsProvider;
    private final Region region;
    private final Privilege privilege;
    private final Boolean isCacheEnabled;
    private final S3ControlAsyncClient s3control;
    private final StsAsyncClient stsAsyncClient;
    private final S3AccessGrantsStaticOperationToPermissionMapper permissionMapper;
    private final S3AccessGrantsCachedCredentialsProvider cache;
    private final boolean enableFallback;
    private final MetricPublisher metricsPublisher;
    private String CONTACT_TEAM_MESSAGE_TEMPLATE = "An internal exception has occurred. Valid %s was not passed to the %s. Please contact S3 access grants plugin team!";

    public S3AccessGrantsIdentityProvider(@NotNull IdentityProvider<? extends AwsCredentialsIdentity> credentialsProvider, @NotNull Region region, @NotNull StsAsyncClient stsAsyncClient, @NotNull Privilege privilege, @NotNull Boolean isCacheEnabled, @NotNull S3ControlAsyncClient s3ControlAsyncClient, @NotNull S3AccessGrantsCachedCredentialsProvider cache, @NotNull boolean enableFallback, @NotNull MetricPublisher metricsPublisher) {
        S3AccessGrantsUtils.argumentNotNull(credentialsProvider, "Expecting an Identity Provider to be specified while configuring S3Clients!");
        S3AccessGrantsUtils.argumentNotNull(region, "Expecting a region to be configured on the S3Clients!");
        S3AccessGrantsUtils.argumentNotNull(stsAsyncClient, String.format(this.CONTACT_TEAM_MESSAGE_TEMPLATE, "sts client", "identity provider"));
        this.credentialsProvider = credentialsProvider;
        this.region = region;
        this.stsAsyncClient = stsAsyncClient;
        this.privilege = privilege;
        this.isCacheEnabled = isCacheEnabled;
        this.s3control = s3ControlAsyncClient;
        this.permissionMapper = new S3AccessGrantsStaticOperationToPermissionMapper();
        this.cache = cache;
        this.enableFallback = enableFallback;
        this.metricsPublisher = metricsPublisher;
    }

    public Class<AwsCredentialsIdentity> identityType() {
        return AwsCredentialsIdentity.class;
    }

    public CompletableFuture<? extends AwsCredentialsIdentity> resolveIdentity(ResolveIdentityRequest resolveIdentityRequest) {
        CompletableFuture userCredentials = null;
        try {
            String accountId = this.getCallerAccountID().join().account();
            this.validateRequestParameters(resolveIdentityRequest, accountId, this.privilege, this.isCacheEnabled);
            userCredentials = this.credentialsProvider.resolveIdentity(resolveIdentityRequest);
            String S3Prefix = resolveIdentityRequest.property(S3AccessGrantsUtils.PREFIX_PROPERTY).toString();
            String operation = resolveIdentityRequest.property(S3AccessGrantsUtils.OPERATION_PROPERTY).toString();
            S3AccessGrantsUtils.logger.debug(() -> " Call access grants with the following request params! ");
            S3AccessGrantsUtils.logger.debug(() -> " S3Prefix : " + S3Prefix);
            S3AccessGrantsUtils.logger.debug(() -> " caller accountID : " + accountId);
            S3AccessGrantsUtils.logger.debug(() -> " operation : " + operation);
            Permission permission = this.permissionMapper.getPermission(operation);
            this.verifyIfValidBucket(S3Prefix);
            S3AccessGrantsUtils.logger.debug(() -> " permission : " + permission);
            return this.isCacheEnabled != false ? this.getCredentialsFromCache((AwsCredentialsIdentity)userCredentials.join(), permission, S3Prefix, accountId) : this.getCredentialsFromAccessGrants(this.createDataAccessRequest(accountId, S3Prefix, permission, this.privilege));
        }
        catch (SdkServiceException e) {
            if (this.shouldFallbackToDefaultCredentialsForThisCase(e.statusCode(), e.getCause()).booleanValue()) {
                return userCredentials;
            }
            throw e;
        }
    }

    protected void verifyIfValidBucket(String prefix) {
        if (prefix.split("/")[2].equals("null")) {
            throw new IllegalArgumentException("Please specify a valid bucket name for the operation!");
        }
    }

    private GetDataAccessRequest createDataAccessRequest(String accountId, String S3Prefix, Permission permission, Privilege privilege) {
        GetDataAccessRequest dataAccessRequest = (GetDataAccessRequest)GetDataAccessRequest.builder().accountId(accountId).target(S3Prefix).permission(permission).privilege(privilege).build();
        return dataAccessRequest;
    }

    CompletableFuture<? extends AwsCredentialsIdentity> getCredentialsFromAccessGrants(GetDataAccessRequest getDataAccessRequest) {
        S3AccessGrantsUtils.argumentNotNull(getDataAccessRequest, String.format(this.CONTACT_TEAM_MESSAGE_TEMPLATE, "request", "for calling access grants"));
        S3AccessGrantsUtils.logger.debug(() -> " Calling S3 Access Grants to validate access permissions!");
        return this.s3control.getDataAccess(getDataAccessRequest).thenApply(getDataAccessResponse -> {
            Credentials credentials = getDataAccessResponse.credentials();
            return AwsSessionCredentials.builder().accessKeyId(credentials.accessKeyId()).secretAccessKey(credentials.secretAccessKey()).sessionToken(credentials.sessionToken()).build();
        });
    }

    CompletableFuture<? extends AwsCredentialsIdentity> getCredentialsFromCache(AwsCredentialsIdentity credentials, Permission permission, String S3Prefix, String accountId) {
        try {
            CompletionStage completionStage = this.cache.getDataAccess(credentials, permission, S3Prefix, accountId).exceptionally(e -> {
                SdkServiceException throwableException = this.unwrapAndBuildException((Throwable)e);
                if (this.shouldFallbackToDefaultCredentialsForThisCase(throwableException.statusCode(), throwableException).booleanValue()) {
                    return credentials;
                }
                throw throwableException;
            });
            return completionStage;
        }
        catch (Exception e2) {
            SdkServiceException throwableException = this.unwrapAndBuildException(e2);
            if (this.shouldFallbackToDefaultCredentialsForThisCase(throwableException.statusCode(), throwableException).booleanValue()) {
                CompletableFuture<AwsCredentialsIdentity> completableFuture = CompletableFuture.supplyAsync(() -> credentials);
                return completableFuture;
            }
            throw throwableException;
        }
        finally {
            if (this.metricsPublisher != null) {
                this.publishMetrics();
            }
        }
    }

    private void publishMetrics() {
        try {
            this.metricsPublisher.publish(this.cache.getAccessGrantsMetrics().collect());
            this.metricsPublisher.close();
        }
        catch (Exception e) {
            S3AccessGrantsUtils.logger.warn(() -> "Something went wrong while publishing metrics using the metrics publisher. Please contact S3 access grants plugin team!");
            S3AccessGrantsUtils.logger.warn(() -> "cause for metrics publisher error : " + e.getMessage());
        }
    }

    private void validateRequestParameters(ResolveIdentityRequest resolveIdentityRequest, String accountId, Privilege privilege, Boolean isCacheEnabled) {
        S3AccessGrantsUtils.logger.debug(() -> "Validating the request parameters before sending a request to S3 Access grants!");
        S3AccessGrantsUtils.argumentNotNull(resolveIdentityRequest, String.format(this.CONTACT_TEAM_MESSAGE_TEMPLATE, "request", "identity provider"));
        S3AccessGrantsUtils.argumentNotNull(accountId, "Expecting account id to be configured on the plugin!");
        S3AccessGrantsUtils.argumentNotNull(privilege, String.format(this.CONTACT_TEAM_MESSAGE_TEMPLATE, "privilege", "identity provider"));
        S3AccessGrantsUtils.argumentNotNull(isCacheEnabled, String.format(this.CONTACT_TEAM_MESSAGE_TEMPLATE, "cache setting", "identity provider"));
        Pattern pattern = Pattern.compile("s3://[a-z0-9.-]*");
        S3AccessGrantsUtils.argumentNotNull(resolveIdentityRequest.property(S3AccessGrantsUtils.PREFIX_PROPERTY), String.format(this.CONTACT_TEAM_MESSAGE_TEMPLATE, "S3Prefix", "identity provider"));
        Validate.isTrue((boolean)pattern.matcher(resolveIdentityRequest.property(S3AccessGrantsUtils.PREFIX_PROPERTY).toString()).find(), (String)String.format(this.CONTACT_TEAM_MESSAGE_TEMPLATE, "S3Prefix", "identity provider"), (Object[])new Object[0]);
        S3AccessGrantsUtils.argumentNotNull(resolveIdentityRequest.property(S3AccessGrantsUtils.OPERATION_PROPERTY), String.format(this.CONTACT_TEAM_MESSAGE_TEMPLATE, "operation", "identity provider"));
        S3AccessGrantsUtils.logger.debug(() -> "Validation Complete. The request parameters can be forwarded to S3 Access grants!");
    }

    private SdkServiceException unwrapAndBuildException(Throwable e) {
        while (e.getCause() != null) {
            e = e.getCause();
        }
        if (e instanceof S3ControlException) {
            S3ControlException exc = (S3ControlException)e;
            return SdkServiceException.builder().statusCode(exc.statusCode()).message(exc.getMessage()).cause(e).build();
        }
        return SdkServiceException.builder().message(e.getMessage()).cause(e).build();
    }

    Boolean shouldFallbackToDefaultCredentialsForThisCase(int statusCode, Throwable cause) {
        if (this.enableFallback) {
            S3AccessGrantsUtils.logger.debug(() -> " Fall back enabled on the plugin! falling back to evaluate permission through policies!");
            return true;
        }
        if (statusCode == 404 && cause instanceof UnsupportedOperationException) {
            S3AccessGrantsUtils.logger.debug(() -> " Operation not supported by S3 access grants! fall back to evaluate permission through policies!");
            return true;
        }
        S3AccessGrantsUtils.logger.error(() -> " Fall back not enabled! An attempt will not be made to evaluate permissions through policies! " + cause.getMessage());
        return false;
    }

    CompletableFuture<GetCallerIdentityResponse> getCallerAccountID() {
        S3AccessGrantsUtils.logger.debug(() -> "requesting STS to fetch caller accountID!");
        return this.stsAsyncClient.getCallerIdentity();
    }
}

