/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wellarchitected.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.wellarchitected.WellArchitectedClient;
import software.amazon.awssdk.services.wellarchitected.internal.UserAgentUtils;
import software.amazon.awssdk.services.wellarchitected.model.ListCheckDetailsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListCheckDetailsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.wellarchitected.WellArchitectedClient#listCheckDetailsPaginator(software.amazon.awssdk.services.wellarchitected.model.ListCheckDetailsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.wellarchitected.model.ListCheckDetailsResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.wellarchitected.paginators.ListCheckDetailsIterable responses = client.listCheckDetailsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.wellarchitected.paginators.ListCheckDetailsIterable responses = client
 *             .listCheckDetailsPaginator(request);
 *     for (software.amazon.awssdk.services.wellarchitected.model.ListCheckDetailsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.wellarchitected.paginators.ListCheckDetailsIterable responses = client.listCheckDetailsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the paginator.
 * It only limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listCheckDetails(software.amazon.awssdk.services.wellarchitected.model.ListCheckDetailsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListCheckDetailsIterable implements SdkIterable<ListCheckDetailsResponse> {
    private final WellArchitectedClient client;

    private final ListCheckDetailsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListCheckDetailsIterable(WellArchitectedClient client, ListCheckDetailsRequest firstRequest) {
        this.client = client;
        this.firstRequest = UserAgentUtils.applyPaginatorUserAgent(firstRequest);
        this.nextPageFetcher = new ListCheckDetailsResponseFetcher();
    }

    @Override
    public Iterator<ListCheckDetailsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class ListCheckDetailsResponseFetcher implements SyncPageFetcher<ListCheckDetailsResponse> {
        @Override
        public boolean hasNextPage(ListCheckDetailsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListCheckDetailsResponse nextPage(ListCheckDetailsResponse previousPage) {
            if (previousPage == null) {
                return client.listCheckDetails(firstRequest);
            }
            return client.listCheckDetails(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
