/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.verifiedpermissions.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an entity that can be referenced in a Cedar policy.
 * </p>
 * <p>
 * This data type is used as one of the fields in the <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_EntitiesDefinition.html"
 * >EntitiesDefinition</a> structure.
 * </p>
 * <p>
 * <code>{ "identifier": { "entityType": "Photo", "entityId": "VacationPhoto94.jpg" }, "attributes": {}, "parents": [ { "entityType": "Album", "entityId": "alice_folder" } ] }</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EntityItem implements SdkPojo, Serializable, ToCopyableBuilder<EntityItem.Builder, EntityItem> {
    private static final SdkField<EntityIdentifier> IDENTIFIER_FIELD = SdkField
            .<EntityIdentifier> builder(MarshallingType.SDK_POJO).memberName("identifier").getter(getter(EntityItem::identifier))
            .setter(setter(Builder::identifier)).constructor(EntityIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identifier").build()).build();

    private static final SdkField<Map<String, AttributeValue>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, AttributeValue>> builder(MarshallingType.MAP)
            .memberName("attributes")
            .getter(getter(EntityItem::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<EntityIdentifier>> PARENTS_FIELD = SdkField
            .<List<EntityIdentifier>> builder(MarshallingType.LIST)
            .memberName("parents")
            .getter(getter(EntityItem::parents))
            .setter(setter(Builder::parents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EntityIdentifier> builder(MarshallingType.SDK_POJO)
                                            .constructor(EntityIdentifier::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTIFIER_FIELD,
            ATTRIBUTES_FIELD, PARENTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final EntityIdentifier identifier;

    private final Map<String, AttributeValue> attributes;

    private final List<EntityIdentifier> parents;

    private EntityItem(BuilderImpl builder) {
        this.identifier = builder.identifier;
        this.attributes = builder.attributes;
        this.parents = builder.parents;
    }

    /**
     * <p>
     * The identifier of the entity.
     * </p>
     * 
     * @return The identifier of the entity.
     */
    public final EntityIdentifier identifier() {
        return identifier;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of attributes for the entity.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return A list of attributes for the entity.
     */
    public final Map<String, AttributeValue> attributes() {
        return attributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parents property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParents() {
        return parents != null && !(parents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The parent entities in the hierarchy that contains the entity. A principal or resource entity can be defined with
     * at most 99 <i>transitive parents</i> per authorization request.
     * </p>
     * <p>
     * A transitive parent is an entity in the hierarchy of entities including all direct parents, and parents of
     * parents. For example, a user can be a member of 91 groups if one of those groups is a member of eight groups, for
     * a total of 100: one entity, 91 entity parents, and eight parents of parents.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParents} method.
     * </p>
     * 
     * @return The parent entities in the hierarchy that contains the entity. A principal or resource entity can be
     *         defined with at most 99 <i>transitive parents</i> per authorization request. </p>
     *         <p>
     *         A transitive parent is an entity in the hierarchy of entities including all direct parents, and parents
     *         of parents. For example, a user can be a member of 91 groups if one of those groups is a member of eight
     *         groups, for a total of 100: one entity, 91 entity parents, and eight parents of parents.
     */
    public final List<EntityIdentifier> parents() {
        return parents;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(identifier());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasParents() ? parents() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EntityItem)) {
            return false;
        }
        EntityItem other = (EntityItem) obj;
        return Objects.equals(identifier(), other.identifier()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributes(), other.attributes()) && hasParents() == other.hasParents()
                && Objects.equals(parents(), other.parents());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EntityItem").add("Identifier", identifier())
                .add("Attributes", hasAttributes() ? attributes() : null).add("Parents", hasParents() ? parents() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "identifier":
            return Optional.ofNullable(clazz.cast(identifier()));
        case "attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "parents":
            return Optional.ofNullable(clazz.cast(parents()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("identifier", IDENTIFIER_FIELD);
        map.put("attributes", ATTRIBUTES_FIELD);
        map.put("parents", PARENTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EntityItem, T> g) {
        return obj -> g.apply((EntityItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EntityItem> {
        /**
         * <p>
         * The identifier of the entity.
         * </p>
         * 
         * @param identifier
         *        The identifier of the entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifier(EntityIdentifier identifier);

        /**
         * <p>
         * The identifier of the entity.
         * </p>
         * This is a convenience method that creates an instance of the {@link EntityIdentifier.Builder} avoiding the
         * need to create one manually via {@link EntityIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EntityIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #identifier(EntityIdentifier)}.
         * 
         * @param identifier
         *        a consumer that will call methods on {@link EntityIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #identifier(EntityIdentifier)
         */
        default Builder identifier(Consumer<EntityIdentifier.Builder> identifier) {
            return identifier(EntityIdentifier.builder().applyMutation(identifier).build());
        }

        /**
         * <p>
         * A list of attributes for the entity.
         * </p>
         * 
         * @param attributes
         *        A list of attributes for the entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, AttributeValue> attributes);

        /**
         * <p>
         * The parent entities in the hierarchy that contains the entity. A principal or resource entity can be defined
         * with at most 99 <i>transitive parents</i> per authorization request.
         * </p>
         * <p>
         * A transitive parent is an entity in the hierarchy of entities including all direct parents, and parents of
         * parents. For example, a user can be a member of 91 groups if one of those groups is a member of eight groups,
         * for a total of 100: one entity, 91 entity parents, and eight parents of parents.
         * </p>
         * 
         * @param parents
         *        The parent entities in the hierarchy that contains the entity. A principal or resource entity can be
         *        defined with at most 99 <i>transitive parents</i> per authorization request. </p>
         *        <p>
         *        A transitive parent is an entity in the hierarchy of entities including all direct parents, and
         *        parents of parents. For example, a user can be a member of 91 groups if one of those groups is a
         *        member of eight groups, for a total of 100: one entity, 91 entity parents, and eight parents of
         *        parents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parents(Collection<EntityIdentifier> parents);

        /**
         * <p>
         * The parent entities in the hierarchy that contains the entity. A principal or resource entity can be defined
         * with at most 99 <i>transitive parents</i> per authorization request.
         * </p>
         * <p>
         * A transitive parent is an entity in the hierarchy of entities including all direct parents, and parents of
         * parents. For example, a user can be a member of 91 groups if one of those groups is a member of eight groups,
         * for a total of 100: one entity, 91 entity parents, and eight parents of parents.
         * </p>
         * 
         * @param parents
         *        The parent entities in the hierarchy that contains the entity. A principal or resource entity can be
         *        defined with at most 99 <i>transitive parents</i> per authorization request. </p>
         *        <p>
         *        A transitive parent is an entity in the hierarchy of entities including all direct parents, and
         *        parents of parents. For example, a user can be a member of 91 groups if one of those groups is a
         *        member of eight groups, for a total of 100: one entity, 91 entity parents, and eight parents of
         *        parents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parents(EntityIdentifier... parents);

        /**
         * <p>
         * The parent entities in the hierarchy that contains the entity. A principal or resource entity can be defined
         * with at most 99 <i>transitive parents</i> per authorization request.
         * </p>
         * <p>
         * A transitive parent is an entity in the hierarchy of entities including all direct parents, and parents of
         * parents. For example, a user can be a member of 91 groups if one of those groups is a member of eight groups,
         * for a total of 100: one entity, 91 entity parents, and eight parents of parents.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.verifiedpermissions.model.EntityIdentifier.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.verifiedpermissions.model.EntityIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.verifiedpermissions.model.EntityIdentifier.Builder#build()} is called
         * immediately and its result is passed to {@link #parents(List<EntityIdentifier>)}.
         * 
         * @param parents
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.verifiedpermissions.model.EntityIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parents(java.util.Collection<EntityIdentifier>)
         */
        Builder parents(Consumer<EntityIdentifier.Builder>... parents);
    }

    static final class BuilderImpl implements Builder {
        private EntityIdentifier identifier;

        private Map<String, AttributeValue> attributes = DefaultSdkAutoConstructMap.getInstance();

        private List<EntityIdentifier> parents = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EntityItem model) {
            identifier(model.identifier);
            attributes(model.attributes);
            parents(model.parents);
        }

        public final EntityIdentifier.Builder getIdentifier() {
            return identifier != null ? identifier.toBuilder() : null;
        }

        public final void setIdentifier(EntityIdentifier.BuilderImpl identifier) {
            this.identifier = identifier != null ? identifier.build() : null;
        }

        @Override
        public final Builder identifier(EntityIdentifier identifier) {
            this.identifier = identifier;
            return this;
        }

        public final Map<String, AttributeValue.Builder> getAttributes() {
            Map<String, AttributeValue.Builder> result = EntityAttributesCopier.copyToBuilder(this.attributes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setAttributes(Map<String, AttributeValue.BuilderImpl> attributes) {
            this.attributes = EntityAttributesCopier.copyFromBuilder(attributes);
        }

        @Override
        public final Builder attributes(Map<String, AttributeValue> attributes) {
            this.attributes = EntityAttributesCopier.copy(attributes);
            return this;
        }

        public final List<EntityIdentifier.Builder> getParents() {
            List<EntityIdentifier.Builder> result = ParentListCopier.copyToBuilder(this.parents);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParents(Collection<EntityIdentifier.BuilderImpl> parents) {
            this.parents = ParentListCopier.copyFromBuilder(parents);
        }

        @Override
        public final Builder parents(Collection<EntityIdentifier> parents) {
            this.parents = ParentListCopier.copy(parents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parents(EntityIdentifier... parents) {
            parents(Arrays.asList(parents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parents(Consumer<EntityIdentifier.Builder>... parents) {
            parents(Stream.of(parents).map(c -> EntityIdentifier.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public EntityItem build() {
            return new EntityItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
