/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamwrite;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.config.AwsClientOption;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRefreshCache;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRequest;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.timestreamwrite.model.AccessDeniedException;
import software.amazon.awssdk.services.timestreamwrite.model.ConflictException;
import software.amazon.awssdk.services.timestreamwrite.model.CreateDatabaseRequest;
import software.amazon.awssdk.services.timestreamwrite.model.CreateDatabaseResponse;
import software.amazon.awssdk.services.timestreamwrite.model.CreateTableRequest;
import software.amazon.awssdk.services.timestreamwrite.model.CreateTableResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DeleteDatabaseRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DeleteDatabaseResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DeleteTableRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DeleteTableResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeDatabaseRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeDatabaseResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeTableRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeTableResponse;
import software.amazon.awssdk.services.timestreamwrite.model.InternalServerException;
import software.amazon.awssdk.services.timestreamwrite.model.InvalidEndpointException;
import software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesRequest;
import software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ListTablesRequest;
import software.amazon.awssdk.services.timestreamwrite.model.ListTablesResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.timestreamwrite.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.timestreamwrite.model.RejectedRecordsException;
import software.amazon.awssdk.services.timestreamwrite.model.ResourceNotFoundException;
import software.amazon.awssdk.services.timestreamwrite.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.timestreamwrite.model.TagResourceRequest;
import software.amazon.awssdk.services.timestreamwrite.model.TagResourceResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ThrottlingException;
import software.amazon.awssdk.services.timestreamwrite.model.TimestreamWriteException;
import software.amazon.awssdk.services.timestreamwrite.model.TimestreamWriteRequest;
import software.amazon.awssdk.services.timestreamwrite.model.UntagResourceRequest;
import software.amazon.awssdk.services.timestreamwrite.model.UntagResourceResponse;
import software.amazon.awssdk.services.timestreamwrite.model.UpdateDatabaseRequest;
import software.amazon.awssdk.services.timestreamwrite.model.UpdateDatabaseResponse;
import software.amazon.awssdk.services.timestreamwrite.model.UpdateTableRequest;
import software.amazon.awssdk.services.timestreamwrite.model.UpdateTableResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ValidationException;
import software.amazon.awssdk.services.timestreamwrite.model.WriteRecordsRequest;
import software.amazon.awssdk.services.timestreamwrite.model.WriteRecordsResponse;
import software.amazon.awssdk.services.timestreamwrite.paginators.ListDatabasesPublisher;
import software.amazon.awssdk.services.timestreamwrite.paginators.ListTablesPublisher;
import software.amazon.awssdk.services.timestreamwrite.transform.CreateDatabaseRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.CreateTableRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DeleteDatabaseRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DeleteTableRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DescribeDatabaseRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DescribeTableRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.ListDatabasesRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.UpdateDatabaseRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.UpdateTableRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.WriteRecordsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TimestreamWriteAsyncClient}.
 *
 * @see TimestreamWriteAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTimestreamWriteAsyncClient implements TimestreamWriteAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTimestreamWriteAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private EndpointDiscoveryRefreshCache endpointDiscoveryCache;

    protected DefaultTimestreamWriteAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        if (clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED)) {
            this.endpointDiscoveryCache = EndpointDiscoveryRefreshCache.create(TimestreamWriteAsyncEndpointDiscoveryCacheLoader
                    .create(this));
            if (clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE) {
                log.warn("Endpoint discovery is enabled for this client, and an endpoint override was also specified. This will disable endpoint discovery for methods that require it, instead using the specified endpoint override. This may or may not be what you intended.");
            }
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new Timestream database. If the KMS key is not specified, the database will be encrypted with a
     * Timestream managed KMS key located in your account. Refer to <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk">AWS managed KMS
     * keys</a> for more info. Service quotas apply. For more information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a> in the
     * Timestream Developer Guide.
     * </p>
     *
     * @param createDatabaseRequest
     * @return A Java Future containing the result of the CreateDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Timestream was unable to process this request because it contains resource that
     *         already exists.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>ServiceQuotaExceededException Instance quota of resource exceeded for this account.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.CreateDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/CreateDatabase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatabaseResponse> createDatabase(CreateDatabaseRequest createDatabaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatabase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<CreateDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatabaseRequest, CreateDatabaseResponse>()
                            .withOperationName("CreateDatabase")
                            .withMarshaller(new CreateDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(createDatabaseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDatabaseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDatabaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The CreateTable operation adds a new table to an existing database in your account. In an AWS account, table
     * names must be at least unique within each Region if they are in the same database. You may have identical table
     * names in the same Region if the tables are in seperate databases. While creating the table, you must specify the
     * table name, database name, and the retention properties. Service quotas apply. For more information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a> in the
     * Timestream Developer Guide.
     * </p>
     *
     * @param createTableRequest
     * @return A Java Future containing the result of the CreateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Timestream was unable to process this request because it contains resource that
     *         already exists.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ServiceQuotaExceededException Instance quota of resource exceeded for this account.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.CreateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/CreateTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTableResponse> createTable(CreateTableRequest createTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<CreateTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTableRequest, CreateTableResponse>()
                            .withOperationName("CreateTable").withMarshaller(new CreateTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(createTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a given Timestream database. <i>This is an irreversible operation. After a database is deleted, the time
     * series data from its tables cannot be recovered.</i>
     * </p>
     * <p>
     * All tables in the database must be deleted first, or a ValidationException error will be thrown.
     * </p>
     * <p>
     * Due to the nature of distributed retries, the operation can return either success or a ResourceNotFoundException.
     * Clients should consider them equivalent.
     * </p>
     *
     * @param deleteDatabaseRequest
     * @return A Java Future containing the result of the DeleteDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.DeleteDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DeleteDatabase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatabaseResponse> deleteDatabase(DeleteDatabaseRequest deleteDatabaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatabase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DeleteDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatabaseRequest, DeleteDatabaseResponse>()
                            .withOperationName("DeleteDatabase")
                            .withMarshaller(new DeleteDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(deleteDatabaseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDatabaseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteDatabaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a given Timestream table. This is an irreversible operation. After a Timestream database table is
     * deleted, the time series data stored in the table cannot be recovered.
     * </p>
     * <p>
     * Due to the nature of distributed retries, the operation can return either success or a ResourceNotFoundException.
     * Clients should consider them equivalent.
     * </p>
     *
     * @param deleteTableRequest
     * @return A Java Future containing the result of the DeleteTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.DeleteTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DeleteTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTableResponse> deleteTable(DeleteTableRequest deleteTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DeleteTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTableRequest, DeleteTableResponse>()
                            .withOperationName("DeleteTable").withMarshaller(new DeleteTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(deleteTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the database, including the database name, time that the database was created, and the
     * total number of tables found within the database. Service quotas apply. For more information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a> in the
     * Timestream Developer Guide.
     * </p>
     *
     * @param describeDatabaseRequest
     * @return A Java Future containing the result of the DescribeDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.DescribeDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DescribeDatabase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatabaseResponse> describeDatabase(DescribeDatabaseRequest describeDatabaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDatabase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatabaseRequest, DescribeDatabaseResponse>()
                            .withOperationName("DescribeDatabase")
                            .withMarshaller(new DescribeDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeDatabaseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeDatabaseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeDatabaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * DescribeEndpoints returns a list of available endpoints to make Timestream API calls against. This API is
     * available through both Write and Query.
     * </p>
     * <p>
     * Because Timestream’s SDKs are designed to transparently work with the service’s architecture, including the
     * management and mapping of the service endpoints, <i>it is not recommended that you use this API unless</i>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Your application uses a programming language that does not yet have SDK support
     * </p>
     * </li>
     * <li>
     * <p>
     * You require better control over the client-side implementation
     * </p>
     * </li>
     * </ul>
     * <p>
     * For detailed information on how to use DescribeEndpoints, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/Using-API.endpoint-discovery.html">The
     * Endpoint Discovery Pattern and REST APIs</a>.
     * </p>
     *
     * @param describeEndpointsRequest
     * @return A Java Future containing the result of the DescribeEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DescribeEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEndpointsResponse> describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                            .withOperationName("DescribeEndpoints")
                            .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEndpointsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeEndpointsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the table, including the table name, database name, retention duration of the memory
     * store and the magnetic store. Service quotas apply. For more information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a> in the
     * Timestream Developer Guide.
     * </p>
     *
     * @param describeTableRequest
     * @return A Java Future containing the result of the DescribeTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.DescribeTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DescribeTable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTableResponse> describeTable(DescribeTableRequest describeTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTableRequest, DescribeTableResponse>()
                            .withOperationName("DescribeTable")
                            .withMarshaller(new DescribeTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of your Timestream databases. Service quotas apply. For more information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a> in the
     * Timestream Developer Guide.
     * </p>
     *
     * @param listDatabasesRequest
     * @return A Java Future containing the result of the ListDatabases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListDatabases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatabasesResponse> listDatabases(ListDatabasesRequest listDatabasesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatabases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatabasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListDatabasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatabasesRequest, ListDatabasesResponse>()
                            .withOperationName("ListDatabases")
                            .withMarshaller(new ListDatabasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listDatabasesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDatabasesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDatabasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of your Timestream databases. Service quotas apply. For more information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a> in the
     * Timestream Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatabases(software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListDatabasesPublisher publisher = client.listDatabasesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListDatabasesPublisher publisher = client.listDatabasesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatabases(software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesRequest)} operation.</b>
     * </p>
     *
     * @param listDatabasesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListDatabases"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDatabasesPublisher listDatabasesPaginator(ListDatabasesRequest listDatabasesRequest) {
        return new ListDatabasesPublisher(this, applyPaginatorUserAgent(listDatabasesRequest));
    }

    /**
     * <p>
     * A list of tables, along with the name, status and retention properties of each table.
     * </p>
     *
     * @param listTablesRequest
     * @return A Java Future containing the result of the ListTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListTables" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTablesResponse> listTables(ListTablesRequest listTablesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListTablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listTablesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTablesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListTablesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A list of tables, along with the name, status and retention properties of each table.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTables(software.amazon.awssdk.services.timestreamwrite.model.ListTablesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.timestreamwrite.model.ListTablesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.timestreamwrite.model.ListTablesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTables(software.amazon.awssdk.services.timestreamwrite.model.ListTablesRequest)} operation.</b>
     * </p>
     *
     * @param listTablesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListTables" target="_top">AWS
     *      API Documentation</a>
     */
    public ListTablesPublisher listTablesPaginator(ListTablesRequest listTablesRequest) {
        return new ListTablesPublisher(this, applyPaginatorUserAgent(listTablesRequest));
    }

    /**
     * <p>
     * List all tags on a Timestream resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associate a set of tags with a Timestream resource. You can then activate these user-defined tags so that they
     * appear on the Billing and Cost Management console for cost allocation tracking.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ServiceQuotaExceededException Instance quota of resource exceeded for this account.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association of tags from a Timestream resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException Instance quota of resource exceeded for this account.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the KMS key for an existing database. While updating the database, you must specify the database name
     * and the identifier of the new KMS key to be used (<code>KmsKeyId</code>). If there are any concurrent
     * <code>UpdateDatabase</code> requests, first writer wins.
     * </p>
     *
     * @param updateDatabaseRequest
     * @return A Java Future containing the result of the UpdateDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ServiceQuotaExceededException Instance quota of resource exceeded for this account.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.UpdateDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/UpdateDatabase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatabaseResponse> updateDatabase(UpdateDatabaseRequest updateDatabaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatabase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatabaseRequest, UpdateDatabaseResponse>()
                            .withOperationName("UpdateDatabase")
                            .withMarshaller(new UpdateDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateDatabaseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateDatabaseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateDatabaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the retention duration of the memory store and magnetic store for your Timestream table. Note that the
     * change in retention duration takes effect immediately. For example, if the retention period of the memory store
     * was initially set to 2 hours and then changed to 24 hours, the memory store will be capable of holding 24 hours
     * of data, but will be populated with 24 hours of data 22 hours after this change was made. Timestream does not
     * retrieve data from the magnetic store to populate the memory store.
     * </p>
     * <p>
     * Service quotas apply. For more information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a> in the
     * Timestream Developer Guide.
     * </p>
     *
     * @param updateTableRequest
     * @return A Java Future containing the result of the UpdateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.UpdateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/UpdateTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTableResponse> updateTable(UpdateTableRequest updateTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTableRequest, UpdateTableResponse>()
                            .withOperationName("UpdateTable").withMarshaller(new UpdateTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The WriteRecords operation enables you to write your time series data into Timestream. You can specify a single
     * data point or a batch of data points to be inserted into the system. Timestream offers you with a flexible schema
     * that auto detects the column names and data types for your Timestream tables based on the dimension names and
     * data types of the data points you specify when invoking writes into the database. Timestream support eventual
     * consistency read semantics. This means that when you query data immediately after writing a batch of data into
     * Timestream, the query results might not reflect the results of a recently completed write operation. The results
     * may also include some stale data. If you repeat the query request after a short time, the results should return
     * the latest data. Service quotas apply. For more information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a> in the
     * Timestream Developer Guide.
     * </p>
     *
     * @param writeRecordsRequest
     * @return A Java Future containing the result of the WriteRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException Too many requests were made by a user exceeding service quotas. The request was
     *         throttled.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>RejectedRecordsException WriteRecords would throw this exception in the following cases: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Records with duplicate data where there are multiple records with the same dimensions, timestamps, and
     *         measure names but different measure values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Records with timestamps that lie outside the retention duration of the memory store
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Records with dimensions or measures that exceed the Timestream defined limits.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Access Management</a>
     *         in the Timestream Developer Guide.</li>
     *         <li>InvalidEndpointException The requested endpoint was invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamWriteException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamWriteAsyncClient.WriteRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/WriteRecords" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<WriteRecordsResponse> writeRecords(WriteRecordsRequest writeRecordsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, writeRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "WriteRecords");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<WriteRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    WriteRecordsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<WriteRecordsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<WriteRecordsRequest, WriteRecordsResponse>()
                            .withOperationName("WriteRecords").withMarshaller(new WriteRecordsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(writeRecordsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = writeRecordsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<WriteRecordsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TimestreamWriteException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEndpointException")
                                .exceptionBuilderSupplier(InvalidEndpointException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RejectedRecordsException")
                                .exceptionBuilderSupplier(RejectedRecordsException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends TimestreamWriteRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
