/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.textract.model.AnalyzeDocumentRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeDocumentResponse;
import software.amazon.awssdk.services.textract.model.DetectDocumentTextRequest;
import software.amazon.awssdk.services.textract.model.DetectDocumentTextResponse;
import software.amazon.awssdk.services.textract.model.GetDocumentAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetDocumentAnalysisResponse;
import software.amazon.awssdk.services.textract.model.GetDocumentTextDetectionRequest;
import software.amazon.awssdk.services.textract.model.GetDocumentTextDetectionResponse;
import software.amazon.awssdk.services.textract.model.StartDocumentAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartDocumentAnalysisResponse;
import software.amazon.awssdk.services.textract.model.StartDocumentTextDetectionRequest;
import software.amazon.awssdk.services.textract.model.StartDocumentTextDetectionResponse;

/**
 * Service client for accessing Amazon Textract asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * Amazon Textract detects and analyzes text in documents and converts it into machine-readable text. This is the API
 * reference documentation for Amazon Textract.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface TextractAsyncClient extends SdkClient {
    String SERVICE_NAME = "textract";

    /**
     * Create a {@link TextractAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static TextractAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link TextractAsyncClient}.
     */
    static TextractAsyncClientBuilder builder() {
        return new DefaultTextractAsyncClientBuilder();
    }

    /**
     * <p>
     * Analyzes an input document for relationships between detected items.
     * </p>
     * <p>
     * The types of information returned are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of <code>FeatureTypes</code>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * You can choose which type of analysis to perform by specifying the <code>FeatureTypes</code> list.
     * </p>
     * <p>
     * The output is returned in a list of <code>Block</code> objects.
     * </p>
     * <p>
     * <code>AnalyzeDocument</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentAnalysis</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param analyzeDocumentRequest
     * @return A Java Future containing the result of the AnalyzeDocument operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>UnsupportedDocumentException The format of the input document isn't supported. Documents for
     *         synchronous operations can be in PNG or JPEG format. Documents for asynchronous operations can also be in
     *         PDF format.</li>
     *         <li>DocumentTooLargeException The document can't be processed because it's too large. The maximum
     *         document size for synchronous operations 10 MB. The maximum document size for asynchronous operations is
     *         500 MB for PDF files.</li>
     *         <li>BadDocumentException Amazon Textract isn't able to read the document. For more information on the
     *         document limits in Amazon Textract, see <a>limits</a>.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>HumanLoopQuotaExceededException Indicates you have exceeded the maximum number of active human in the
     *         loop workflows available</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.AnalyzeDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeDocument" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<AnalyzeDocumentResponse> analyzeDocument(AnalyzeDocumentRequest analyzeDocumentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Analyzes an input document for relationships between detected items.
     * </p>
     * <p>
     * The types of information returned are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of <code>FeatureTypes</code>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * You can choose which type of analysis to perform by specifying the <code>FeatureTypes</code> list.
     * </p>
     * <p>
     * The output is returned in a list of <code>Block</code> objects.
     * </p>
     * <p>
     * <code>AnalyzeDocument</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentAnalysis</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AnalyzeDocumentRequest.Builder} avoiding the need
     * to create one manually via {@link AnalyzeDocumentRequest#builder()}
     * </p>
     *
     * @param analyzeDocumentRequest
     *        A {@link Consumer} that will call methods on {@link AnalyzeDocumentRequest.Builder} to create a request.
     * @return A Java Future containing the result of the AnalyzeDocument operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>UnsupportedDocumentException The format of the input document isn't supported. Documents for
     *         synchronous operations can be in PNG or JPEG format. Documents for asynchronous operations can also be in
     *         PDF format.</li>
     *         <li>DocumentTooLargeException The document can't be processed because it's too large. The maximum
     *         document size for synchronous operations 10 MB. The maximum document size for asynchronous operations is
     *         500 MB for PDF files.</li>
     *         <li>BadDocumentException Amazon Textract isn't able to read the document. For more information on the
     *         document limits in Amazon Textract, see <a>limits</a>.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>HumanLoopQuotaExceededException Indicates you have exceeded the maximum number of active human in the
     *         loop workflows available</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.AnalyzeDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeDocument" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<AnalyzeDocumentResponse> analyzeDocument(
            Consumer<AnalyzeDocumentRequest.Builder> analyzeDocumentRequest) {
        return analyzeDocument(AnalyzeDocumentRequest.builder().applyMutation(analyzeDocumentRequest).build());
    }

    /**
     * <p>
     * Detects text in the input document. Amazon Textract can detect lines of text and the words that make up a line of
     * text. The input document must be an image in JPEG or PNG format. <code>DetectDocumentText</code> returns the
     * detected text in an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * <code>DetectDocumentText</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentTextDetection</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param detectDocumentTextRequest
     * @return A Java Future containing the result of the DetectDocumentText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>UnsupportedDocumentException The format of the input document isn't supported. Documents for
     *         synchronous operations can be in PNG or JPEG format. Documents for asynchronous operations can also be in
     *         PDF format.</li>
     *         <li>DocumentTooLargeException The document can't be processed because it's too large. The maximum
     *         document size for synchronous operations 10 MB. The maximum document size for asynchronous operations is
     *         500 MB for PDF files.</li>
     *         <li>BadDocumentException Amazon Textract isn't able to read the document. For more information on the
     *         document limits in Amazon Textract, see <a>limits</a>.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.DetectDocumentText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/DetectDocumentText" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DetectDocumentTextResponse> detectDocumentText(DetectDocumentTextRequest detectDocumentTextRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Detects text in the input document. Amazon Textract can detect lines of text and the words that make up a line of
     * text. The input document must be an image in JPEG or PNG format. <code>DetectDocumentText</code> returns the
     * detected text in an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * <code>DetectDocumentText</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentTextDetection</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DetectDocumentTextRequest.Builder} avoiding the
     * need to create one manually via {@link DetectDocumentTextRequest#builder()}
     * </p>
     *
     * @param detectDocumentTextRequest
     *        A {@link Consumer} that will call methods on {@link DetectDocumentTextRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DetectDocumentText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>UnsupportedDocumentException The format of the input document isn't supported. Documents for
     *         synchronous operations can be in PNG or JPEG format. Documents for asynchronous operations can also be in
     *         PDF format.</li>
     *         <li>DocumentTooLargeException The document can't be processed because it's too large. The maximum
     *         document size for synchronous operations 10 MB. The maximum document size for asynchronous operations is
     *         500 MB for PDF files.</li>
     *         <li>BadDocumentException Amazon Textract isn't able to read the document. For more information on the
     *         document limits in Amazon Textract, see <a>limits</a>.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.DetectDocumentText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/DetectDocumentText" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DetectDocumentTextResponse> detectDocumentText(
            Consumer<DetectDocumentTextRequest.Builder> detectDocumentTextRequest) {
        return detectDocumentText(DetectDocumentTextRequest.builder().applyMutation(detectDocumentTextRequest).build());
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <a>StartDocumentAnalysis</a>, which returns a job identifier (
     * <code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartDocumentAnalysis</code>. To get the results of the text-detection operation, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * <code>GetDocumentAnalysis</code> returns an array of <a>Block</a> objects. The following types of information are
     * returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of the <code>StartDocumentAnalysis</code> <code>FeatureTypes</code> input parameter).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * Use the <code>MaxResults</code> parameter to limit the number of blocks that are returned. If there are more
     * results than specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response
     * contains a pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param getDocumentAnalysisRequest
     * @return A Java Future containing the result of the GetDocumentAnalysis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InvalidJobIdException An invalid job identifier was passed to <a>GetDocumentAnalysis</a> or to
     *         <a>GetDocumentAnalysis</a>.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.GetDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetDocumentAnalysisResponse> getDocumentAnalysis(
            GetDocumentAnalysisRequest getDocumentAnalysisRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <a>StartDocumentAnalysis</a>, which returns a job identifier (
     * <code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartDocumentAnalysis</code>. To get the results of the text-detection operation, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * <code>GetDocumentAnalysis</code> returns an array of <a>Block</a> objects. The following types of information are
     * returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of the <code>StartDocumentAnalysis</code> <code>FeatureTypes</code> input parameter).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * Use the <code>MaxResults</code> parameter to limit the number of blocks that are returned. If there are more
     * results than specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response
     * contains a pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDocumentAnalysisRequest.Builder} avoiding the
     * need to create one manually via {@link GetDocumentAnalysisRequest#builder()}
     * </p>
     *
     * @param getDocumentAnalysisRequest
     *        A {@link Consumer} that will call methods on {@link GetDocumentAnalysisRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetDocumentAnalysis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InvalidJobIdException An invalid job identifier was passed to <a>GetDocumentAnalysis</a> or to
     *         <a>GetDocumentAnalysis</a>.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.GetDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetDocumentAnalysisResponse> getDocumentAnalysis(
            Consumer<GetDocumentAnalysisRequest.Builder> getDocumentAnalysisRequest) {
        return getDocumentAnalysis(GetDocumentAnalysisRequest.builder().applyMutation(getDocumentAnalysisRequest).build());
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that detects text in a document. Amazon Textract
     * can detect lines of text and the words that make up a line of text.
     * </p>
     * <p>
     * You start asynchronous text detection by calling <a>StartDocumentTextDetection</a>, which returns a job
     * identifier (<code>JobId</code>). When the text detection operation finishes, Amazon Textract publishes a
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial
     * call to <code>StartDocumentTextDetection</code>. To get the results of the text-detection operation, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentTextDetection</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * <code>GetDocumentTextDetection</code> returns an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentTextDetection</code>, and populate the <code>NextToken</code> request parameter with the token
     * value that's returned from the previous call to <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param getDocumentTextDetectionRequest
     * @return A Java Future containing the result of the GetDocumentTextDetection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InvalidJobIdException An invalid job identifier was passed to <a>GetDocumentAnalysis</a> or to
     *         <a>GetDocumentAnalysis</a>.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.GetDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetDocumentTextDetectionResponse> getDocumentTextDetection(
            GetDocumentTextDetectionRequest getDocumentTextDetectionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that detects text in a document. Amazon Textract
     * can detect lines of text and the words that make up a line of text.
     * </p>
     * <p>
     * You start asynchronous text detection by calling <a>StartDocumentTextDetection</a>, which returns a job
     * identifier (<code>JobId</code>). When the text detection operation finishes, Amazon Textract publishes a
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial
     * call to <code>StartDocumentTextDetection</code>. To get the results of the text-detection operation, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentTextDetection</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * <code>GetDocumentTextDetection</code> returns an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentTextDetection</code>, and populate the <code>NextToken</code> request parameter with the token
     * value that's returned from the previous call to <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDocumentTextDetectionRequest.Builder} avoiding
     * the need to create one manually via {@link GetDocumentTextDetectionRequest#builder()}
     * </p>
     *
     * @param getDocumentTextDetectionRequest
     *        A {@link Consumer} that will call methods on {@link GetDocumentTextDetectionRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetDocumentTextDetection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InvalidJobIdException An invalid job identifier was passed to <a>GetDocumentAnalysis</a> or to
     *         <a>GetDocumentAnalysis</a>.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.GetDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetDocumentTextDetectionResponse> getDocumentTextDetection(
            Consumer<GetDocumentTextDetectionRequest.Builder> getDocumentTextDetectionRequest) {
        return getDocumentTextDetection(GetDocumentTextDetectionRequest.builder().applyMutation(getDocumentTextDetectionRequest)
                .build());
    }

    /**
     * <p>
     * Starts the asynchronous analysis of an input document for relationships between detected items such as key-value
     * pairs, tables, and selection elements.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> can analyze text in documents that are in JPEG, PNG, and PDF format. The
     * documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name and file name
     * of the document.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * from the initial call to <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param startDocumentAnalysisRequest
     * @return A Java Future containing the result of the StartDocumentAnalysis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>InvalidKmsKeyException Indicates you do not have decrypt permissions with the KMS key entered, or the
     *         KMS key was entered incorrectly.</li>
     *         <li>UnsupportedDocumentException The format of the input document isn't supported. Documents for
     *         synchronous operations can be in PNG or JPEG format. Documents for asynchronous operations can also be in
     *         PDF format.</li>
     *         <li>DocumentTooLargeException The document can't be processed because it's too large. The maximum
     *         document size for synchronous operations 10 MB. The maximum document size for asynchronous operations is
     *         500 MB for PDF files.</li>
     *         <li>BadDocumentException Amazon Textract isn't able to read the document. For more information on the
     *         document limits in Amazon Textract, see <a>limits</a>.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>IdempotentParameterMismatchException A <code>ClientRequestToken</code> input parameter was reused
     *         with an operation, but at least one of the other input parameters is different from the previous call to
     *         the operation.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>LimitExceededException An Amazon Textract service limit was exceeded. For example, if you start too
     *         many asynchronous jobs concurrently, calls to start operations (<code>StartDocumentTextDetection</code>,
     *         for example) raise a LimitExceededException exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Textract service limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.StartDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartDocumentAnalysisResponse> startDocumentAnalysis(
            StartDocumentAnalysisRequest startDocumentAnalysisRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts the asynchronous analysis of an input document for relationships between detected items such as key-value
     * pairs, tables, and selection elements.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> can analyze text in documents that are in JPEG, PNG, and PDF format. The
     * documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name and file name
     * of the document.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * from the initial call to <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartDocumentAnalysisRequest.Builder} avoiding the
     * need to create one manually via {@link StartDocumentAnalysisRequest#builder()}
     * </p>
     *
     * @param startDocumentAnalysisRequest
     *        A {@link Consumer} that will call methods on {@link StartDocumentAnalysisRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the StartDocumentAnalysis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>InvalidKmsKeyException Indicates you do not have decrypt permissions with the KMS key entered, or the
     *         KMS key was entered incorrectly.</li>
     *         <li>UnsupportedDocumentException The format of the input document isn't supported. Documents for
     *         synchronous operations can be in PNG or JPEG format. Documents for asynchronous operations can also be in
     *         PDF format.</li>
     *         <li>DocumentTooLargeException The document can't be processed because it's too large. The maximum
     *         document size for synchronous operations 10 MB. The maximum document size for asynchronous operations is
     *         500 MB for PDF files.</li>
     *         <li>BadDocumentException Amazon Textract isn't able to read the document. For more information on the
     *         document limits in Amazon Textract, see <a>limits</a>.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>IdempotentParameterMismatchException A <code>ClientRequestToken</code> input parameter was reused
     *         with an operation, but at least one of the other input parameters is different from the previous call to
     *         the operation.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>LimitExceededException An Amazon Textract service limit was exceeded. For example, if you start too
     *         many asynchronous jobs concurrently, calls to start operations (<code>StartDocumentTextDetection</code>,
     *         for example) raise a LimitExceededException exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Textract service limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.StartDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartDocumentAnalysisResponse> startDocumentAnalysis(
            Consumer<StartDocumentAnalysisRequest.Builder> startDocumentAnalysisRequest) {
        return startDocumentAnalysis(StartDocumentAnalysisRequest.builder().applyMutation(startDocumentAnalysisRequest).build());
    }

    /**
     * <p>
     * Starts the asynchronous detection of text in a document. Amazon Textract can detect lines of text and the words
     * that make up a line of text.
     * </p>
     * <p>
     * <code>StartDocumentTextDetection</code> can analyze text in documents that are in JPEG, PNG, and PDF format. The
     * documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name and file name
     * of the document.
     * </p>
     * <p>
     * <code>StartTextDetection</code> returns a job identifier (<code>JobId</code>) that you use to get the results of
     * the operation. When text detection is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text detection operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentTextDetection</a>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param startDocumentTextDetectionRequest
     * @return A Java Future containing the result of the StartDocumentTextDetection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>InvalidKmsKeyException Indicates you do not have decrypt permissions with the KMS key entered, or the
     *         KMS key was entered incorrectly.</li>
     *         <li>UnsupportedDocumentException The format of the input document isn't supported. Documents for
     *         synchronous operations can be in PNG or JPEG format. Documents for asynchronous operations can also be in
     *         PDF format.</li>
     *         <li>DocumentTooLargeException The document can't be processed because it's too large. The maximum
     *         document size for synchronous operations 10 MB. The maximum document size for asynchronous operations is
     *         500 MB for PDF files.</li>
     *         <li>BadDocumentException Amazon Textract isn't able to read the document. For more information on the
     *         document limits in Amazon Textract, see <a>limits</a>.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>IdempotentParameterMismatchException A <code>ClientRequestToken</code> input parameter was reused
     *         with an operation, but at least one of the other input parameters is different from the previous call to
     *         the operation.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>LimitExceededException An Amazon Textract service limit was exceeded. For example, if you start too
     *         many asynchronous jobs concurrently, calls to start operations (<code>StartDocumentTextDetection</code>,
     *         for example) raise a LimitExceededException exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Textract service limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.StartDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartDocumentTextDetectionResponse> startDocumentTextDetection(
            StartDocumentTextDetectionRequest startDocumentTextDetectionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts the asynchronous detection of text in a document. Amazon Textract can detect lines of text and the words
     * that make up a line of text.
     * </p>
     * <p>
     * <code>StartDocumentTextDetection</code> can analyze text in documents that are in JPEG, PNG, and PDF format. The
     * documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name and file name
     * of the document.
     * </p>
     * <p>
     * <code>StartTextDetection</code> returns a job identifier (<code>JobId</code>) that you use to get the results of
     * the operation. When text detection is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text detection operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentTextDetection</a>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartDocumentTextDetectionRequest.Builder} avoiding
     * the need to create one manually via {@link StartDocumentTextDetectionRequest#builder()}
     * </p>
     *
     * @param startDocumentTextDetectionRequest
     *        A {@link Consumer} that will call methods on {@link StartDocumentTextDetectionRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the StartDocumentTextDetection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An input parameter violated a constraint. For example, in synchronous
     *         operations, an <code>InvalidParameterException</code> exception occurs when neither of the
     *         <code>S3Object</code> or <code>Bytes</code> values are supplied in the <code>Document</code> request
     *         parameter. Validate your parameter before calling the API operation again.</li>
     *         <li>InvalidS3ObjectException Amazon Textract is unable to access the S3 object that's specified in the
     *         request. for more information, <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to Amazon
     *         S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     *         </li>
     *         <li>InvalidKmsKeyException Indicates you do not have decrypt permissions with the KMS key entered, or the
     *         KMS key was entered incorrectly.</li>
     *         <li>UnsupportedDocumentException The format of the input document isn't supported. Documents for
     *         synchronous operations can be in PNG or JPEG format. Documents for asynchronous operations can also be in
     *         PDF format.</li>
     *         <li>DocumentTooLargeException The document can't be processed because it's too large. The maximum
     *         document size for synchronous operations 10 MB. The maximum document size for asynchronous operations is
     *         500 MB for PDF files.</li>
     *         <li>BadDocumentException Amazon Textract isn't able to read the document. For more information on the
     *         document limits in Amazon Textract, see <a>limits</a>.</li>
     *         <li>AccessDeniedException You aren't authorized to perform the action. Use the Amazon Resource Name (ARN)
     *         of an authorized user or IAM role to perform the operation.</li>
     *         <li>ProvisionedThroughputExceededException The number of requests exceeded your throughput limit. If you
     *         want to increase this limit, contact Amazon Textract.</li>
     *         <li>InternalServerErrorException Amazon Textract experienced a service issue. Try your call again.</li>
     *         <li>IdempotentParameterMismatchException A <code>ClientRequestToken</code> input parameter was reused
     *         with an operation, but at least one of the other input parameters is different from the previous call to
     *         the operation.</li>
     *         <li>ThrottlingException Amazon Textract is temporarily unable to process the request. Try your call
     *         again.</li>
     *         <li>LimitExceededException An Amazon Textract service limit was exceeded. For example, if you start too
     *         many asynchronous jobs concurrently, calls to start operations (<code>StartDocumentTextDetection</code>,
     *         for example) raise a LimitExceededException exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Textract service limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TextractException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TextractAsyncClient.StartDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartDocumentTextDetectionResponse> startDocumentTextDetection(
            Consumer<StartDocumentTextDetectionRequest.Builder> startDocumentTextDetectionRequest) {
        return startDocumentTextDetection(StartDocumentTextDetectionRequest.builder()
                .applyMutation(startDocumentTextDetectionRequest).build());
    }
}
