/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.support.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.support.transform.CommunicationMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A communication associated with an AWS Support case. The communication consists of the case ID, the message body,
 * attachment information, the account email address, and the date and time of the communication.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Communication implements StructuredPojo, ToCopyableBuilder<Communication.Builder, Communication> {
    private final String caseId;

    private final String body;

    private final String submittedBy;

    private final String timeCreated;

    private final List<AttachmentDetails> attachmentSet;

    private Communication(BuilderImpl builder) {
        this.caseId = builder.caseId;
        this.body = builder.body;
        this.submittedBy = builder.submittedBy;
        this.timeCreated = builder.timeCreated;
        this.attachmentSet = builder.attachmentSet;
    }

    /**
     * <p>
     * The AWS Support case ID requested or returned in the call. The case ID is an alphanumeric string formatted as
     * shown in this example: case-<i>12345678910-2013-c4c1d2bf33c5cf47</i>
     * </p>
     * 
     * @return The AWS Support case ID requested or returned in the call. The case ID is an alphanumeric string
     *         formatted as shown in this example: case-<i>12345678910-2013-c4c1d2bf33c5cf47</i>
     */
    public String caseId() {
        return caseId;
    }

    /**
     * <p>
     * The text of the communication between the customer and AWS Support.
     * </p>
     * 
     * @return The text of the communication between the customer and AWS Support.
     */
    public String body() {
        return body;
    }

    /**
     * <p>
     * The email address of the account that submitted the AWS Support case.
     * </p>
     * 
     * @return The email address of the account that submitted the AWS Support case.
     */
    public String submittedBy() {
        return submittedBy;
    }

    /**
     * <p>
     * The time the communication was created.
     * </p>
     * 
     * @return The time the communication was created.
     */
    public String timeCreated() {
        return timeCreated;
    }

    /**
     * <p>
     * Information about the attachments to the case communication.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the attachments to the case communication.
     */
    public List<AttachmentDetails> attachmentSet() {
        return attachmentSet;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(caseId());
        hashCode = 31 * hashCode + Objects.hashCode(body());
        hashCode = 31 * hashCode + Objects.hashCode(submittedBy());
        hashCode = 31 * hashCode + Objects.hashCode(timeCreated());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentSet());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Communication)) {
            return false;
        }
        Communication other = (Communication) obj;
        return Objects.equals(caseId(), other.caseId()) && Objects.equals(body(), other.body())
                && Objects.equals(submittedBy(), other.submittedBy()) && Objects.equals(timeCreated(), other.timeCreated())
                && Objects.equals(attachmentSet(), other.attachmentSet());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (caseId() != null) {
            sb.append("CaseId: ").append(caseId()).append(",");
        }
        if (body() != null) {
            sb.append("Body: ").append(body()).append(",");
        }
        if (submittedBy() != null) {
            sb.append("SubmittedBy: ").append(submittedBy()).append(",");
        }
        if (timeCreated() != null) {
            sb.append("TimeCreated: ").append(timeCreated()).append(",");
        }
        if (attachmentSet() != null) {
            sb.append("AttachmentSet: ").append(attachmentSet()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "caseId":
            return Optional.of(clazz.cast(caseId()));
        case "body":
            return Optional.of(clazz.cast(body()));
        case "submittedBy":
            return Optional.of(clazz.cast(submittedBy()));
        case "timeCreated":
            return Optional.of(clazz.cast(timeCreated()));
        case "attachmentSet":
            return Optional.of(clazz.cast(attachmentSet()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        CommunicationMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Communication> {
        /**
         * <p>
         * The AWS Support case ID requested or returned in the call. The case ID is an alphanumeric string formatted as
         * shown in this example: case-<i>12345678910-2013-c4c1d2bf33c5cf47</i>
         * </p>
         * 
         * @param caseId
         *        The AWS Support case ID requested or returned in the call. The case ID is an alphanumeric string
         *        formatted as shown in this example: case-<i>12345678910-2013-c4c1d2bf33c5cf47</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder caseId(String caseId);

        /**
         * <p>
         * The text of the communication between the customer and AWS Support.
         * </p>
         * 
         * @param body
         *        The text of the communication between the customer and AWS Support.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(String body);

        /**
         * <p>
         * The email address of the account that submitted the AWS Support case.
         * </p>
         * 
         * @param submittedBy
         *        The email address of the account that submitted the AWS Support case.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder submittedBy(String submittedBy);

        /**
         * <p>
         * The time the communication was created.
         * </p>
         * 
         * @param timeCreated
         *        The time the communication was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeCreated(String timeCreated);

        /**
         * <p>
         * Information about the attachments to the case communication.
         * </p>
         * 
         * @param attachmentSet
         *        Information about the attachments to the case communication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentSet(Collection<AttachmentDetails> attachmentSet);

        /**
         * <p>
         * Information about the attachments to the case communication.
         * </p>
         * 
         * @param attachmentSet
         *        Information about the attachments to the case communication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentSet(AttachmentDetails... attachmentSet);
    }

    static final class BuilderImpl implements Builder {
        private String caseId;

        private String body;

        private String submittedBy;

        private String timeCreated;

        private List<AttachmentDetails> attachmentSet;

        private BuilderImpl() {
        }

        private BuilderImpl(Communication model) {
            caseId(model.caseId);
            body(model.body);
            submittedBy(model.submittedBy);
            timeCreated(model.timeCreated);
            attachmentSet(model.attachmentSet);
        }

        public final String getCaseId() {
            return caseId;
        }

        @Override
        public final Builder caseId(String caseId) {
            this.caseId = caseId;
            return this;
        }

        public final void setCaseId(String caseId) {
            this.caseId = caseId;
        }

        public final String getBody() {
            return body;
        }

        @Override
        public final Builder body(String body) {
            this.body = body;
            return this;
        }

        public final void setBody(String body) {
            this.body = body;
        }

        public final String getSubmittedBy() {
            return submittedBy;
        }

        @Override
        public final Builder submittedBy(String submittedBy) {
            this.submittedBy = submittedBy;
            return this;
        }

        public final void setSubmittedBy(String submittedBy) {
            this.submittedBy = submittedBy;
        }

        public final String getTimeCreated() {
            return timeCreated;
        }

        @Override
        public final Builder timeCreated(String timeCreated) {
            this.timeCreated = timeCreated;
            return this;
        }

        public final void setTimeCreated(String timeCreated) {
            this.timeCreated = timeCreated;
        }

        public final Collection<AttachmentDetails.Builder> getAttachmentSet() {
            return attachmentSet != null ? attachmentSet.stream().map(AttachmentDetails::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder attachmentSet(Collection<AttachmentDetails> attachmentSet) {
            this.attachmentSet = AttachmentSetCopier.copy(attachmentSet);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentSet(AttachmentDetails... attachmentSet) {
            attachmentSet(Arrays.asList(attachmentSet));
            return this;
        }

        public final void setAttachmentSet(Collection<AttachmentDetails.BuilderImpl> attachmentSet) {
            this.attachmentSet = AttachmentSetCopier.copyFromBuilder(attachmentSet);
        }

        @Override
        public Communication build() {
            return new Communication(this);
        }
    }
}
