/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssooidc;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenRequest;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenResponse;
import software.amazon.awssdk.services.ssooidc.model.RegisterClientRequest;
import software.amazon.awssdk.services.ssooidc.model.RegisterClientResponse;
import software.amazon.awssdk.services.ssooidc.model.StartDeviceAuthorizationRequest;
import software.amazon.awssdk.services.ssooidc.model.StartDeviceAuthorizationResponse;

/**
 * Service client for accessing SSO OIDC asynchronously. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * AWS IAM Identity Center (successor to AWS Single Sign-On) OpenID Connect (OIDC) is a web service that enables a
 * client (such as AWS CLI or a native application) to register with IAM Identity Center. The service also enables the
 * client to fetch the user’s access token upon successful authentication and authorization with IAM Identity Center.
 * </p>
 * <note>
 * <p>
 * Although AWS Single Sign-On was renamed, the <code>sso</code> and <code>identitystore</code> API namespaces will
 * continue to retain their original name for backward compatibility purposes. For more information, see <a
 * href="https://docs.aws.amazon.com/singlesignon/latest/userguide/what-is.html#renamed">IAM Identity Center rename</a>.
 * </p>
 * </note>
 * <p>
 * <b>Considerations for Using This Guide</b>
 * </p>
 * <p>
 * Before you begin using this guide, we recommend that you first review the following important information about how
 * the IAM Identity Center OIDC service works.
 * </p>
 * <ul>
 * <li>
 * <p>
 * The IAM Identity Center OIDC service currently implements only the portions of the OAuth 2.0 Device Authorization
 * Grant standard (<a href="https://tools.ietf.org/html/rfc8628">https://tools.ietf.org/html/rfc8628</a>) that are
 * necessary to enable single sign-on authentication with the AWS CLI. Support for other OIDC flows frequently needed
 * for native applications, such as Authorization Code Flow (+ PKCE), will be addressed in future releases.
 * </p>
 * </li>
 * <li>
 * <p>
 * The service emits only OIDC access tokens, such that obtaining a new token (For example, token refresh) requires
 * explicit user re-authentication.
 * </p>
 * </li>
 * <li>
 * <p>
 * The access tokens provided by this service grant access to all AWS account entitlements assigned to an IAM Identity
 * Center user, not just a particular application.
 * </p>
 * </li>
 * <li>
 * <p>
 * The documentation in this guide does not describe the mechanism to convert the access token into AWS Auth (“sigv4”)
 * credentials for use with IAM-protected AWS service endpoints. For more information, see <a
 * href="https://docs.aws.amazon.com/singlesignon/latest/PortalAPIReference/API_GetRoleCredentials.html"
 * >GetRoleCredentials</a> in the <i>IAM Identity Center Portal API Reference Guide</i>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For general information about IAM Identity Center, see <a
 * href="https://docs.aws.amazon.com/singlesignon/latest/userguide/what-is.html">What is IAM Identity Center?</a> in the
 * <i>IAM Identity Center User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface SsoOidcAsyncClient extends SdkClient {
    String SERVICE_NAME = "awsssooidc";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "oidc";

    /**
     * <p>
     * Creates and returns an access token for the authorized client. The access token issued will be used to fetch
     * short-term credentials for the assigned roles in the AWS account.
     * </p>
     *
     * @param createTokenRequest
     * @return A Java Future containing the result of the CreateToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidClientException Indicates that the <code>clientId</code> or <code>clientSecret</code> in the
     *         request is invalid. For example, this can occur when a client sends an incorrect <code>clientId</code> or
     *         an expired <code>clientSecret</code>.</li>
     *         <li>InvalidGrantException Indicates that a request contains an invalid grant. This can occur if a client
     *         makes a <a>CreateToken</a> request with an invalid grant type.</li>
     *         <li>UnauthorizedClientException Indicates that the client is not currently authorized to make the
     *         request. This can happen when a <code>clientId</code> is not issued for a public client.</li>
     *         <li>UnsupportedGrantTypeException Indicates that the grant type in the request is not supported by the
     *         service.</li>
     *         <li>InvalidScopeException Indicates that the scope provided in the request is invalid.</li>
     *         <li>AuthorizationPendingException Indicates that a request to authorize a client with an access user
     *         session token is pending.</li>
     *         <li>SlowDownException Indicates that the client is making the request too frequently and is more than the
     *         service can handle.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ExpiredTokenException Indicates that the token issued by the service is expired and is no longer
     *         valid.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.CreateToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/CreateToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateTokenResponse> createToken(CreateTokenRequest createTokenRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates and returns an access token for the authorized client. The access token issued will be used to fetch
     * short-term credentials for the assigned roles in the AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateTokenRequest.Builder} avoiding the need to
     * create one manually via {@link CreateTokenRequest#builder()}
     * </p>
     *
     * @param createTokenRequest
     *        A {@link Consumer} that will call methods on {@link CreateTokenRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidClientException Indicates that the <code>clientId</code> or <code>clientSecret</code> in the
     *         request is invalid. For example, this can occur when a client sends an incorrect <code>clientId</code> or
     *         an expired <code>clientSecret</code>.</li>
     *         <li>InvalidGrantException Indicates that a request contains an invalid grant. This can occur if a client
     *         makes a <a>CreateToken</a> request with an invalid grant type.</li>
     *         <li>UnauthorizedClientException Indicates that the client is not currently authorized to make the
     *         request. This can happen when a <code>clientId</code> is not issued for a public client.</li>
     *         <li>UnsupportedGrantTypeException Indicates that the grant type in the request is not supported by the
     *         service.</li>
     *         <li>InvalidScopeException Indicates that the scope provided in the request is invalid.</li>
     *         <li>AuthorizationPendingException Indicates that a request to authorize a client with an access user
     *         session token is pending.</li>
     *         <li>SlowDownException Indicates that the client is making the request too frequently and is more than the
     *         service can handle.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ExpiredTokenException Indicates that the token issued by the service is expired and is no longer
     *         valid.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.CreateToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/CreateToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateTokenResponse> createToken(Consumer<CreateTokenRequest.Builder> createTokenRequest) {
        return createToken(CreateTokenRequest.builder().applyMutation(createTokenRequest).build());
    }

    /**
     * <p>
     * Registers a client with IAM Identity Center. This allows clients to initiate device authorization. The output
     * should be persisted for reuse through many authentication requests.
     * </p>
     *
     * @param registerClientRequest
     * @return A Java Future containing the result of the RegisterClient operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidScopeException Indicates that the scope provided in the request is invalid.</li>
     *         <li>InvalidClientMetadataException Indicates that the client information sent in the request during
     *         registration is invalid.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.RegisterClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/RegisterClient" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<RegisterClientResponse> registerClient(RegisterClientRequest registerClientRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Registers a client with IAM Identity Center. This allows clients to initiate device authorization. The output
     * should be persisted for reuse through many authentication requests.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RegisterClientRequest.Builder} avoiding the need to
     * create one manually via {@link RegisterClientRequest#builder()}
     * </p>
     *
     * @param registerClientRequest
     *        A {@link Consumer} that will call methods on {@link RegisterClientRequest.Builder} to create a request.
     * @return A Java Future containing the result of the RegisterClient operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidScopeException Indicates that the scope provided in the request is invalid.</li>
     *         <li>InvalidClientMetadataException Indicates that the client information sent in the request during
     *         registration is invalid.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.RegisterClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/RegisterClient" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<RegisterClientResponse> registerClient(Consumer<RegisterClientRequest.Builder> registerClientRequest) {
        return registerClient(RegisterClientRequest.builder().applyMutation(registerClientRequest).build());
    }

    /**
     * <p>
     * Initiates device authorization by requesting a pair of verification codes from the authorization service.
     * </p>
     *
     * @param startDeviceAuthorizationRequest
     * @return A Java Future containing the result of the StartDeviceAuthorization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidClientException Indicates that the <code>clientId</code> or <code>clientSecret</code> in the
     *         request is invalid. For example, this can occur when a client sends an incorrect <code>clientId</code> or
     *         an expired <code>clientSecret</code>.</li>
     *         <li>UnauthorizedClientException Indicates that the client is not currently authorized to make the
     *         request. This can happen when a <code>clientId</code> is not issued for a public client.</li>
     *         <li>SlowDownException Indicates that the client is making the request too frequently and is more than the
     *         service can handle.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.StartDeviceAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/StartDeviceAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartDeviceAuthorizationResponse> startDeviceAuthorization(
            StartDeviceAuthorizationRequest startDeviceAuthorizationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Initiates device authorization by requesting a pair of verification codes from the authorization service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartDeviceAuthorizationRequest.Builder} avoiding
     * the need to create one manually via {@link StartDeviceAuthorizationRequest#builder()}
     * </p>
     *
     * @param startDeviceAuthorizationRequest
     *        A {@link Consumer} that will call methods on {@link StartDeviceAuthorizationRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the StartDeviceAuthorization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidClientException Indicates that the <code>clientId</code> or <code>clientSecret</code> in the
     *         request is invalid. For example, this can occur when a client sends an incorrect <code>clientId</code> or
     *         an expired <code>clientSecret</code>.</li>
     *         <li>UnauthorizedClientException Indicates that the client is not currently authorized to make the
     *         request. This can happen when a <code>clientId</code> is not issued for a public client.</li>
     *         <li>SlowDownException Indicates that the client is making the request too frequently and is more than the
     *         service can handle.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.StartDeviceAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/StartDeviceAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartDeviceAuthorizationResponse> startDeviceAuthorization(
            Consumer<StartDeviceAuthorizationRequest.Builder> startDeviceAuthorizationRequest) {
        return startDeviceAuthorization(StartDeviceAuthorizationRequest.builder().applyMutation(startDeviceAuthorizationRequest)
                .build());
    }

    /**
     * Create a {@link SsoOidcAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static SsoOidcAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link SsoOidcAsyncClient}.
     */
    static SsoOidcAsyncClientBuilder builder() {
        return new DefaultSsoOidcAsyncClientBuilder();
    }
}
