/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssooidc;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.ssooidc.model.AccessDeniedException;
import software.amazon.awssdk.services.ssooidc.model.AuthorizationPendingException;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenRequest;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenResponse;
import software.amazon.awssdk.services.ssooidc.model.ExpiredTokenException;
import software.amazon.awssdk.services.ssooidc.model.InternalServerException;
import software.amazon.awssdk.services.ssooidc.model.InvalidClientException;
import software.amazon.awssdk.services.ssooidc.model.InvalidClientMetadataException;
import software.amazon.awssdk.services.ssooidc.model.InvalidGrantException;
import software.amazon.awssdk.services.ssooidc.model.InvalidRequestException;
import software.amazon.awssdk.services.ssooidc.model.InvalidScopeException;
import software.amazon.awssdk.services.ssooidc.model.RegisterClientRequest;
import software.amazon.awssdk.services.ssooidc.model.RegisterClientResponse;
import software.amazon.awssdk.services.ssooidc.model.SlowDownException;
import software.amazon.awssdk.services.ssooidc.model.SsoOidcException;
import software.amazon.awssdk.services.ssooidc.model.StartDeviceAuthorizationRequest;
import software.amazon.awssdk.services.ssooidc.model.StartDeviceAuthorizationResponse;
import software.amazon.awssdk.services.ssooidc.model.UnauthorizedClientException;
import software.amazon.awssdk.services.ssooidc.model.UnsupportedGrantTypeException;

/**
 * Service client for accessing SSO OIDC. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * AWS Single Sign-On (SSO) OpenID Connect (OIDC) is a web service that enables a client (such as AWS CLI or a native
 * application) to register with AWS SSO. The service also enables the client to fetch the user’s access token upon
 * successful authentication and authorization with AWS SSO. This service conforms with the OAuth 2.0 based
 * implementation of the device authorization grant standard (<a
 * href="https://tools.ietf.org/html/rfc8628">https://tools.ietf.org/html/rfc8628</a>).
 * </p>
 * <p>
 * For general information about AWS SSO, see <a
 * href="https://docs.aws.amazon.com/singlesignon/latest/userguide/what-is.html">What is AWS Single Sign-On?</a> in the
 * <i>AWS SSO User Guide</i>.
 * </p>
 * <p>
 * This API reference guide describes the AWS SSO OIDC operations that you can call programatically and includes
 * detailed information on data types and errors.
 * </p>
 * <note>
 * <p>
 * AWS provides SDKs that consist of libraries and sample code for various programming languages and platforms such as
 * Java, Ruby, .Net, iOS, and Android. The SDKs provide a convenient way to create programmatic access to AWS SSO and
 * other AWS services. For more information about the AWS SDKs, including how to download and install them, see <a
 * href="http://aws.amazon.com/tools/">Tools for Amazon Web Services</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public interface SsoOidcClient extends SdkClient {
    String SERVICE_NAME = "awsssooidc";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "oidc";

    /**
     * Create a {@link SsoOidcClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static SsoOidcClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link SsoOidcClient}.
     */
    static SsoOidcClientBuilder builder() {
        return new DefaultSsoOidcClientBuilder();
    }

    /**
     * <p>
     * Creates and returns an access token for the authorized client. The access token issued will be used to fetch
     * short-term credentials for the assigned roles in the AWS account.
     * </p>
     *
     * @param createTokenRequest
     * @return Result of the CreateToken operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidClientException
     *         Indicates that the <code>clientId</code> or <code>clientSecret</code> in the request is invalid. For
     *         example, this can occur when a client sends an incorrect <code>clientId</code> or an expired
     *         <code>clientSecret</code>.
     * @throws InvalidGrantException
     *         Indicates that a request contains an invalid grant. This can occur if a client makes a <a>CreateToken</a>
     *         request with an invalid grant type.
     * @throws UnauthorizedClientException
     *         Indicates that the client is not currently authorized to make the request. This can happen when a
     *         <code>clientId</code> is not issued for a public client.
     * @throws UnsupportedGrantTypeException
     *         Indicates that the grant type in the request is not supported by the service.
     * @throws InvalidScopeException
     *         Indicates that the scope provided in the request is invalid.
     * @throws AuthorizationPendingException
     *         Indicates that a request to authorize a client with an access user session token is pending.
     * @throws SlowDownException
     *         Indicates that the client is making the request too frequently and is more than the service can handle.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ExpiredTokenException
     *         Indicates that the token issued by the service is expired and is no longer valid.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.CreateToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/CreateToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateTokenResponse createToken(CreateTokenRequest createTokenRequest) throws InvalidRequestException,
            InvalidClientException, InvalidGrantException, UnauthorizedClientException, UnsupportedGrantTypeException,
            InvalidScopeException, AuthorizationPendingException, SlowDownException, AccessDeniedException,
            ExpiredTokenException, InternalServerException, AwsServiceException, SdkClientException, SsoOidcException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates and returns an access token for the authorized client. The access token issued will be used to fetch
     * short-term credentials for the assigned roles in the AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateTokenRequest.Builder} avoiding the need to
     * create one manually via {@link CreateTokenRequest#builder()}
     * </p>
     *
     * @param createTokenRequest
     *        A {@link Consumer} that will call methods on {@link CreateTokenRequest.Builder} to create a request.
     * @return Result of the CreateToken operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidClientException
     *         Indicates that the <code>clientId</code> or <code>clientSecret</code> in the request is invalid. For
     *         example, this can occur when a client sends an incorrect <code>clientId</code> or an expired
     *         <code>clientSecret</code>.
     * @throws InvalidGrantException
     *         Indicates that a request contains an invalid grant. This can occur if a client makes a <a>CreateToken</a>
     *         request with an invalid grant type.
     * @throws UnauthorizedClientException
     *         Indicates that the client is not currently authorized to make the request. This can happen when a
     *         <code>clientId</code> is not issued for a public client.
     * @throws UnsupportedGrantTypeException
     *         Indicates that the grant type in the request is not supported by the service.
     * @throws InvalidScopeException
     *         Indicates that the scope provided in the request is invalid.
     * @throws AuthorizationPendingException
     *         Indicates that a request to authorize a client with an access user session token is pending.
     * @throws SlowDownException
     *         Indicates that the client is making the request too frequently and is more than the service can handle.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ExpiredTokenException
     *         Indicates that the token issued by the service is expired and is no longer valid.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.CreateToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/CreateToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateTokenResponse createToken(Consumer<CreateTokenRequest.Builder> createTokenRequest)
            throws InvalidRequestException, InvalidClientException, InvalidGrantException, UnauthorizedClientException,
            UnsupportedGrantTypeException, InvalidScopeException, AuthorizationPendingException, SlowDownException,
            AccessDeniedException, ExpiredTokenException, InternalServerException, AwsServiceException, SdkClientException,
            SsoOidcException {
        return createToken(CreateTokenRequest.builder().applyMutation(createTokenRequest).build());
    }

    /**
     * <p>
     * Registers a client with AWS SSO. This allows clients to initiate device authorization. The output should be
     * persisted for reuse through many authentication requests.
     * </p>
     *
     * @param registerClientRequest
     * @return Result of the RegisterClient operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidScopeException
     *         Indicates that the scope provided in the request is invalid.
     * @throws InvalidClientMetadataException
     *         Indicates that the client information sent in the request during registration is invalid.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.RegisterClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/RegisterClient" target="_top">AWS API
     *      Documentation</a>
     */
    default RegisterClientResponse registerClient(RegisterClientRequest registerClientRequest) throws InvalidRequestException,
            InvalidScopeException, InvalidClientMetadataException, InternalServerException, AwsServiceException,
            SdkClientException, SsoOidcException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Registers a client with AWS SSO. This allows clients to initiate device authorization. The output should be
     * persisted for reuse through many authentication requests.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RegisterClientRequest.Builder} avoiding the need to
     * create one manually via {@link RegisterClientRequest#builder()}
     * </p>
     *
     * @param registerClientRequest
     *        A {@link Consumer} that will call methods on {@link RegisterClientRequest.Builder} to create a request.
     * @return Result of the RegisterClient operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidScopeException
     *         Indicates that the scope provided in the request is invalid.
     * @throws InvalidClientMetadataException
     *         Indicates that the client information sent in the request during registration is invalid.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.RegisterClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/RegisterClient" target="_top">AWS API
     *      Documentation</a>
     */
    default RegisterClientResponse registerClient(Consumer<RegisterClientRequest.Builder> registerClientRequest)
            throws InvalidRequestException, InvalidScopeException, InvalidClientMetadataException, InternalServerException,
            AwsServiceException, SdkClientException, SsoOidcException {
        return registerClient(RegisterClientRequest.builder().applyMutation(registerClientRequest).build());
    }

    /**
     * <p>
     * Initiates device authorization by requesting a pair of verification codes from the authorization service.
     * </p>
     *
     * @param startDeviceAuthorizationRequest
     * @return Result of the StartDeviceAuthorization operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidClientException
     *         Indicates that the <code>clientId</code> or <code>clientSecret</code> in the request is invalid. For
     *         example, this can occur when a client sends an incorrect <code>clientId</code> or an expired
     *         <code>clientSecret</code>.
     * @throws UnauthorizedClientException
     *         Indicates that the client is not currently authorized to make the request. This can happen when a
     *         <code>clientId</code> is not issued for a public client.
     * @throws SlowDownException
     *         Indicates that the client is making the request too frequently and is more than the service can handle.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.StartDeviceAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/StartDeviceAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    default StartDeviceAuthorizationResponse startDeviceAuthorization(
            StartDeviceAuthorizationRequest startDeviceAuthorizationRequest) throws InvalidRequestException,
            InvalidClientException, UnauthorizedClientException, SlowDownException, InternalServerException, AwsServiceException,
            SdkClientException, SsoOidcException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Initiates device authorization by requesting a pair of verification codes from the authorization service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartDeviceAuthorizationRequest.Builder} avoiding
     * the need to create one manually via {@link StartDeviceAuthorizationRequest#builder()}
     * </p>
     *
     * @param startDeviceAuthorizationRequest
     *        A {@link Consumer} that will call methods on {@link StartDeviceAuthorizationRequest.Builder} to create a
     *        request.
     * @return Result of the StartDeviceAuthorization operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidClientException
     *         Indicates that the <code>clientId</code> or <code>clientSecret</code> in the request is invalid. For
     *         example, this can occur when a client sends an incorrect <code>clientId</code> or an expired
     *         <code>clientSecret</code>.
     * @throws UnauthorizedClientException
     *         Indicates that the client is not currently authorized to make the request. This can happen when a
     *         <code>clientId</code> is not issued for a public client.
     * @throws SlowDownException
     *         Indicates that the client is making the request too frequently and is more than the service can handle.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.StartDeviceAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/StartDeviceAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    default StartDeviceAuthorizationResponse startDeviceAuthorization(
            Consumer<StartDeviceAuthorizationRequest.Builder> startDeviceAuthorizationRequest) throws InvalidRequestException,
            InvalidClientException, UnauthorizedClientException, SlowDownException, InternalServerException, AwsServiceException,
            SdkClientException, SsoOidcException {
        return startDeviceAuthorization(StartDeviceAuthorizationRequest.builder().applyMutation(startDeviceAuthorizationRequest)
                .build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }
}
