/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The service setting data structure.
 * </p>
 * <p>
 * <code>ServiceSetting</code> is an account-level setting for an AWS service. This setting defines how a user interacts
 * with or uses a service or a feature of a service. For example, if an AWS service charges money to the account based
 * on feature or service usage, then the AWS service team might create a default setting of "false". This means the user
 * can't use this feature unless they change the setting to "true" and intentionally opt in for a paid feature.
 * </p>
 * <p>
 * Services map a <code>SettingId</code> object to a setting value. AWS services teams define the default value for a
 * <code>SettingId</code>. You can't create a new <code>SettingId</code>, but you can overwrite the default value if you
 * have the <code>ssm:UpdateServiceSetting</code> permission for the setting. Use the <a>UpdateServiceSetting</a> API
 * action to change the default setting. Or, use the <a>ResetServiceSetting</a> to change the value back to the original
 * value defined by the AWS service team.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceSetting implements SdkPojo, Serializable, ToCopyableBuilder<ServiceSetting.Builder, ServiceSetting> {
    private static final SdkField<String> SETTING_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ServiceSetting::settingId)).setter(setter(Builder::settingId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SettingId").build()).build();

    private static final SdkField<String> SETTING_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ServiceSetting::settingValue)).setter(setter(Builder::settingValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SettingValue").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ServiceSetting::lastModifiedDate)).setter(setter(Builder::lastModifiedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedDate").build()).build();

    private static final SdkField<String> LAST_MODIFIED_USER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ServiceSetting::lastModifiedUser)).setter(setter(Builder::lastModifiedUser))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedUser").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ServiceSetting::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ServiceSetting::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SETTING_ID_FIELD,
            SETTING_VALUE_FIELD, LAST_MODIFIED_DATE_FIELD, LAST_MODIFIED_USER_FIELD, ARN_FIELD, STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String settingId;

    private final String settingValue;

    private final Instant lastModifiedDate;

    private final String lastModifiedUser;

    private final String arn;

    private final String status;

    private ServiceSetting(BuilderImpl builder) {
        this.settingId = builder.settingId;
        this.settingValue = builder.settingValue;
        this.lastModifiedDate = builder.lastModifiedDate;
        this.lastModifiedUser = builder.lastModifiedUser;
        this.arn = builder.arn;
        this.status = builder.status;
    }

    /**
     * <p>
     * The ID of the service setting.
     * </p>
     * 
     * @return The ID of the service setting.
     */
    public String settingId() {
        return settingId;
    }

    /**
     * <p>
     * The value of the service setting.
     * </p>
     * 
     * @return The value of the service setting.
     */
    public String settingValue() {
        return settingValue;
    }

    /**
     * <p>
     * The last time the service setting was modified.
     * </p>
     * 
     * @return The last time the service setting was modified.
     */
    public Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    /**
     * <p>
     * The ARN of the last modified user. This field is populated only if the setting value was overwritten.
     * </p>
     * 
     * @return The ARN of the last modified user. This field is populated only if the setting value was overwritten.
     */
    public String lastModifiedUser() {
        return lastModifiedUser;
    }

    /**
     * <p>
     * The ARN of the service setting.
     * </p>
     * 
     * @return The ARN of the service setting.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The status of the service setting. The value can be Default, Customized or PendingUpdate.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Default: The current setting uses a default value provisioned by the AWS service team.
     * </p>
     * </li>
     * <li>
     * <p>
     * Customized: The current setting use a custom value specified by the customer.
     * </p>
     * </li>
     * <li>
     * <p>
     * PendingUpdate: The current setting uses a default or custom value, but a setting change request is pending
     * approval.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The status of the service setting. The value can be Default, Customized or PendingUpdate.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Default: The current setting uses a default value provisioned by the AWS service team.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Customized: The current setting use a custom value specified by the customer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PendingUpdate: The current setting uses a default or custom value, but a setting change request is
     *         pending approval.
     *         </p>
     *         </li>
     */
    public String status() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(settingId());
        hashCode = 31 * hashCode + Objects.hashCode(settingValue());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedUser());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceSetting)) {
            return false;
        }
        ServiceSetting other = (ServiceSetting) obj;
        return Objects.equals(settingId(), other.settingId()) && Objects.equals(settingValue(), other.settingValue())
                && Objects.equals(lastModifiedDate(), other.lastModifiedDate())
                && Objects.equals(lastModifiedUser(), other.lastModifiedUser()) && Objects.equals(arn(), other.arn())
                && Objects.equals(status(), other.status());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ServiceSetting").add("SettingId", settingId()).add("SettingValue", settingValue())
                .add("LastModifiedDate", lastModifiedDate()).add("LastModifiedUser", lastModifiedUser()).add("ARN", arn())
                .add("Status", status()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SettingId":
            return Optional.ofNullable(clazz.cast(settingId()));
        case "SettingValue":
            return Optional.ofNullable(clazz.cast(settingValue()));
        case "LastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        case "LastModifiedUser":
            return Optional.ofNullable(clazz.cast(lastModifiedUser()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ServiceSetting, T> g) {
        return obj -> g.apply((ServiceSetting) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceSetting> {
        /**
         * <p>
         * The ID of the service setting.
         * </p>
         * 
         * @param settingId
         *        The ID of the service setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settingId(String settingId);

        /**
         * <p>
         * The value of the service setting.
         * </p>
         * 
         * @param settingValue
         *        The value of the service setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settingValue(String settingValue);

        /**
         * <p>
         * The last time the service setting was modified.
         * </p>
         * 
         * @param lastModifiedDate
         *        The last time the service setting was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);

        /**
         * <p>
         * The ARN of the last modified user. This field is populated only if the setting value was overwritten.
         * </p>
         * 
         * @param lastModifiedUser
         *        The ARN of the last modified user. This field is populated only if the setting value was overwritten.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedUser(String lastModifiedUser);

        /**
         * <p>
         * The ARN of the service setting.
         * </p>
         * 
         * @param arn
         *        The ARN of the service setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The status of the service setting. The value can be Default, Customized or PendingUpdate.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Default: The current setting uses a default value provisioned by the AWS service team.
         * </p>
         * </li>
         * <li>
         * <p>
         * Customized: The current setting use a custom value specified by the customer.
         * </p>
         * </li>
         * <li>
         * <p>
         * PendingUpdate: The current setting uses a default or custom value, but a setting change request is pending
         * approval.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the service setting. The value can be Default, Customized or PendingUpdate.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Default: The current setting uses a default value provisioned by the AWS service team.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Customized: The current setting use a custom value specified by the customer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PendingUpdate: The current setting uses a default or custom value, but a setting change request is
         *        pending approval.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);
    }

    static final class BuilderImpl implements Builder {
        private String settingId;

        private String settingValue;

        private Instant lastModifiedDate;

        private String lastModifiedUser;

        private String arn;

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceSetting model) {
            settingId(model.settingId);
            settingValue(model.settingValue);
            lastModifiedDate(model.lastModifiedDate);
            lastModifiedUser(model.lastModifiedUser);
            arn(model.arn);
            status(model.status);
        }

        public final String getSettingId() {
            return settingId;
        }

        @Override
        public final Builder settingId(String settingId) {
            this.settingId = settingId;
            return this;
        }

        public final void setSettingId(String settingId) {
            this.settingId = settingId;
        }

        public final String getSettingValue() {
            return settingValue;
        }

        @Override
        public final Builder settingValue(String settingValue) {
            this.settingValue = settingValue;
            return this;
        }

        public final void setSettingValue(String settingValue) {
            this.settingValue = settingValue;
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        public final String getLastModifiedUser() {
            return lastModifiedUser;
        }

        @Override
        public final Builder lastModifiedUser(String lastModifiedUser) {
            this.lastModifiedUser = lastModifiedUser;
            return this;
        }

        public final void setLastModifiedUser(String lastModifiedUser) {
            this.lastModifiedUser = lastModifiedUser;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public ServiceSetting build() {
            return new ServiceSetting(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
