/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class RegisterTaskWithMaintenanceWindowRequest extends SsmRequest implements
        ToCopyableBuilder<RegisterTaskWithMaintenanceWindowRequest.Builder, RegisterTaskWithMaintenanceWindowRequest> {
    private static final SdkField<String> WINDOW_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WindowId").getter(getter(RegisterTaskWithMaintenanceWindowRequest::windowId))
            .setter(setter(Builder::windowId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowId").build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(RegisterTaskWithMaintenanceWindowRequest::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskArn").getter(getter(RegisterTaskWithMaintenanceWindowRequest::taskArn))
            .setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskArn").build()).build();

    private static final SdkField<String> SERVICE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceRoleArn").getter(getter(RegisterTaskWithMaintenanceWindowRequest::serviceRoleArn))
            .setter(setter(Builder::serviceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceRoleArn").build()).build();

    private static final SdkField<String> TASK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskType").getter(getter(RegisterTaskWithMaintenanceWindowRequest::taskTypeAsString))
            .setter(setter(Builder::taskType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskType").build()).build();

    private static final SdkField<Map<String, MaintenanceWindowTaskParameterValueExpression>> TASK_PARAMETERS_FIELD = SdkField
            .<Map<String, MaintenanceWindowTaskParameterValueExpression>> builder(MarshallingType.MAP)
            .memberName("TaskParameters")
            .getter(getter(RegisterTaskWithMaintenanceWindowRequest::taskParameters))
            .setter(setter(Builder::taskParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<MaintenanceWindowTaskParameterValueExpression> builder(MarshallingType.SDK_POJO)
                                            .constructor(MaintenanceWindowTaskParameterValueExpression::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<MaintenanceWindowTaskInvocationParameters> TASK_INVOCATION_PARAMETERS_FIELD = SdkField
            .<MaintenanceWindowTaskInvocationParameters> builder(MarshallingType.SDK_POJO).memberName("TaskInvocationParameters")
            .getter(getter(RegisterTaskWithMaintenanceWindowRequest::taskInvocationParameters))
            .setter(setter(Builder::taskInvocationParameters)).constructor(MaintenanceWindowTaskInvocationParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskInvocationParameters").build())
            .build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority").getter(getter(RegisterTaskWithMaintenanceWindowRequest::priority))
            .setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<String> MAX_CONCURRENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaxConcurrency").getter(getter(RegisterTaskWithMaintenanceWindowRequest::maxConcurrency))
            .setter(setter(Builder::maxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrency").build()).build();

    private static final SdkField<String> MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaxErrors").getter(getter(RegisterTaskWithMaintenanceWindowRequest::maxErrors))
            .setter(setter(Builder::maxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxErrors").build()).build();

    private static final SdkField<LoggingInfo> LOGGING_INFO_FIELD = SdkField.<LoggingInfo> builder(MarshallingType.SDK_POJO)
            .memberName("LoggingInfo").getter(getter(RegisterTaskWithMaintenanceWindowRequest::loggingInfo))
            .setter(setter(Builder::loggingInfo)).constructor(LoggingInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoggingInfo").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(RegisterTaskWithMaintenanceWindowRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(RegisterTaskWithMaintenanceWindowRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(RegisterTaskWithMaintenanceWindowRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> CUTOFF_BEHAVIOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CutoffBehavior").getter(getter(RegisterTaskWithMaintenanceWindowRequest::cutoffBehaviorAsString))
            .setter(setter(Builder::cutoffBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CutoffBehavior").build()).build();

    private static final SdkField<AlarmConfiguration> ALARM_CONFIGURATION_FIELD = SdkField
            .<AlarmConfiguration> builder(MarshallingType.SDK_POJO).memberName("AlarmConfiguration")
            .getter(getter(RegisterTaskWithMaintenanceWindowRequest::alarmConfiguration))
            .setter(setter(Builder::alarmConfiguration)).constructor(AlarmConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WINDOW_ID_FIELD,
            TARGETS_FIELD, TASK_ARN_FIELD, SERVICE_ROLE_ARN_FIELD, TASK_TYPE_FIELD, TASK_PARAMETERS_FIELD,
            TASK_INVOCATION_PARAMETERS_FIELD, PRIORITY_FIELD, MAX_CONCURRENCY_FIELD, MAX_ERRORS_FIELD, LOGGING_INFO_FIELD,
            NAME_FIELD, DESCRIPTION_FIELD, CLIENT_TOKEN_FIELD, CUTOFF_BEHAVIOR_FIELD, ALARM_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String windowId;

    private final List<Target> targets;

    private final String taskArn;

    private final String serviceRoleArn;

    private final String taskType;

    private final Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters;

    private final MaintenanceWindowTaskInvocationParameters taskInvocationParameters;

    private final Integer priority;

    private final String maxConcurrency;

    private final String maxErrors;

    private final LoggingInfo loggingInfo;

    private final String name;

    private final String description;

    private final String clientToken;

    private final String cutoffBehavior;

    private final AlarmConfiguration alarmConfiguration;

    private RegisterTaskWithMaintenanceWindowRequest(BuilderImpl builder) {
        super(builder);
        this.windowId = builder.windowId;
        this.targets = builder.targets;
        this.taskArn = builder.taskArn;
        this.serviceRoleArn = builder.serviceRoleArn;
        this.taskType = builder.taskType;
        this.taskParameters = builder.taskParameters;
        this.taskInvocationParameters = builder.taskInvocationParameters;
        this.priority = builder.priority;
        this.maxConcurrency = builder.maxConcurrency;
        this.maxErrors = builder.maxErrors;
        this.loggingInfo = builder.loggingInfo;
        this.name = builder.name;
        this.description = builder.description;
        this.clientToken = builder.clientToken;
        this.cutoffBehavior = builder.cutoffBehavior;
        this.alarmConfiguration = builder.alarmConfiguration;
    }

    /**
     * <p>
     * The ID of the maintenance window the task should be added to.
     * </p>
     * 
     * @return The ID of the maintenance window the task should be added to.
     */
    public final String windowId() {
        return windowId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Targets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The targets (either managed nodes or maintenance window targets).
     * </p>
     * <note>
     * <p>
     * One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the task,
     * targets are optional for other maintenance window task types (Automation, Lambda, and Step Functions). For more
     * information about running tasks that don't specify targets, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
     * >Registering maintenance window tasks without targets</a> in the <i>Amazon Web Services Systems Manager User
     * Guide</i>.
     * </p>
     * </note>
     * <p>
     * Specify managed nodes using the following format:
     * </p>
     * <p>
     * <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;</code>
     * </p>
     * <p>
     * Specify maintenance window targets using the following format:
     * </p>
     * <p>
     * <code>Key=WindowTargetIds,Values=&lt;window-target-id-1&gt;,&lt;window-target-id-2&gt;</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargets} method.
     * </p>
     * 
     * @return The targets (either managed nodes or maintenance window targets).</p> <note>
     *         <p>
     *         One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the
     *         task, targets are optional for other maintenance window task types (Automation, Lambda, and Step
     *         Functions). For more information about running tasks that don't specify targets, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
     *         >Registering maintenance window tasks without targets</a> in the <i>Amazon Web Services Systems Manager
     *         User Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         Specify managed nodes using the following format:
     *         </p>
     *         <p>
     *         <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;</code>
     *         </p>
     *         <p>
     *         Specify maintenance window targets using the following format:
     *         </p>
     *         <p>
     *         <code>Key=WindowTargetIds,Values=&lt;window-target-id-1&gt;,&lt;window-target-id-2&gt;</code>
     */
    public final List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The ARN of the task to run.
     * </p>
     * 
     * @return The ARN of the task to run.
     */
    public final String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM service role for Amazon Web Services Systems Manager to assume when
     * running a maintenance window task. If you do not specify a service role ARN, Systems Manager uses a
     * service-linked role in your account. If no appropriate service-linked role for Systems Manager exists in your
     * account, it is created when you run <code>RegisterTaskWithMaintenanceWindow</code>.
     * </p>
     * <p>
     * However, for an improved security posture, we strongly recommend creating a custom policy and custom service role
     * for running your maintenance window tasks. The policy can be crafted to provide only the permissions needed for
     * your particular maintenance window tasks. For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-permissions.html">Setting
     * up Maintenance Windows</a> in the in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM service role for Amazon Web Services Systems Manager to assume
     *         when running a maintenance window task. If you do not specify a service role ARN, Systems Manager uses a
     *         service-linked role in your account. If no appropriate service-linked role for Systems Manager exists in
     *         your account, it is created when you run <code>RegisterTaskWithMaintenanceWindow</code>.</p>
     *         <p>
     *         However, for an improved security posture, we strongly recommend creating a custom policy and custom
     *         service role for running your maintenance window tasks. The policy can be crafted to provide only the
     *         permissions needed for your particular maintenance window tasks. For more information, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-permissions.html"
     *         >Setting up Maintenance Windows</a> in the in the <i>Amazon Web Services Systems Manager User Guide</i>.
     */
    public final String serviceRoleArn() {
        return serviceRoleArn;
    }

    /**
     * <p>
     * The type of task being registered.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskTypeAsString}.
     * </p>
     * 
     * @return The type of task being registered.
     * @see MaintenanceWindowTaskType
     */
    public final MaintenanceWindowTaskType taskType() {
        return MaintenanceWindowTaskType.fromValue(taskType);
    }

    /**
     * <p>
     * The type of task being registered.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskTypeAsString}.
     * </p>
     * 
     * @return The type of task being registered.
     * @see MaintenanceWindowTaskType
     */
    public final String taskTypeAsString() {
        return taskType;
    }

    /**
     * For responses, this returns true if the service returned a value for the TaskParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTaskParameters() {
        return taskParameters != null && !(taskParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The parameters that should be passed to the task when it is run.
     * </p>
     * <note>
     * <p>
     * <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs, instead
     * use the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure. For information
     * about how Systems Manager handles these options for the supported maintenance window task types, see
     * <a>MaintenanceWindowTaskInvocationParameters</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTaskParameters} method.
     * </p>
     * 
     * @return The parameters that should be passed to the task when it is run.</p> <note>
     *         <p>
     *         <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs,
     *         instead use the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure.
     *         For information about how Systems Manager handles these options for the supported maintenance window task
     *         types, see <a>MaintenanceWindowTaskInvocationParameters</a>.
     *         </p>
     */
    public final Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters() {
        return taskParameters;
    }

    /**
     * <p>
     * The parameters that the task should use during execution. Populate only the fields that match the task type. All
     * other fields should be empty.
     * </p>
     * 
     * @return The parameters that the task should use during execution. Populate only the fields that match the task
     *         type. All other fields should be empty.
     */
    public final MaintenanceWindowTaskInvocationParameters taskInvocationParameters() {
        return taskInvocationParameters;
    }

    /**
     * <p>
     * The priority of the task in the maintenance window, the lower the number the higher the priority. Tasks in a
     * maintenance window are scheduled in priority order with tasks that have the same priority scheduled in parallel.
     * </p>
     * 
     * @return The priority of the task in the maintenance window, the lower the number the higher the priority. Tasks
     *         in a maintenance window are scheduled in priority order with tasks that have the same priority scheduled
     *         in parallel.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * <p>
     * The maximum number of targets this task can be run for, in parallel.
     * </p>
     * <note>
     * <p>
     * Although this element is listed as "Required: No", a value can be omitted only when you are registering or
     * updating a <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
     * >targetless task</a> You must provide a value in all other cases.
     * </p>
     * <p>
     * For maintenance window tasks without a target specified, you can't supply a value for this option. Instead, the
     * system inserts a placeholder value of <code>1</code>. This value doesn't affect the running of your task.
     * </p>
     * </note>
     * 
     * @return The maximum number of targets this task can be run for, in parallel.</p> <note>
     *         <p>
     *         Although this element is listed as "Required: No", a value can be omitted only when you are registering
     *         or updating a <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
     *         >targetless task</a> You must provide a value in all other cases.
     *         </p>
     *         <p>
     *         For maintenance window tasks without a target specified, you can't supply a value for this option.
     *         Instead, the system inserts a placeholder value of <code>1</code>. This value doesn't affect the running
     *         of your task.
     *         </p>
     */
    public final String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The maximum number of errors allowed before this task stops being scheduled.
     * </p>
     * <note>
     * <p>
     * Although this element is listed as "Required: No", a value can be omitted only when you are registering or
     * updating a <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
     * >targetless task</a> You must provide a value in all other cases.
     * </p>
     * <p>
     * For maintenance window tasks without a target specified, you can't supply a value for this option. Instead, the
     * system inserts a placeholder value of <code>1</code>. This value doesn't affect the running of your task.
     * </p>
     * </note>
     * 
     * @return The maximum number of errors allowed before this task stops being scheduled.</p> <note>
     *         <p>
     *         Although this element is listed as "Required: No", a value can be omitted only when you are registering
     *         or updating a <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
     *         >targetless task</a> You must provide a value in all other cases.
     *         </p>
     *         <p>
     *         For maintenance window tasks without a target specified, you can't supply a value for this option.
     *         Instead, the system inserts a placeholder value of <code>1</code>. This value doesn't affect the running
     *         of your task.
     *         </p>
     */
    public final String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * A structure containing information about an Amazon Simple Storage Service (Amazon S3) bucket to write managed
     * node-level logs to.
     * </p>
     * <note>
     * <p>
     * <code>LoggingInfo</code> has been deprecated. To specify an Amazon Simple Storage Service (Amazon S3) bucket to
     * contain logs, instead use the <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options in the
     * <code>TaskInvocationParameters</code> structure. For information about how Amazon Web Services Systems Manager
     * handles these options for the supported maintenance window task types, see
     * <a>MaintenanceWindowTaskInvocationParameters</a>.
     * </p>
     * </note>
     * 
     * @return A structure containing information about an Amazon Simple Storage Service (Amazon S3) bucket to write
     *         managed node-level logs to. </p> <note>
     *         <p>
     *         <code>LoggingInfo</code> has been deprecated. To specify an Amazon Simple Storage Service (Amazon S3)
     *         bucket to contain logs, instead use the <code>OutputS3BucketName</code> and
     *         <code>OutputS3KeyPrefix</code> options in the <code>TaskInvocationParameters</code> structure. For
     *         information about how Amazon Web Services Systems Manager handles these options for the supported
     *         maintenance window task types, see <a>MaintenanceWindowTaskInvocationParameters</a>.
     *         </p>
     */
    public final LoggingInfo loggingInfo() {
        return loggingInfo;
    }

    /**
     * <p>
     * An optional name for the task.
     * </p>
     * 
     * @return An optional name for the task.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * An optional description for the task.
     * </p>
     * 
     * @return An optional description for the task.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * User-provided idempotency token.
     * </p>
     * 
     * @return User-provided idempotency token.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * Indicates whether tasks should continue to run after the cutoff time specified in the maintenance windows is
     * reached.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CONTINUE_TASK</code>: When the cutoff time is reached, any tasks that are running continue. The default
     * value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCEL_TASK</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Automation, Lambda, Step Functions tasks: When the cutoff time is reached, any task invocations that are
     * already running continue, but no new task invocations are started.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Run Command tasks: When the cutoff time is reached, the system sends a <a>CancelCommand</a> operation that
     * attempts to cancel the command associated with the task. However, there is no guarantee that the command will be
     * terminated and the underlying process stopped.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The status for tasks that are not completed is <code>TIMED_OUT</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #cutoffBehavior}
     * will return {@link MaintenanceWindowTaskCutoffBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #cutoffBehaviorAsString}.
     * </p>
     * 
     * @return Indicates whether tasks should continue to run after the cutoff time specified in the maintenance windows
     *         is reached. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONTINUE_TASK</code>: When the cutoff time is reached, any tasks that are running continue. The
     *         default value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCEL_TASK</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For Automation, Lambda, Step Functions tasks: When the cutoff time is reached, any task invocations that
     *         are already running continue, but no new task invocations are started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Run Command tasks: When the cutoff time is reached, the system sends a <a>CancelCommand</a> operation
     *         that attempts to cancel the command associated with the task. However, there is no guarantee that the
     *         command will be terminated and the underlying process stopped.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The status for tasks that are not completed is <code>TIMED_OUT</code>.
     *         </p>
     *         </li>
     * @see MaintenanceWindowTaskCutoffBehavior
     */
    public final MaintenanceWindowTaskCutoffBehavior cutoffBehavior() {
        return MaintenanceWindowTaskCutoffBehavior.fromValue(cutoffBehavior);
    }

    /**
     * <p>
     * Indicates whether tasks should continue to run after the cutoff time specified in the maintenance windows is
     * reached.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CONTINUE_TASK</code>: When the cutoff time is reached, any tasks that are running continue. The default
     * value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCEL_TASK</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Automation, Lambda, Step Functions tasks: When the cutoff time is reached, any task invocations that are
     * already running continue, but no new task invocations are started.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Run Command tasks: When the cutoff time is reached, the system sends a <a>CancelCommand</a> operation that
     * attempts to cancel the command associated with the task. However, there is no guarantee that the command will be
     * terminated and the underlying process stopped.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The status for tasks that are not completed is <code>TIMED_OUT</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #cutoffBehavior}
     * will return {@link MaintenanceWindowTaskCutoffBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #cutoffBehaviorAsString}.
     * </p>
     * 
     * @return Indicates whether tasks should continue to run after the cutoff time specified in the maintenance windows
     *         is reached. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONTINUE_TASK</code>: When the cutoff time is reached, any tasks that are running continue. The
     *         default value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCEL_TASK</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For Automation, Lambda, Step Functions tasks: When the cutoff time is reached, any task invocations that
     *         are already running continue, but no new task invocations are started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Run Command tasks: When the cutoff time is reached, the system sends a <a>CancelCommand</a> operation
     *         that attempts to cancel the command associated with the task. However, there is no guarantee that the
     *         command will be terminated and the underlying process stopped.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The status for tasks that are not completed is <code>TIMED_OUT</code>.
     *         </p>
     *         </li>
     * @see MaintenanceWindowTaskCutoffBehavior
     */
    public final String cutoffBehaviorAsString() {
        return cutoffBehavior;
    }

    /**
     * <p>
     * The CloudWatch alarm you want to apply to your maintenance window task.
     * </p>
     * 
     * @return The CloudWatch alarm you want to apply to your maintenance window task.
     */
    public final AlarmConfiguration alarmConfiguration() {
        return alarmConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(windowId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTaskParameters() ? taskParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(taskInvocationParameters());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(loggingInfo());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(cutoffBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(alarmConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RegisterTaskWithMaintenanceWindowRequest)) {
            return false;
        }
        RegisterTaskWithMaintenanceWindowRequest other = (RegisterTaskWithMaintenanceWindowRequest) obj;
        return Objects.equals(windowId(), other.windowId()) && hasTargets() == other.hasTargets()
                && Objects.equals(targets(), other.targets()) && Objects.equals(taskArn(), other.taskArn())
                && Objects.equals(serviceRoleArn(), other.serviceRoleArn())
                && Objects.equals(taskTypeAsString(), other.taskTypeAsString())
                && hasTaskParameters() == other.hasTaskParameters() && Objects.equals(taskParameters(), other.taskParameters())
                && Objects.equals(taskInvocationParameters(), other.taskInvocationParameters())
                && Objects.equals(priority(), other.priority()) && Objects.equals(maxConcurrency(), other.maxConcurrency())
                && Objects.equals(maxErrors(), other.maxErrors()) && Objects.equals(loggingInfo(), other.loggingInfo())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(clientToken(), other.clientToken())
                && Objects.equals(cutoffBehaviorAsString(), other.cutoffBehaviorAsString())
                && Objects.equals(alarmConfiguration(), other.alarmConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RegisterTaskWithMaintenanceWindowRequest").add("WindowId", windowId())
                .add("Targets", hasTargets() ? targets() : null).add("TaskArn", taskArn())
                .add("ServiceRoleArn", serviceRoleArn()).add("TaskType", taskTypeAsString())
                .add("TaskParameters", taskParameters() == null ? null : "*** Sensitive Data Redacted ***")
                .add("TaskInvocationParameters", taskInvocationParameters()).add("Priority", priority())
                .add("MaxConcurrency", maxConcurrency()).add("MaxErrors", maxErrors()).add("LoggingInfo", loggingInfo())
                .add("Name", name()).add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ClientToken", clientToken()).add("CutoffBehavior", cutoffBehaviorAsString())
                .add("AlarmConfiguration", alarmConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WindowId":
            return Optional.ofNullable(clazz.cast(windowId()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "TaskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "ServiceRoleArn":
            return Optional.ofNullable(clazz.cast(serviceRoleArn()));
        case "TaskType":
            return Optional.ofNullable(clazz.cast(taskTypeAsString()));
        case "TaskParameters":
            return Optional.ofNullable(clazz.cast(taskParameters()));
        case "TaskInvocationParameters":
            return Optional.ofNullable(clazz.cast(taskInvocationParameters()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "MaxConcurrency":
            return Optional.ofNullable(clazz.cast(maxConcurrency()));
        case "MaxErrors":
            return Optional.ofNullable(clazz.cast(maxErrors()));
        case "LoggingInfo":
            return Optional.ofNullable(clazz.cast(loggingInfo()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "CutoffBehavior":
            return Optional.ofNullable(clazz.cast(cutoffBehaviorAsString()));
        case "AlarmConfiguration":
            return Optional.ofNullable(clazz.cast(alarmConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("WindowId", WINDOW_ID_FIELD);
        map.put("Targets", TARGETS_FIELD);
        map.put("TaskArn", TASK_ARN_FIELD);
        map.put("ServiceRoleArn", SERVICE_ROLE_ARN_FIELD);
        map.put("TaskType", TASK_TYPE_FIELD);
        map.put("TaskParameters", TASK_PARAMETERS_FIELD);
        map.put("TaskInvocationParameters", TASK_INVOCATION_PARAMETERS_FIELD);
        map.put("Priority", PRIORITY_FIELD);
        map.put("MaxConcurrency", MAX_CONCURRENCY_FIELD);
        map.put("MaxErrors", MAX_ERRORS_FIELD);
        map.put("LoggingInfo", LOGGING_INFO_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("ClientToken", CLIENT_TOKEN_FIELD);
        map.put("CutoffBehavior", CUTOFF_BEHAVIOR_FIELD);
        map.put("AlarmConfiguration", ALARM_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RegisterTaskWithMaintenanceWindowRequest, T> g) {
        return obj -> g.apply((RegisterTaskWithMaintenanceWindowRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, RegisterTaskWithMaintenanceWindowRequest> {
        /**
         * <p>
         * The ID of the maintenance window the task should be added to.
         * </p>
         * 
         * @param windowId
         *        The ID of the maintenance window the task should be added to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowId(String windowId);

        /**
         * <p>
         * The targets (either managed nodes or maintenance window targets).
         * </p>
         * <note>
         * <p>
         * One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the task,
         * targets are optional for other maintenance window task types (Automation, Lambda, and Step Functions). For
         * more information about running tasks that don't specify targets, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         * >Registering maintenance window tasks without targets</a> in the <i>Amazon Web Services Systems Manager User
         * Guide</i>.
         * </p>
         * </note>
         * <p>
         * Specify managed nodes using the following format:
         * </p>
         * <p>
         * <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;</code>
         * </p>
         * <p>
         * Specify maintenance window targets using the following format:
         * </p>
         * <p>
         * <code>Key=WindowTargetIds,Values=&lt;window-target-id-1&gt;,&lt;window-target-id-2&gt;</code>
         * </p>
         * 
         * @param targets
         *        The targets (either managed nodes or maintenance window targets).</p> <note>
         *        <p>
         *        One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the
         *        task, targets are optional for other maintenance window task types (Automation, Lambda, and Step
         *        Functions). For more information about running tasks that don't specify targets, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         *        >Registering maintenance window tasks without targets</a> in the <i>Amazon Web Services Systems
         *        Manager User Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        Specify managed nodes using the following format:
         *        </p>
         *        <p>
         *        <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;</code>
         *        </p>
         *        <p>
         *        Specify maintenance window targets using the following format:
         *        </p>
         *        <p>
         *        <code>Key=WindowTargetIds,Values=&lt;window-target-id-1&gt;,&lt;window-target-id-2&gt;</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets (either managed nodes or maintenance window targets).
         * </p>
         * <note>
         * <p>
         * One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the task,
         * targets are optional for other maintenance window task types (Automation, Lambda, and Step Functions). For
         * more information about running tasks that don't specify targets, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         * >Registering maintenance window tasks without targets</a> in the <i>Amazon Web Services Systems Manager User
         * Guide</i>.
         * </p>
         * </note>
         * <p>
         * Specify managed nodes using the following format:
         * </p>
         * <p>
         * <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;</code>
         * </p>
         * <p>
         * Specify maintenance window targets using the following format:
         * </p>
         * <p>
         * <code>Key=WindowTargetIds,Values=&lt;window-target-id-1&gt;,&lt;window-target-id-2&gt;</code>
         * </p>
         * 
         * @param targets
         *        The targets (either managed nodes or maintenance window targets).</p> <note>
         *        <p>
         *        One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the
         *        task, targets are optional for other maintenance window task types (Automation, Lambda, and Step
         *        Functions). For more information about running tasks that don't specify targets, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         *        >Registering maintenance window tasks without targets</a> in the <i>Amazon Web Services Systems
         *        Manager User Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        Specify managed nodes using the following format:
         *        </p>
         *        <p>
         *        <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;</code>
         *        </p>
         *        <p>
         *        Specify maintenance window targets using the following format:
         *        </p>
         *        <p>
         *        <code>Key=WindowTargetIds,Values=&lt;window-target-id-1&gt;,&lt;window-target-id-2&gt;</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets (either managed nodes or maintenance window targets).
         * </p>
         * <note>
         * <p>
         * One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the task,
         * targets are optional for other maintenance window task types (Automation, Lambda, and Step Functions). For
         * more information about running tasks that don't specify targets, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         * >Registering maintenance window tasks without targets</a> in the <i>Amazon Web Services Systems Manager User
         * Guide</i>.
         * </p>
         * </note>
         * <p>
         * Specify managed nodes using the following format:
         * </p>
         * <p>
         * <code>Key=InstanceIds,Values=&lt;instance-id-1&gt;,&lt;instance-id-2&gt;</code>
         * </p>
         * <p>
         * Specify maintenance window targets using the following format:
         * </p>
         * <p>
         * <code>Key=WindowTargetIds,Values=&lt;window-target-id-1&gt;,&lt;window-target-id-2&gt;</code>
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.Target.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ssm.model.Target#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ssm.model.Target.Builder#build()}
         * is called immediately and its result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ssm.model.Target.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(java.util.Collection<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The ARN of the task to run.
         * </p>
         * 
         * @param taskArn
         *        The ARN of the task to run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM service role for Amazon Web Services Systems Manager to assume when
         * running a maintenance window task. If you do not specify a service role ARN, Systems Manager uses a
         * service-linked role in your account. If no appropriate service-linked role for Systems Manager exists in your
         * account, it is created when you run <code>RegisterTaskWithMaintenanceWindow</code>.
         * </p>
         * <p>
         * However, for an improved security posture, we strongly recommend creating a custom policy and custom service
         * role for running your maintenance window tasks. The policy can be crafted to provide only the permissions
         * needed for your particular maintenance window tasks. For more information, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-permissions.html"
         * >Setting up Maintenance Windows</a> in the in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param serviceRoleArn
         *        The Amazon Resource Name (ARN) of the IAM service role for Amazon Web Services Systems Manager to
         *        assume when running a maintenance window task. If you do not specify a service role ARN, Systems
         *        Manager uses a service-linked role in your account. If no appropriate service-linked role for Systems
         *        Manager exists in your account, it is created when you run
         *        <code>RegisterTaskWithMaintenanceWindow</code>.</p>
         *        <p>
         *        However, for an improved security posture, we strongly recommend creating a custom policy and custom
         *        service role for running your maintenance window tasks. The policy can be crafted to provide only the
         *        permissions needed for your particular maintenance window tasks. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-permissions.html"
         *        >Setting up Maintenance Windows</a> in the in the <i>Amazon Web Services Systems Manager User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRoleArn(String serviceRoleArn);

        /**
         * <p>
         * The type of task being registered.
         * </p>
         * 
         * @param taskType
         *        The type of task being registered.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder taskType(String taskType);

        /**
         * <p>
         * The type of task being registered.
         * </p>
         * 
         * @param taskType
         *        The type of task being registered.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder taskType(MaintenanceWindowTaskType taskType);

        /**
         * <p>
         * The parameters that should be passed to the task when it is run.
         * </p>
         * <note>
         * <p>
         * <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs,
         * instead use the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure. For
         * information about how Systems Manager handles these options for the supported maintenance window task types,
         * see <a>MaintenanceWindowTaskInvocationParameters</a>.
         * </p>
         * </note>
         * 
         * @param taskParameters
         *        The parameters that should be passed to the task when it is run.</p> <note>
         *        <p>
         *        <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs,
         *        instead use the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure.
         *        For information about how Systems Manager handles these options for the supported maintenance window
         *        task types, see <a>MaintenanceWindowTaskInvocationParameters</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters);

        /**
         * <p>
         * The parameters that the task should use during execution. Populate only the fields that match the task type.
         * All other fields should be empty.
         * </p>
         * 
         * @param taskInvocationParameters
         *        The parameters that the task should use during execution. Populate only the fields that match the task
         *        type. All other fields should be empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskInvocationParameters(MaintenanceWindowTaskInvocationParameters taskInvocationParameters);

        /**
         * <p>
         * The parameters that the task should use during execution. Populate only the fields that match the task type.
         * All other fields should be empty.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link MaintenanceWindowTaskInvocationParameters.Builder} avoiding the need to create one manually via
         * {@link MaintenanceWindowTaskInvocationParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MaintenanceWindowTaskInvocationParameters.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #taskInvocationParameters(MaintenanceWindowTaskInvocationParameters)}.
         * 
         * @param taskInvocationParameters
         *        a consumer that will call methods on {@link MaintenanceWindowTaskInvocationParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #taskInvocationParameters(MaintenanceWindowTaskInvocationParameters)
         */
        default Builder taskInvocationParameters(
                Consumer<MaintenanceWindowTaskInvocationParameters.Builder> taskInvocationParameters) {
            return taskInvocationParameters(MaintenanceWindowTaskInvocationParameters.builder()
                    .applyMutation(taskInvocationParameters).build());
        }

        /**
         * <p>
         * The priority of the task in the maintenance window, the lower the number the higher the priority. Tasks in a
         * maintenance window are scheduled in priority order with tasks that have the same priority scheduled in
         * parallel.
         * </p>
         * 
         * @param priority
         *        The priority of the task in the maintenance window, the lower the number the higher the priority.
         *        Tasks in a maintenance window are scheduled in priority order with tasks that have the same priority
         *        scheduled in parallel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The maximum number of targets this task can be run for, in parallel.
         * </p>
         * <note>
         * <p>
         * Although this element is listed as "Required: No", a value can be omitted only when you are registering or
         * updating a <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         * >targetless task</a> You must provide a value in all other cases.
         * </p>
         * <p>
         * For maintenance window tasks without a target specified, you can't supply a value for this option. Instead,
         * the system inserts a placeholder value of <code>1</code>. This value doesn't affect the running of your task.
         * </p>
         * </note>
         * 
         * @param maxConcurrency
         *        The maximum number of targets this task can be run for, in parallel.</p> <note>
         *        <p>
         *        Although this element is listed as "Required: No", a value can be omitted only when you are
         *        registering or updating a <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         *        >targetless task</a> You must provide a value in all other cases.
         *        </p>
         *        <p>
         *        For maintenance window tasks without a target specified, you can't supply a value for this option.
         *        Instead, the system inserts a placeholder value of <code>1</code>. This value doesn't affect the
         *        running of your task.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The maximum number of errors allowed before this task stops being scheduled.
         * </p>
         * <note>
         * <p>
         * Although this element is listed as "Required: No", a value can be omitted only when you are registering or
         * updating a <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         * >targetless task</a> You must provide a value in all other cases.
         * </p>
         * <p>
         * For maintenance window tasks without a target specified, you can't supply a value for this option. Instead,
         * the system inserts a placeholder value of <code>1</code>. This value doesn't affect the running of your task.
         * </p>
         * </note>
         * 
         * @param maxErrors
         *        The maximum number of errors allowed before this task stops being scheduled.</p> <note>
         *        <p>
         *        Although this element is listed as "Required: No", a value can be omitted only when you are
         *        registering or updating a <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
         *        >targetless task</a> You must provide a value in all other cases.
         *        </p>
         *        <p>
         *        For maintenance window tasks without a target specified, you can't supply a value for this option.
         *        Instead, the system inserts a placeholder value of <code>1</code>. This value doesn't affect the
         *        running of your task.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * A structure containing information about an Amazon Simple Storage Service (Amazon S3) bucket to write managed
         * node-level logs to.
         * </p>
         * <note>
         * <p>
         * <code>LoggingInfo</code> has been deprecated. To specify an Amazon Simple Storage Service (Amazon S3) bucket
         * to contain logs, instead use the <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options
         * in the <code>TaskInvocationParameters</code> structure. For information about how Amazon Web Services Systems
         * Manager handles these options for the supported maintenance window task types, see
         * <a>MaintenanceWindowTaskInvocationParameters</a>.
         * </p>
         * </note>
         * 
         * @param loggingInfo
         *        A structure containing information about an Amazon Simple Storage Service (Amazon S3) bucket to write
         *        managed node-level logs to. </p> <note>
         *        <p>
         *        <code>LoggingInfo</code> has been deprecated. To specify an Amazon Simple Storage Service (Amazon S3)
         *        bucket to contain logs, instead use the <code>OutputS3BucketName</code> and
         *        <code>OutputS3KeyPrefix</code> options in the <code>TaskInvocationParameters</code> structure. For
         *        information about how Amazon Web Services Systems Manager handles these options for the supported
         *        maintenance window task types, see <a>MaintenanceWindowTaskInvocationParameters</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingInfo(LoggingInfo loggingInfo);

        /**
         * <p>
         * A structure containing information about an Amazon Simple Storage Service (Amazon S3) bucket to write managed
         * node-level logs to.
         * </p>
         * <note>
         * <p>
         * <code>LoggingInfo</code> has been deprecated. To specify an Amazon Simple Storage Service (Amazon S3) bucket
         * to contain logs, instead use the <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options
         * in the <code>TaskInvocationParameters</code> structure. For information about how Amazon Web Services Systems
         * Manager handles these options for the supported maintenance window task types, see
         * <a>MaintenanceWindowTaskInvocationParameters</a>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link LoggingInfo.Builder} avoiding the
         * need to create one manually via {@link LoggingInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LoggingInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #loggingInfo(LoggingInfo)}.
         * 
         * @param loggingInfo
         *        a consumer that will call methods on {@link LoggingInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loggingInfo(LoggingInfo)
         */
        default Builder loggingInfo(Consumer<LoggingInfo.Builder> loggingInfo) {
            return loggingInfo(LoggingInfo.builder().applyMutation(loggingInfo).build());
        }

        /**
         * <p>
         * An optional name for the task.
         * </p>
         * 
         * @param name
         *        An optional name for the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * An optional description for the task.
         * </p>
         * 
         * @param description
         *        An optional description for the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * User-provided idempotency token.
         * </p>
         * 
         * @param clientToken
         *        User-provided idempotency token.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * Indicates whether tasks should continue to run after the cutoff time specified in the maintenance windows is
         * reached.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CONTINUE_TASK</code>: When the cutoff time is reached, any tasks that are running continue. The default
         * value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCEL_TASK</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For Automation, Lambda, Step Functions tasks: When the cutoff time is reached, any task invocations that are
         * already running continue, but no new task invocations are started.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Run Command tasks: When the cutoff time is reached, the system sends a <a>CancelCommand</a> operation
         * that attempts to cancel the command associated with the task. However, there is no guarantee that the command
         * will be terminated and the underlying process stopped.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The status for tasks that are not completed is <code>TIMED_OUT</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cutoffBehavior
         *        Indicates whether tasks should continue to run after the cutoff time specified in the maintenance
         *        windows is reached. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONTINUE_TASK</code>: When the cutoff time is reached, any tasks that are running continue. The
         *        default value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCEL_TASK</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For Automation, Lambda, Step Functions tasks: When the cutoff time is reached, any task invocations
         *        that are already running continue, but no new task invocations are started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Run Command tasks: When the cutoff time is reached, the system sends a <a>CancelCommand</a>
         *        operation that attempts to cancel the command associated with the task. However, there is no guarantee
         *        that the command will be terminated and the underlying process stopped.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The status for tasks that are not completed is <code>TIMED_OUT</code>.
         *        </p>
         *        </li>
         * @see MaintenanceWindowTaskCutoffBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskCutoffBehavior
         */
        Builder cutoffBehavior(String cutoffBehavior);

        /**
         * <p>
         * Indicates whether tasks should continue to run after the cutoff time specified in the maintenance windows is
         * reached.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CONTINUE_TASK</code>: When the cutoff time is reached, any tasks that are running continue. The default
         * value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCEL_TASK</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For Automation, Lambda, Step Functions tasks: When the cutoff time is reached, any task invocations that are
         * already running continue, but no new task invocations are started.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Run Command tasks: When the cutoff time is reached, the system sends a <a>CancelCommand</a> operation
         * that attempts to cancel the command associated with the task. However, there is no guarantee that the command
         * will be terminated and the underlying process stopped.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The status for tasks that are not completed is <code>TIMED_OUT</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cutoffBehavior
         *        Indicates whether tasks should continue to run after the cutoff time specified in the maintenance
         *        windows is reached. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONTINUE_TASK</code>: When the cutoff time is reached, any tasks that are running continue. The
         *        default value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCEL_TASK</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For Automation, Lambda, Step Functions tasks: When the cutoff time is reached, any task invocations
         *        that are already running continue, but no new task invocations are started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Run Command tasks: When the cutoff time is reached, the system sends a <a>CancelCommand</a>
         *        operation that attempts to cancel the command associated with the task. However, there is no guarantee
         *        that the command will be terminated and the underlying process stopped.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The status for tasks that are not completed is <code>TIMED_OUT</code>.
         *        </p>
         *        </li>
         * @see MaintenanceWindowTaskCutoffBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskCutoffBehavior
         */
        Builder cutoffBehavior(MaintenanceWindowTaskCutoffBehavior cutoffBehavior);

        /**
         * <p>
         * The CloudWatch alarm you want to apply to your maintenance window task.
         * </p>
         * 
         * @param alarmConfiguration
         *        The CloudWatch alarm you want to apply to your maintenance window task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmConfiguration(AlarmConfiguration alarmConfiguration);

        /**
         * <p>
         * The CloudWatch alarm you want to apply to your maintenance window task.
         * </p>
         * This is a convenience method that creates an instance of the {@link AlarmConfiguration.Builder} avoiding the
         * need to create one manually via {@link AlarmConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AlarmConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #alarmConfiguration(AlarmConfiguration)}.
         * 
         * @param alarmConfiguration
         *        a consumer that will call methods on {@link AlarmConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alarmConfiguration(AlarmConfiguration)
         */
        default Builder alarmConfiguration(Consumer<AlarmConfiguration.Builder> alarmConfiguration) {
            return alarmConfiguration(AlarmConfiguration.builder().applyMutation(alarmConfiguration).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String windowId;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String taskArn;

        private String serviceRoleArn;

        private String taskType;

        private Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters = DefaultSdkAutoConstructMap
                .getInstance();

        private MaintenanceWindowTaskInvocationParameters taskInvocationParameters;

        private Integer priority;

        private String maxConcurrency;

        private String maxErrors;

        private LoggingInfo loggingInfo;

        private String name;

        private String description;

        private String clientToken;

        private String cutoffBehavior;

        private AlarmConfiguration alarmConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(RegisterTaskWithMaintenanceWindowRequest model) {
            super(model);
            windowId(model.windowId);
            targets(model.targets);
            taskArn(model.taskArn);
            serviceRoleArn(model.serviceRoleArn);
            taskType(model.taskType);
            taskParameters(model.taskParameters);
            taskInvocationParameters(model.taskInvocationParameters);
            priority(model.priority);
            maxConcurrency(model.maxConcurrency);
            maxErrors(model.maxErrors);
            loggingInfo(model.loggingInfo);
            name(model.name);
            description(model.description);
            clientToken(model.clientToken);
            cutoffBehavior(model.cutoffBehavior);
            alarmConfiguration(model.alarmConfiguration);
        }

        public final String getWindowId() {
            return windowId;
        }

        public final void setWindowId(String windowId) {
            this.windowId = windowId;
        }

        @Override
        public final Builder windowId(String windowId) {
            this.windowId = windowId;
            return this;
        }

        public final List<Target.Builder> getTargets() {
            List<Target.Builder> result = TargetsCopier.copyToBuilder(this.targets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final String getServiceRoleArn() {
            return serviceRoleArn;
        }

        public final void setServiceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
        }

        @Override
        public final Builder serviceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
            return this;
        }

        public final String getTaskType() {
            return taskType;
        }

        public final void setTaskType(String taskType) {
            this.taskType = taskType;
        }

        @Override
        public final Builder taskType(String taskType) {
            this.taskType = taskType;
            return this;
        }

        @Override
        public final Builder taskType(MaintenanceWindowTaskType taskType) {
            this.taskType(taskType == null ? null : taskType.toString());
            return this;
        }

        public final Map<String, MaintenanceWindowTaskParameterValueExpression.Builder> getTaskParameters() {
            Map<String, MaintenanceWindowTaskParameterValueExpression.Builder> result = MaintenanceWindowTaskParametersCopier
                    .copyToBuilder(this.taskParameters);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setTaskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression.BuilderImpl> taskParameters) {
            this.taskParameters = MaintenanceWindowTaskParametersCopier.copyFromBuilder(taskParameters);
        }

        @Override
        public final Builder taskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters) {
            this.taskParameters = MaintenanceWindowTaskParametersCopier.copy(taskParameters);
            return this;
        }

        public final MaintenanceWindowTaskInvocationParameters.Builder getTaskInvocationParameters() {
            return taskInvocationParameters != null ? taskInvocationParameters.toBuilder() : null;
        }

        public final void setTaskInvocationParameters(
                MaintenanceWindowTaskInvocationParameters.BuilderImpl taskInvocationParameters) {
            this.taskInvocationParameters = taskInvocationParameters != null ? taskInvocationParameters.build() : null;
        }

        @Override
        public final Builder taskInvocationParameters(MaintenanceWindowTaskInvocationParameters taskInvocationParameters) {
            this.taskInvocationParameters = taskInvocationParameters;
            return this;
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        @Override
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final LoggingInfo.Builder getLoggingInfo() {
            return loggingInfo != null ? loggingInfo.toBuilder() : null;
        }

        public final void setLoggingInfo(LoggingInfo.BuilderImpl loggingInfo) {
            this.loggingInfo = loggingInfo != null ? loggingInfo.build() : null;
        }

        @Override
        public final Builder loggingInfo(LoggingInfo loggingInfo) {
            this.loggingInfo = loggingInfo;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final String getCutoffBehavior() {
            return cutoffBehavior;
        }

        public final void setCutoffBehavior(String cutoffBehavior) {
            this.cutoffBehavior = cutoffBehavior;
        }

        @Override
        public final Builder cutoffBehavior(String cutoffBehavior) {
            this.cutoffBehavior = cutoffBehavior;
            return this;
        }

        @Override
        public final Builder cutoffBehavior(MaintenanceWindowTaskCutoffBehavior cutoffBehavior) {
            this.cutoffBehavior(cutoffBehavior == null ? null : cutoffBehavior.toString());
            return this;
        }

        public final AlarmConfiguration.Builder getAlarmConfiguration() {
            return alarmConfiguration != null ? alarmConfiguration.toBuilder() : null;
        }

        public final void setAlarmConfiguration(AlarmConfiguration.BuilderImpl alarmConfiguration) {
            this.alarmConfiguration = alarmConfiguration != null ? alarmConfiguration.build() : null;
        }

        @Override
        public final Builder alarmConfiguration(AlarmConfiguration alarmConfiguration) {
            this.alarmConfiguration = alarmConfiguration;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RegisterTaskWithMaintenanceWindowRequest build() {
            return new RegisterTaskWithMaintenanceWindowRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
