/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a reviewer's response to a document review request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentReviewerResponseSource implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentReviewerResponseSource.Builder, DocumentReviewerResponseSource> {
    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime").getter(getter(DocumentReviewerResponseSource::createTime))
            .setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime").build()).build();

    private static final SdkField<Instant> UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedTime").getter(getter(DocumentReviewerResponseSource::updatedTime))
            .setter(setter(Builder::updatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedTime").build()).build();

    private static final SdkField<String> REVIEW_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReviewStatus").getter(getter(DocumentReviewerResponseSource::reviewStatusAsString))
            .setter(setter(Builder::reviewStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReviewStatus").build()).build();

    private static final SdkField<List<DocumentReviewCommentSource>> COMMENT_FIELD = SdkField
            .<List<DocumentReviewCommentSource>> builder(MarshallingType.LIST)
            .memberName("Comment")
            .getter(getter(DocumentReviewerResponseSource::comment))
            .setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentReviewCommentSource> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentReviewCommentSource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REVIEWER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Reviewer").getter(getter(DocumentReviewerResponseSource::reviewer)).setter(setter(Builder::reviewer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Reviewer").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATE_TIME_FIELD,
            UPDATED_TIME_FIELD, REVIEW_STATUS_FIELD, COMMENT_FIELD, REVIEWER_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant createTime;

    private final Instant updatedTime;

    private final String reviewStatus;

    private final List<DocumentReviewCommentSource> comment;

    private final String reviewer;

    private DocumentReviewerResponseSource(BuilderImpl builder) {
        this.createTime = builder.createTime;
        this.updatedTime = builder.updatedTime;
        this.reviewStatus = builder.reviewStatus;
        this.comment = builder.comment;
        this.reviewer = builder.reviewer;
    }

    /**
     * <p>
     * The date and time that a reviewer entered a response to a document review request.
     * </p>
     * 
     * @return The date and time that a reviewer entered a response to a document review request.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The date and time that a reviewer last updated a response to a document review request.
     * </p>
     * 
     * @return The date and time that a reviewer last updated a response to a document review request.
     */
    public final Instant updatedTime() {
        return updatedTime;
    }

    /**
     * <p>
     * The current review status of a new custom SSM document created by a member of your organization, or of the latest
     * version of an existing SSM document.
     * </p>
     * <p>
     * Only one version of a document can be in the APPROVED state at a time. When a new version is approved, the status
     * of the previous version changes to REJECTED.
     * </p>
     * <p>
     * Only one version of a document can be in review, or PENDING, at a time.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reviewStatus} will
     * return {@link ReviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #reviewStatusAsString}.
     * </p>
     * 
     * @return The current review status of a new custom SSM document created by a member of your organization, or of
     *         the latest version of an existing SSM document.</p>
     *         <p>
     *         Only one version of a document can be in the APPROVED state at a time. When a new version is approved,
     *         the status of the previous version changes to REJECTED.
     *         </p>
     *         <p>
     *         Only one version of a document can be in review, or PENDING, at a time.
     * @see ReviewStatus
     */
    public final ReviewStatus reviewStatus() {
        return ReviewStatus.fromValue(reviewStatus);
    }

    /**
     * <p>
     * The current review status of a new custom SSM document created by a member of your organization, or of the latest
     * version of an existing SSM document.
     * </p>
     * <p>
     * Only one version of a document can be in the APPROVED state at a time. When a new version is approved, the status
     * of the previous version changes to REJECTED.
     * </p>
     * <p>
     * Only one version of a document can be in review, or PENDING, at a time.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reviewStatus} will
     * return {@link ReviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #reviewStatusAsString}.
     * </p>
     * 
     * @return The current review status of a new custom SSM document created by a member of your organization, or of
     *         the latest version of an existing SSM document.</p>
     *         <p>
     *         Only one version of a document can be in the APPROVED state at a time. When a new version is approved,
     *         the status of the previous version changes to REJECTED.
     *         </p>
     *         <p>
     *         Only one version of a document can be in review, or PENDING, at a time.
     * @see ReviewStatus
     */
    public final String reviewStatusAsString() {
        return reviewStatus;
    }

    /**
     * Returns true if the Comment property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasComment() {
        return comment != null && !(comment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The comment entered by a reviewer as part of their document review response.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasComment()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The comment entered by a reviewer as part of their document review response.
     */
    public final List<DocumentReviewCommentSource> comment() {
        return comment;
    }

    /**
     * <p>
     * The user in your organization assigned to review a document request.
     * </p>
     * 
     * @return The user in your organization assigned to review a document request.
     */
    public final String reviewer() {
        return reviewer;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(updatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(reviewStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasComment() ? comment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(reviewer());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentReviewerResponseSource)) {
            return false;
        }
        DocumentReviewerResponseSource other = (DocumentReviewerResponseSource) obj;
        return Objects.equals(createTime(), other.createTime()) && Objects.equals(updatedTime(), other.updatedTime())
                && Objects.equals(reviewStatusAsString(), other.reviewStatusAsString()) && hasComment() == other.hasComment()
                && Objects.equals(comment(), other.comment()) && Objects.equals(reviewer(), other.reviewer());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentReviewerResponseSource").add("CreateTime", createTime())
                .add("UpdatedTime", updatedTime()).add("ReviewStatus", reviewStatusAsString())
                .add("Comment", hasComment() ? comment() : null).add("Reviewer", reviewer()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "UpdatedTime":
            return Optional.ofNullable(clazz.cast(updatedTime()));
        case "ReviewStatus":
            return Optional.ofNullable(clazz.cast(reviewStatusAsString()));
        case "Comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "Reviewer":
            return Optional.ofNullable(clazz.cast(reviewer()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentReviewerResponseSource, T> g) {
        return obj -> g.apply((DocumentReviewerResponseSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentReviewerResponseSource> {
        /**
         * <p>
         * The date and time that a reviewer entered a response to a document review request.
         * </p>
         * 
         * @param createTime
         *        The date and time that a reviewer entered a response to a document review request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The date and time that a reviewer last updated a response to a document review request.
         * </p>
         * 
         * @param updatedTime
         *        The date and time that a reviewer last updated a response to a document review request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedTime(Instant updatedTime);

        /**
         * <p>
         * The current review status of a new custom SSM document created by a member of your organization, or of the
         * latest version of an existing SSM document.
         * </p>
         * <p>
         * Only one version of a document can be in the APPROVED state at a time. When a new version is approved, the
         * status of the previous version changes to REJECTED.
         * </p>
         * <p>
         * Only one version of a document can be in review, or PENDING, at a time.
         * </p>
         * 
         * @param reviewStatus
         *        The current review status of a new custom SSM document created by a member of your organization, or of
         *        the latest version of an existing SSM document.</p>
         *        <p>
         *        Only one version of a document can be in the APPROVED state at a time. When a new version is approved,
         *        the status of the previous version changes to REJECTED.
         *        </p>
         *        <p>
         *        Only one version of a document can be in review, or PENDING, at a time.
         * @see ReviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReviewStatus
         */
        Builder reviewStatus(String reviewStatus);

        /**
         * <p>
         * The current review status of a new custom SSM document created by a member of your organization, or of the
         * latest version of an existing SSM document.
         * </p>
         * <p>
         * Only one version of a document can be in the APPROVED state at a time. When a new version is approved, the
         * status of the previous version changes to REJECTED.
         * </p>
         * <p>
         * Only one version of a document can be in review, or PENDING, at a time.
         * </p>
         * 
         * @param reviewStatus
         *        The current review status of a new custom SSM document created by a member of your organization, or of
         *        the latest version of an existing SSM document.</p>
         *        <p>
         *        Only one version of a document can be in the APPROVED state at a time. When a new version is approved,
         *        the status of the previous version changes to REJECTED.
         *        </p>
         *        <p>
         *        Only one version of a document can be in review, or PENDING, at a time.
         * @see ReviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReviewStatus
         */
        Builder reviewStatus(ReviewStatus reviewStatus);

        /**
         * <p>
         * The comment entered by a reviewer as part of their document review response.
         * </p>
         * 
         * @param comment
         *        The comment entered by a reviewer as part of their document review response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(Collection<DocumentReviewCommentSource> comment);

        /**
         * <p>
         * The comment entered by a reviewer as part of their document review response.
         * </p>
         * 
         * @param comment
         *        The comment entered by a reviewer as part of their document review response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(DocumentReviewCommentSource... comment);

        /**
         * <p>
         * The comment entered by a reviewer as part of their document review response.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DocumentReviewCommentSource>.Builder}
         * avoiding the need to create one manually via {@link List<DocumentReviewCommentSource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DocumentReviewCommentSource>.Builder#build()} is called
         * immediately and its result is passed to {@link #comment(List<DocumentReviewCommentSource>)}.
         * 
         * @param comment
         *        a consumer that will call methods on {@link List<DocumentReviewCommentSource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #comment(List<DocumentReviewCommentSource>)
         */
        Builder comment(Consumer<DocumentReviewCommentSource.Builder>... comment);

        /**
         * <p>
         * The user in your organization assigned to review a document request.
         * </p>
         * 
         * @param reviewer
         *        The user in your organization assigned to review a document request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reviewer(String reviewer);
    }

    static final class BuilderImpl implements Builder {
        private Instant createTime;

        private Instant updatedTime;

        private String reviewStatus;

        private List<DocumentReviewCommentSource> comment = DefaultSdkAutoConstructList.getInstance();

        private String reviewer;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentReviewerResponseSource model) {
            createTime(model.createTime);
            updatedTime(model.updatedTime);
            reviewStatus(model.reviewStatus);
            comment(model.comment);
            reviewer(model.reviewer);
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        public final Instant getUpdatedTime() {
            return updatedTime;
        }

        @Override
        public final Builder updatedTime(Instant updatedTime) {
            this.updatedTime = updatedTime;
            return this;
        }

        public final void setUpdatedTime(Instant updatedTime) {
            this.updatedTime = updatedTime;
        }

        public final String getReviewStatus() {
            return reviewStatus;
        }

        @Override
        public final Builder reviewStatus(String reviewStatus) {
            this.reviewStatus = reviewStatus;
            return this;
        }

        @Override
        public final Builder reviewStatus(ReviewStatus reviewStatus) {
            this.reviewStatus(reviewStatus == null ? null : reviewStatus.toString());
            return this;
        }

        public final void setReviewStatus(String reviewStatus) {
            this.reviewStatus = reviewStatus;
        }

        public final Collection<DocumentReviewCommentSource.Builder> getComment() {
            if (comment instanceof SdkAutoConstructList) {
                return null;
            }
            return comment != null ? comment.stream().map(DocumentReviewCommentSource::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder comment(Collection<DocumentReviewCommentSource> comment) {
            this.comment = DocumentReviewCommentListCopier.copy(comment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder comment(DocumentReviewCommentSource... comment) {
            comment(Arrays.asList(comment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder comment(Consumer<DocumentReviewCommentSource.Builder>... comment) {
            comment(Stream.of(comment).map(c -> DocumentReviewCommentSource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setComment(Collection<DocumentReviewCommentSource.BuilderImpl> comment) {
            this.comment = DocumentReviewCommentListCopier.copyFromBuilder(comment);
        }

        public final String getReviewer() {
            return reviewer;
        }

        @Override
        public final Builder reviewer(String reviewer) {
            this.reviewer = reviewer;
            return this;
        }

        public final void setReviewer(String reviewer) {
            this.reviewer = reviewer;
        }

        @Override
        public DocumentReviewerResponseSource build() {
            return new DocumentReviewerResponseSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
