/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about an the execution state of an Automation step.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StepExecution implements SdkPojo, Serializable, ToCopyableBuilder<StepExecution.Builder, StepExecution> {
    private static final SdkField<String> STEP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StepName").getter(getter(StepExecution::stepName)).setter(setter(Builder::stepName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepName").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(StepExecution::action)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<Long> TIMEOUT_SECONDS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("TimeoutSeconds").getter(getter(StepExecution::timeoutSeconds)).setter(setter(Builder::timeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutSeconds").build()).build();

    private static final SdkField<String> ON_FAILURE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OnFailure").getter(getter(StepExecution::onFailure)).setter(setter(Builder::onFailure))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OnFailure").build()).build();

    private static final SdkField<Integer> MAX_ATTEMPTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxAttempts").getter(getter(StepExecution::maxAttempts)).setter(setter(Builder::maxAttempts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxAttempts").build()).build();

    private static final SdkField<Instant> EXECUTION_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExecutionStartTime").getter(getter(StepExecution::executionStartTime))
            .setter(setter(Builder::executionStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionStartTime").build())
            .build();

    private static final SdkField<Instant> EXECUTION_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExecutionEndTime").getter(getter(StepExecution::executionEndTime))
            .setter(setter(Builder::executionEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionEndTime").build()).build();

    private static final SdkField<String> STEP_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StepStatus").getter(getter(StepExecution::stepStatusAsString)).setter(setter(Builder::stepStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepStatus").build()).build();

    private static final SdkField<String> RESPONSE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResponseCode").getter(getter(StepExecution::responseCode)).setter(setter(Builder::responseCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseCode").build()).build();

    private static final SdkField<Map<String, String>> INPUTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Inputs")
            .getter(getter(StepExecution::inputs))
            .setter(setter(Builder::inputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Inputs").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, List<String>>> OUTPUTS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("Outputs")
            .getter(getter(StepExecution::outputs))
            .setter(setter(Builder::outputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Outputs").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> RESPONSE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Response").getter(getter(StepExecution::response)).setter(setter(Builder::response))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Response").build()).build();

    private static final SdkField<String> FAILURE_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureMessage").getter(getter(StepExecution::failureMessage)).setter(setter(Builder::failureMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureMessage").build()).build();

    private static final SdkField<FailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<FailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(StepExecution::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(FailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final SdkField<String> STEP_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StepExecutionId").getter(getter(StepExecution::stepExecutionId))
            .setter(setter(Builder::stepExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepExecutionId").build()).build();

    private static final SdkField<Map<String, List<String>>> OVERRIDDEN_PARAMETERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("OverriddenParameters")
            .getter(getter(StepExecution::overriddenParameters))
            .setter(setter(Builder::overriddenParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OverriddenParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Boolean> IS_END_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN).memberName("IsEnd")
            .getter(getter(StepExecution::isEnd)).setter(setter(Builder::isEnd))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsEnd").build()).build();

    private static final SdkField<String> NEXT_STEP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextStep").getter(getter(StepExecution::nextStep)).setter(setter(Builder::nextStep))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextStep").build()).build();

    private static final SdkField<Boolean> IS_CRITICAL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsCritical").getter(getter(StepExecution::isCritical)).setter(setter(Builder::isCritical))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsCritical").build()).build();

    private static final SdkField<List<String>> VALID_NEXT_STEPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ValidNextSteps")
            .getter(getter(StepExecution::validNextSteps))
            .setter(setter(Builder::validNextSteps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidNextSteps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(StepExecution::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TargetLocation> TARGET_LOCATION_FIELD = SdkField
            .<TargetLocation> builder(MarshallingType.SDK_POJO).memberName("TargetLocation")
            .getter(getter(StepExecution::targetLocation)).setter(setter(Builder::targetLocation))
            .constructor(TargetLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLocation").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STEP_NAME_FIELD, ACTION_FIELD,
            TIMEOUT_SECONDS_FIELD, ON_FAILURE_FIELD, MAX_ATTEMPTS_FIELD, EXECUTION_START_TIME_FIELD, EXECUTION_END_TIME_FIELD,
            STEP_STATUS_FIELD, RESPONSE_CODE_FIELD, INPUTS_FIELD, OUTPUTS_FIELD, RESPONSE_FIELD, FAILURE_MESSAGE_FIELD,
            FAILURE_DETAILS_FIELD, STEP_EXECUTION_ID_FIELD, OVERRIDDEN_PARAMETERS_FIELD, IS_END_FIELD, NEXT_STEP_FIELD,
            IS_CRITICAL_FIELD, VALID_NEXT_STEPS_FIELD, TARGETS_FIELD, TARGET_LOCATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stepName;

    private final String action;

    private final Long timeoutSeconds;

    private final String onFailure;

    private final Integer maxAttempts;

    private final Instant executionStartTime;

    private final Instant executionEndTime;

    private final String stepStatus;

    private final String responseCode;

    private final Map<String, String> inputs;

    private final Map<String, List<String>> outputs;

    private final String response;

    private final String failureMessage;

    private final FailureDetails failureDetails;

    private final String stepExecutionId;

    private final Map<String, List<String>> overriddenParameters;

    private final Boolean isEnd;

    private final String nextStep;

    private final Boolean isCritical;

    private final List<String> validNextSteps;

    private final List<Target> targets;

    private final TargetLocation targetLocation;

    private StepExecution(BuilderImpl builder) {
        this.stepName = builder.stepName;
        this.action = builder.action;
        this.timeoutSeconds = builder.timeoutSeconds;
        this.onFailure = builder.onFailure;
        this.maxAttempts = builder.maxAttempts;
        this.executionStartTime = builder.executionStartTime;
        this.executionEndTime = builder.executionEndTime;
        this.stepStatus = builder.stepStatus;
        this.responseCode = builder.responseCode;
        this.inputs = builder.inputs;
        this.outputs = builder.outputs;
        this.response = builder.response;
        this.failureMessage = builder.failureMessage;
        this.failureDetails = builder.failureDetails;
        this.stepExecutionId = builder.stepExecutionId;
        this.overriddenParameters = builder.overriddenParameters;
        this.isEnd = builder.isEnd;
        this.nextStep = builder.nextStep;
        this.isCritical = builder.isCritical;
        this.validNextSteps = builder.validNextSteps;
        this.targets = builder.targets;
        this.targetLocation = builder.targetLocation;
    }

    /**
     * <p>
     * The name of this execution step.
     * </p>
     * 
     * @return The name of this execution step.
     */
    public String stepName() {
        return stepName;
    }

    /**
     * <p>
     * The action this step performs. The action determines the behavior of the step.
     * </p>
     * 
     * @return The action this step performs. The action determines the behavior of the step.
     */
    public String action() {
        return action;
    }

    /**
     * <p>
     * The timeout seconds of the step.
     * </p>
     * 
     * @return The timeout seconds of the step.
     */
    public Long timeoutSeconds() {
        return timeoutSeconds;
    }

    /**
     * <p>
     * The action to take if the step fails. The default value is Abort.
     * </p>
     * 
     * @return The action to take if the step fails. The default value is Abort.
     */
    public String onFailure() {
        return onFailure;
    }

    /**
     * <p>
     * The maximum number of tries to run the action of the step. The default value is 1.
     * </p>
     * 
     * @return The maximum number of tries to run the action of the step. The default value is 1.
     */
    public Integer maxAttempts() {
        return maxAttempts;
    }

    /**
     * <p>
     * If a step has begun execution, this contains the time the step started. If the step is in Pending status, this
     * field is not populated.
     * </p>
     * 
     * @return If a step has begun execution, this contains the time the step started. If the step is in Pending status,
     *         this field is not populated.
     */
    public Instant executionStartTime() {
        return executionStartTime;
    }

    /**
     * <p>
     * If a step has finished execution, this contains the time the execution ended. If the step has not yet concluded,
     * this field is not populated.
     * </p>
     * 
     * @return If a step has finished execution, this contains the time the execution ended. If the step has not yet
     *         concluded, this field is not populated.
     */
    public Instant executionEndTime() {
        return executionEndTime;
    }

    /**
     * <p>
     * The execution status for this step.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stepStatus} will
     * return {@link AutomationExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stepStatusAsString}.
     * </p>
     * 
     * @return The execution status for this step.
     * @see AutomationExecutionStatus
     */
    public AutomationExecutionStatus stepStatus() {
        return AutomationExecutionStatus.fromValue(stepStatus);
    }

    /**
     * <p>
     * The execution status for this step.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stepStatus} will
     * return {@link AutomationExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stepStatusAsString}.
     * </p>
     * 
     * @return The execution status for this step.
     * @see AutomationExecutionStatus
     */
    public String stepStatusAsString() {
        return stepStatus;
    }

    /**
     * <p>
     * The response code returned by the execution of the step.
     * </p>
     * 
     * @return The response code returned by the execution of the step.
     */
    public String responseCode() {
        return responseCode;
    }

    /**
     * Returns true if the Inputs property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasInputs() {
        return inputs != null && !(inputs instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Fully-resolved values passed into the step before execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInputs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Fully-resolved values passed into the step before execution.
     */
    public Map<String, String> inputs() {
        return inputs;
    }

    /**
     * Returns true if the Outputs property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasOutputs() {
        return outputs != null && !(outputs instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Returned values from the execution of the step.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOutputs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Returned values from the execution of the step.
     */
    public Map<String, List<String>> outputs() {
        return outputs;
    }

    /**
     * <p>
     * A message associated with the response code for an execution.
     * </p>
     * 
     * @return A message associated with the response code for an execution.
     */
    public String response() {
        return response;
    }

    /**
     * <p>
     * If a step failed, this message explains why the execution failed.
     * </p>
     * 
     * @return If a step failed, this message explains why the execution failed.
     */
    public String failureMessage() {
        return failureMessage;
    }

    /**
     * <p>
     * Information about the Automation failure.
     * </p>
     * 
     * @return Information about the Automation failure.
     */
    public FailureDetails failureDetails() {
        return failureDetails;
    }

    /**
     * <p>
     * The unique ID of a step execution.
     * </p>
     * 
     * @return The unique ID of a step execution.
     */
    public String stepExecutionId() {
        return stepExecutionId;
    }

    /**
     * Returns true if the OverriddenParameters property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasOverriddenParameters() {
        return overriddenParameters != null && !(overriddenParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A user-specified list of parameters to override when running a step.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOverriddenParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A user-specified list of parameters to override when running a step.
     */
    public Map<String, List<String>> overriddenParameters() {
        return overriddenParameters;
    }

    /**
     * <p>
     * The flag which can be used to end automation no matter whether the step succeeds or fails.
     * </p>
     * 
     * @return The flag which can be used to end automation no matter whether the step succeeds or fails.
     */
    public Boolean isEnd() {
        return isEnd;
    }

    /**
     * <p>
     * The next step after the step succeeds.
     * </p>
     * 
     * @return The next step after the step succeeds.
     */
    public String nextStep() {
        return nextStep;
    }

    /**
     * <p>
     * The flag which can be used to help decide whether the failure of current step leads to the Automation failure.
     * </p>
     * 
     * @return The flag which can be used to help decide whether the failure of current step leads to the Automation
     *         failure.
     */
    public Boolean isCritical() {
        return isCritical;
    }

    /**
     * Returns true if the ValidNextSteps property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasValidNextSteps() {
        return validNextSteps != null && !(validNextSteps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Strategies used when step fails, we support Continue and Abort. Abort will fail the automation when the step
     * fails. Continue will ignore the failure of current step and allow automation to run the next step. With
     * conditional branching, we add step:stepName to support the automation to go to another specific step.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValidNextSteps()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Strategies used when step fails, we support Continue and Abort. Abort will fail the automation when the
     *         step fails. Continue will ignore the failure of current step and allow automation to run the next step.
     *         With conditional branching, we add step:stepName to support the automation to go to another specific
     *         step.
     */
    public List<String> validNextSteps() {
        return validNextSteps;
    }

    /**
     * Returns true if the Targets property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The targets for the step execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The targets for the step execution.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The combination of AWS Regions and accounts targeted by the current Automation execution.
     * </p>
     * 
     * @return The combination of AWS Regions and accounts targeted by the current Automation execution.
     */
    public TargetLocation targetLocation() {
        return targetLocation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stepName());
        hashCode = 31 * hashCode + Objects.hashCode(action());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(onFailure());
        hashCode = 31 * hashCode + Objects.hashCode(maxAttempts());
        hashCode = 31 * hashCode + Objects.hashCode(executionStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(executionEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(stepStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(responseCode());
        hashCode = 31 * hashCode + Objects.hashCode(inputs());
        hashCode = 31 * hashCode + Objects.hashCode(outputs());
        hashCode = 31 * hashCode + Objects.hashCode(response());
        hashCode = 31 * hashCode + Objects.hashCode(failureMessage());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        hashCode = 31 * hashCode + Objects.hashCode(stepExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(overriddenParameters());
        hashCode = 31 * hashCode + Objects.hashCode(isEnd());
        hashCode = 31 * hashCode + Objects.hashCode(nextStep());
        hashCode = 31 * hashCode + Objects.hashCode(isCritical());
        hashCode = 31 * hashCode + Objects.hashCode(validNextSteps());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(targetLocation());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StepExecution)) {
            return false;
        }
        StepExecution other = (StepExecution) obj;
        return Objects.equals(stepName(), other.stepName()) && Objects.equals(action(), other.action())
                && Objects.equals(timeoutSeconds(), other.timeoutSeconds()) && Objects.equals(onFailure(), other.onFailure())
                && Objects.equals(maxAttempts(), other.maxAttempts())
                && Objects.equals(executionStartTime(), other.executionStartTime())
                && Objects.equals(executionEndTime(), other.executionEndTime())
                && Objects.equals(stepStatusAsString(), other.stepStatusAsString())
                && Objects.equals(responseCode(), other.responseCode()) && Objects.equals(inputs(), other.inputs())
                && Objects.equals(outputs(), other.outputs()) && Objects.equals(response(), other.response())
                && Objects.equals(failureMessage(), other.failureMessage())
                && Objects.equals(failureDetails(), other.failureDetails())
                && Objects.equals(stepExecutionId(), other.stepExecutionId())
                && Objects.equals(overriddenParameters(), other.overriddenParameters()) && Objects.equals(isEnd(), other.isEnd())
                && Objects.equals(nextStep(), other.nextStep()) && Objects.equals(isCritical(), other.isCritical())
                && Objects.equals(validNextSteps(), other.validNextSteps()) && Objects.equals(targets(), other.targets())
                && Objects.equals(targetLocation(), other.targetLocation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("StepExecution").add("StepName", stepName()).add("Action", action())
                .add("TimeoutSeconds", timeoutSeconds()).add("OnFailure", onFailure()).add("MaxAttempts", maxAttempts())
                .add("ExecutionStartTime", executionStartTime()).add("ExecutionEndTime", executionEndTime())
                .add("StepStatus", stepStatusAsString()).add("ResponseCode", responseCode()).add("Inputs", inputs())
                .add("Outputs", outputs()).add("Response", response()).add("FailureMessage", failureMessage())
                .add("FailureDetails", failureDetails()).add("StepExecutionId", stepExecutionId())
                .add("OverriddenParameters", overriddenParameters()).add("IsEnd", isEnd()).add("NextStep", nextStep())
                .add("IsCritical", isCritical()).add("ValidNextSteps", validNextSteps()).add("Targets", targets())
                .add("TargetLocation", targetLocation()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StepName":
            return Optional.ofNullable(clazz.cast(stepName()));
        case "Action":
            return Optional.ofNullable(clazz.cast(action()));
        case "TimeoutSeconds":
            return Optional.ofNullable(clazz.cast(timeoutSeconds()));
        case "OnFailure":
            return Optional.ofNullable(clazz.cast(onFailure()));
        case "MaxAttempts":
            return Optional.ofNullable(clazz.cast(maxAttempts()));
        case "ExecutionStartTime":
            return Optional.ofNullable(clazz.cast(executionStartTime()));
        case "ExecutionEndTime":
            return Optional.ofNullable(clazz.cast(executionEndTime()));
        case "StepStatus":
            return Optional.ofNullable(clazz.cast(stepStatusAsString()));
        case "ResponseCode":
            return Optional.ofNullable(clazz.cast(responseCode()));
        case "Inputs":
            return Optional.ofNullable(clazz.cast(inputs()));
        case "Outputs":
            return Optional.ofNullable(clazz.cast(outputs()));
        case "Response":
            return Optional.ofNullable(clazz.cast(response()));
        case "FailureMessage":
            return Optional.ofNullable(clazz.cast(failureMessage()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        case "StepExecutionId":
            return Optional.ofNullable(clazz.cast(stepExecutionId()));
        case "OverriddenParameters":
            return Optional.ofNullable(clazz.cast(overriddenParameters()));
        case "IsEnd":
            return Optional.ofNullable(clazz.cast(isEnd()));
        case "NextStep":
            return Optional.ofNullable(clazz.cast(nextStep()));
        case "IsCritical":
            return Optional.ofNullable(clazz.cast(isCritical()));
        case "ValidNextSteps":
            return Optional.ofNullable(clazz.cast(validNextSteps()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "TargetLocation":
            return Optional.ofNullable(clazz.cast(targetLocation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StepExecution, T> g) {
        return obj -> g.apply((StepExecution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StepExecution> {
        /**
         * <p>
         * The name of this execution step.
         * </p>
         * 
         * @param stepName
         *        The name of this execution step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepName(String stepName);

        /**
         * <p>
         * The action this step performs. The action determines the behavior of the step.
         * </p>
         * 
         * @param action
         *        The action this step performs. The action determines the behavior of the step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder action(String action);

        /**
         * <p>
         * The timeout seconds of the step.
         * </p>
         * 
         * @param timeoutSeconds
         *        The timeout seconds of the step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutSeconds(Long timeoutSeconds);

        /**
         * <p>
         * The action to take if the step fails. The default value is Abort.
         * </p>
         * 
         * @param onFailure
         *        The action to take if the step fails. The default value is Abort.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onFailure(String onFailure);

        /**
         * <p>
         * The maximum number of tries to run the action of the step. The default value is 1.
         * </p>
         * 
         * @param maxAttempts
         *        The maximum number of tries to run the action of the step. The default value is 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxAttempts(Integer maxAttempts);

        /**
         * <p>
         * If a step has begun execution, this contains the time the step started. If the step is in Pending status,
         * this field is not populated.
         * </p>
         * 
         * @param executionStartTime
         *        If a step has begun execution, this contains the time the step started. If the step is in Pending
         *        status, this field is not populated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionStartTime(Instant executionStartTime);

        /**
         * <p>
         * If a step has finished execution, this contains the time the execution ended. If the step has not yet
         * concluded, this field is not populated.
         * </p>
         * 
         * @param executionEndTime
         *        If a step has finished execution, this contains the time the execution ended. If the step has not yet
         *        concluded, this field is not populated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionEndTime(Instant executionEndTime);

        /**
         * <p>
         * The execution status for this step.
         * </p>
         * 
         * @param stepStatus
         *        The execution status for this step.
         * @see AutomationExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationExecutionStatus
         */
        Builder stepStatus(String stepStatus);

        /**
         * <p>
         * The execution status for this step.
         * </p>
         * 
         * @param stepStatus
         *        The execution status for this step.
         * @see AutomationExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationExecutionStatus
         */
        Builder stepStatus(AutomationExecutionStatus stepStatus);

        /**
         * <p>
         * The response code returned by the execution of the step.
         * </p>
         * 
         * @param responseCode
         *        The response code returned by the execution of the step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCode(String responseCode);

        /**
         * <p>
         * Fully-resolved values passed into the step before execution.
         * </p>
         * 
         * @param inputs
         *        Fully-resolved values passed into the step before execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputs(Map<String, String> inputs);

        /**
         * <p>
         * Returned values from the execution of the step.
         * </p>
         * 
         * @param outputs
         *        Returned values from the execution of the step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Map<String, ? extends Collection<String>> outputs);

        /**
         * <p>
         * A message associated with the response code for an execution.
         * </p>
         * 
         * @param response
         *        A message associated with the response code for an execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder response(String response);

        /**
         * <p>
         * If a step failed, this message explains why the execution failed.
         * </p>
         * 
         * @param failureMessage
         *        If a step failed, this message explains why the execution failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessage(String failureMessage);

        /**
         * <p>
         * Information about the Automation failure.
         * </p>
         * 
         * @param failureDetails
         *        Information about the Automation failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(FailureDetails failureDetails);

        /**
         * <p>
         * Information about the Automation failure.
         * </p>
         * This is a convenience that creates an instance of the {@link FailureDetails.Builder} avoiding the need to
         * create one manually via {@link FailureDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link FailureDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #failureDetails(FailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link FailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(FailureDetails)
         */
        default Builder failureDetails(Consumer<FailureDetails.Builder> failureDetails) {
            return failureDetails(FailureDetails.builder().applyMutation(failureDetails).build());
        }

        /**
         * <p>
         * The unique ID of a step execution.
         * </p>
         * 
         * @param stepExecutionId
         *        The unique ID of a step execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepExecutionId(String stepExecutionId);

        /**
         * <p>
         * A user-specified list of parameters to override when running a step.
         * </p>
         * 
         * @param overriddenParameters
         *        A user-specified list of parameters to override when running a step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overriddenParameters(Map<String, ? extends Collection<String>> overriddenParameters);

        /**
         * <p>
         * The flag which can be used to end automation no matter whether the step succeeds or fails.
         * </p>
         * 
         * @param isEnd
         *        The flag which can be used to end automation no matter whether the step succeeds or fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isEnd(Boolean isEnd);

        /**
         * <p>
         * The next step after the step succeeds.
         * </p>
         * 
         * @param nextStep
         *        The next step after the step succeeds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextStep(String nextStep);

        /**
         * <p>
         * The flag which can be used to help decide whether the failure of current step leads to the Automation
         * failure.
         * </p>
         * 
         * @param isCritical
         *        The flag which can be used to help decide whether the failure of current step leads to the Automation
         *        failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isCritical(Boolean isCritical);

        /**
         * <p>
         * Strategies used when step fails, we support Continue and Abort. Abort will fail the automation when the step
         * fails. Continue will ignore the failure of current step and allow automation to run the next step. With
         * conditional branching, we add step:stepName to support the automation to go to another specific step.
         * </p>
         * 
         * @param validNextSteps
         *        Strategies used when step fails, we support Continue and Abort. Abort will fail the automation when
         *        the step fails. Continue will ignore the failure of current step and allow automation to run the next
         *        step. With conditional branching, we add step:stepName to support the automation to go to another
         *        specific step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validNextSteps(Collection<String> validNextSteps);

        /**
         * <p>
         * Strategies used when step fails, we support Continue and Abort. Abort will fail the automation when the step
         * fails. Continue will ignore the failure of current step and allow automation to run the next step. With
         * conditional branching, we add step:stepName to support the automation to go to another specific step.
         * </p>
         * 
         * @param validNextSteps
         *        Strategies used when step fails, we support Continue and Abort. Abort will fail the automation when
         *        the step fails. Continue will ignore the failure of current step and allow automation to run the next
         *        step. With conditional branching, we add step:stepName to support the automation to go to another
         *        specific step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validNextSteps(String... validNextSteps);

        /**
         * <p>
         * The targets for the step execution.
         * </p>
         * 
         * @param targets
         *        The targets for the step execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets for the step execution.
         * </p>
         * 
         * @param targets
         *        The targets for the step execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets for the step execution.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The combination of AWS Regions and accounts targeted by the current Automation execution.
         * </p>
         * 
         * @param targetLocation
         *        The combination of AWS Regions and accounts targeted by the current Automation execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLocation(TargetLocation targetLocation);

        /**
         * <p>
         * The combination of AWS Regions and accounts targeted by the current Automation execution.
         * </p>
         * This is a convenience that creates an instance of the {@link TargetLocation.Builder} avoiding the need to
         * create one manually via {@link TargetLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link TargetLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #targetLocation(TargetLocation)}.
         * 
         * @param targetLocation
         *        a consumer that will call methods on {@link TargetLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetLocation(TargetLocation)
         */
        default Builder targetLocation(Consumer<TargetLocation.Builder> targetLocation) {
            return targetLocation(TargetLocation.builder().applyMutation(targetLocation).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String stepName;

        private String action;

        private Long timeoutSeconds;

        private String onFailure;

        private Integer maxAttempts;

        private Instant executionStartTime;

        private Instant executionEndTime;

        private String stepStatus;

        private String responseCode;

        private Map<String, String> inputs = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, List<String>> outputs = DefaultSdkAutoConstructMap.getInstance();

        private String response;

        private String failureMessage;

        private FailureDetails failureDetails;

        private String stepExecutionId;

        private Map<String, List<String>> overriddenParameters = DefaultSdkAutoConstructMap.getInstance();

        private Boolean isEnd;

        private String nextStep;

        private Boolean isCritical;

        private List<String> validNextSteps = DefaultSdkAutoConstructList.getInstance();

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private TargetLocation targetLocation;

        private BuilderImpl() {
        }

        private BuilderImpl(StepExecution model) {
            stepName(model.stepName);
            action(model.action);
            timeoutSeconds(model.timeoutSeconds);
            onFailure(model.onFailure);
            maxAttempts(model.maxAttempts);
            executionStartTime(model.executionStartTime);
            executionEndTime(model.executionEndTime);
            stepStatus(model.stepStatus);
            responseCode(model.responseCode);
            inputs(model.inputs);
            outputs(model.outputs);
            response(model.response);
            failureMessage(model.failureMessage);
            failureDetails(model.failureDetails);
            stepExecutionId(model.stepExecutionId);
            overriddenParameters(model.overriddenParameters);
            isEnd(model.isEnd);
            nextStep(model.nextStep);
            isCritical(model.isCritical);
            validNextSteps(model.validNextSteps);
            targets(model.targets);
            targetLocation(model.targetLocation);
        }

        public final String getStepName() {
            return stepName;
        }

        @Override
        public final Builder stepName(String stepName) {
            this.stepName = stepName;
            return this;
        }

        public final void setStepName(String stepName) {
            this.stepName = stepName;
        }

        public final String getAction() {
            return action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        public final Long getTimeoutSeconds() {
            return timeoutSeconds;
        }

        @Override
        public final Builder timeoutSeconds(Long timeoutSeconds) {
            this.timeoutSeconds = timeoutSeconds;
            return this;
        }

        public final void setTimeoutSeconds(Long timeoutSeconds) {
            this.timeoutSeconds = timeoutSeconds;
        }

        public final String getOnFailure() {
            return onFailure;
        }

        @Override
        public final Builder onFailure(String onFailure) {
            this.onFailure = onFailure;
            return this;
        }

        public final void setOnFailure(String onFailure) {
            this.onFailure = onFailure;
        }

        public final Integer getMaxAttempts() {
            return maxAttempts;
        }

        @Override
        public final Builder maxAttempts(Integer maxAttempts) {
            this.maxAttempts = maxAttempts;
            return this;
        }

        public final void setMaxAttempts(Integer maxAttempts) {
            this.maxAttempts = maxAttempts;
        }

        public final Instant getExecutionStartTime() {
            return executionStartTime;
        }

        @Override
        public final Builder executionStartTime(Instant executionStartTime) {
            this.executionStartTime = executionStartTime;
            return this;
        }

        public final void setExecutionStartTime(Instant executionStartTime) {
            this.executionStartTime = executionStartTime;
        }

        public final Instant getExecutionEndTime() {
            return executionEndTime;
        }

        @Override
        public final Builder executionEndTime(Instant executionEndTime) {
            this.executionEndTime = executionEndTime;
            return this;
        }

        public final void setExecutionEndTime(Instant executionEndTime) {
            this.executionEndTime = executionEndTime;
        }

        public final String getStepStatus() {
            return stepStatus;
        }

        @Override
        public final Builder stepStatus(String stepStatus) {
            this.stepStatus = stepStatus;
            return this;
        }

        @Override
        public final Builder stepStatus(AutomationExecutionStatus stepStatus) {
            this.stepStatus(stepStatus == null ? null : stepStatus.toString());
            return this;
        }

        public final void setStepStatus(String stepStatus) {
            this.stepStatus = stepStatus;
        }

        public final String getResponseCode() {
            return responseCode;
        }

        @Override
        public final Builder responseCode(String responseCode) {
            this.responseCode = responseCode;
            return this;
        }

        public final void setResponseCode(String responseCode) {
            this.responseCode = responseCode;
        }

        public final Map<String, String> getInputs() {
            return inputs;
        }

        @Override
        public final Builder inputs(Map<String, String> inputs) {
            this.inputs = NormalStringMapCopier.copy(inputs);
            return this;
        }

        public final void setInputs(Map<String, String> inputs) {
            this.inputs = NormalStringMapCopier.copy(inputs);
        }

        public final Map<String, ? extends Collection<String>> getOutputs() {
            return outputs;
        }

        @Override
        public final Builder outputs(Map<String, ? extends Collection<String>> outputs) {
            this.outputs = AutomationParameterMapCopier.copy(outputs);
            return this;
        }

        public final void setOutputs(Map<String, ? extends Collection<String>> outputs) {
            this.outputs = AutomationParameterMapCopier.copy(outputs);
        }

        public final String getResponse() {
            return response;
        }

        @Override
        public final Builder response(String response) {
            this.response = response;
            return this;
        }

        public final void setResponse(String response) {
            this.response = response;
        }

        public final String getFailureMessage() {
            return failureMessage;
        }

        @Override
        public final Builder failureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
            return this;
        }

        public final void setFailureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
        }

        public final FailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        @Override
        public final Builder failureDetails(FailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final void setFailureDetails(FailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        public final String getStepExecutionId() {
            return stepExecutionId;
        }

        @Override
        public final Builder stepExecutionId(String stepExecutionId) {
            this.stepExecutionId = stepExecutionId;
            return this;
        }

        public final void setStepExecutionId(String stepExecutionId) {
            this.stepExecutionId = stepExecutionId;
        }

        public final Map<String, ? extends Collection<String>> getOverriddenParameters() {
            return overriddenParameters;
        }

        @Override
        public final Builder overriddenParameters(Map<String, ? extends Collection<String>> overriddenParameters) {
            this.overriddenParameters = AutomationParameterMapCopier.copy(overriddenParameters);
            return this;
        }

        public final void setOverriddenParameters(Map<String, ? extends Collection<String>> overriddenParameters) {
            this.overriddenParameters = AutomationParameterMapCopier.copy(overriddenParameters);
        }

        public final Boolean getIsEnd() {
            return isEnd;
        }

        @Override
        public final Builder isEnd(Boolean isEnd) {
            this.isEnd = isEnd;
            return this;
        }

        public final void setIsEnd(Boolean isEnd) {
            this.isEnd = isEnd;
        }

        public final String getNextStep() {
            return nextStep;
        }

        @Override
        public final Builder nextStep(String nextStep) {
            this.nextStep = nextStep;
            return this;
        }

        public final void setNextStep(String nextStep) {
            this.nextStep = nextStep;
        }

        public final Boolean getIsCritical() {
            return isCritical;
        }

        @Override
        public final Builder isCritical(Boolean isCritical) {
            this.isCritical = isCritical;
            return this;
        }

        public final void setIsCritical(Boolean isCritical) {
            this.isCritical = isCritical;
        }

        public final Collection<String> getValidNextSteps() {
            return validNextSteps;
        }

        @Override
        public final Builder validNextSteps(Collection<String> validNextSteps) {
            this.validNextSteps = ValidNextStepListCopier.copy(validNextSteps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validNextSteps(String... validNextSteps) {
            validNextSteps(Arrays.asList(validNextSteps));
            return this;
        }

        public final void setValidNextSteps(Collection<String> validNextSteps) {
            this.validNextSteps = ValidNextStepListCopier.copy(validNextSteps);
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final TargetLocation.Builder getTargetLocation() {
            return targetLocation != null ? targetLocation.toBuilder() : null;
        }

        @Override
        public final Builder targetLocation(TargetLocation targetLocation) {
            this.targetLocation = targetLocation;
            return this;
        }

        public final void setTargetLocation(TargetLocation.BuilderImpl targetLocation) {
            this.targetLocation = targetLocation != null ? targetLocation.build() : null;
        }

        @Override
        public StepExecution build() {
            return new StepExecution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
