/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateOpsItemRequest extends SsmRequest implements
        ToCopyableBuilder<UpdateOpsItemRequest.Builder, UpdateOpsItemRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(UpdateOpsItemRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Map<String, OpsItemDataValue>> OPERATIONAL_DATA_FIELD = SdkField
            .<Map<String, OpsItemDataValue>> builder(MarshallingType.MAP)
            .memberName("OperationalData")
            .getter(getter(UpdateOpsItemRequest::operationalData))
            .setter(setter(Builder::operationalData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationalData").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<OpsItemDataValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpsItemDataValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> OPERATIONAL_DATA_TO_DELETE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("OperationalDataToDelete")
            .getter(getter(UpdateOpsItemRequest::operationalDataToDelete))
            .setter(setter(Builder::operationalDataToDelete))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationalDataToDelete").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<OpsItemNotification>> NOTIFICATIONS_FIELD = SdkField
            .<List<OpsItemNotification>> builder(MarshallingType.LIST)
            .memberName("Notifications")
            .getter(getter(UpdateOpsItemRequest::notifications))
            .setter(setter(Builder::notifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Notifications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OpsItemNotification> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpsItemNotification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority").getter(getter(UpdateOpsItemRequest::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<List<RelatedOpsItem>> RELATED_OPS_ITEMS_FIELD = SdkField
            .<List<RelatedOpsItem>> builder(MarshallingType.LIST)
            .memberName("RelatedOpsItems")
            .getter(getter(UpdateOpsItemRequest::relatedOpsItems))
            .setter(setter(Builder::relatedOpsItems))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedOpsItems").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RelatedOpsItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(RelatedOpsItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(UpdateOpsItemRequest::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> OPS_ITEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OpsItemId").getter(getter(UpdateOpsItemRequest::opsItemId)).setter(setter(Builder::opsItemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OpsItemId").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(UpdateOpsItemRequest::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Category").getter(getter(UpdateOpsItemRequest::category)).setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Category").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(UpdateOpsItemRequest::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            OPERATIONAL_DATA_FIELD, OPERATIONAL_DATA_TO_DELETE_FIELD, NOTIFICATIONS_FIELD, PRIORITY_FIELD,
            RELATED_OPS_ITEMS_FIELD, STATUS_FIELD, OPS_ITEM_ID_FIELD, TITLE_FIELD, CATEGORY_FIELD, SEVERITY_FIELD));

    private final String description;

    private final Map<String, OpsItemDataValue> operationalData;

    private final List<String> operationalDataToDelete;

    private final List<OpsItemNotification> notifications;

    private final Integer priority;

    private final List<RelatedOpsItem> relatedOpsItems;

    private final String status;

    private final String opsItemId;

    private final String title;

    private final String category;

    private final String severity;

    private UpdateOpsItemRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.operationalData = builder.operationalData;
        this.operationalDataToDelete = builder.operationalDataToDelete;
        this.notifications = builder.notifications;
        this.priority = builder.priority;
        this.relatedOpsItems = builder.relatedOpsItems;
        this.status = builder.status;
        this.opsItemId = builder.opsItemId;
        this.title = builder.title;
        this.category = builder.category;
        this.severity = builder.severity;
    }

    /**
     * <p>
     * Update the information about the OpsItem. Provide enough information so that users reading this OpsItem for the
     * first time understand the issue.
     * </p>
     * 
     * @return Update the information about the OpsItem. Provide enough information so that users reading this OpsItem
     *         for the first time understand the issue.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the OperationalData property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasOperationalData() {
        return operationalData != null && !(operationalData instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Add new keys or edit existing key-value pairs of the OperationalData map in the OpsItem object.
     * </p>
     * <p>
     * Operational data is custom data that provides useful reference details about the OpsItem. For example, you can
     * specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You enter
     * operational data as key-value pairs. The key has a maximum length of 128 characters. The value has a maximum size
     * of 20 KB.
     * </p>
     * <important>
     * <p>
     * Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws, /amzn, /ssm.
     * </p>
     * </important>
     * <p>
     * You can choose to make the data searchable by other users in the account or you can restrict search access.
     * Searchable data means that all users with access to the OpsItem Overview page (as provided by the
     * <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is not
     * searchable is only viewable by users who have access to the OpsItem (as provided by the <a>GetOpsItem</a> API
     * action).
     * </p>
     * <p>
     * Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request. Use the
     * <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the OpsItem. To view
     * AWS CLI example commands that use these keys, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
     * >Creating OpsItems manually</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOperationalData()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Add new keys or edit existing key-value pairs of the OperationalData map in the OpsItem object.</p>
     *         <p>
     *         Operational data is custom data that provides useful reference details about the OpsItem. For example,
     *         you can specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You
     *         enter operational data as key-value pairs. The key has a maximum length of 128 characters. The value has
     *         a maximum size of 20 KB.
     *         </p>
     *         <important>
     *         <p>
     *         Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws,
     *         /amzn, /ssm.
     *         </p>
     *         </important>
     *         <p>
     *         You can choose to make the data searchable by other users in the account or you can restrict search
     *         access. Searchable data means that all users with access to the OpsItem Overview page (as provided by the
     *         <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is
     *         not searchable is only viewable by users who have access to the OpsItem (as provided by the
     *         <a>GetOpsItem</a> API action).
     *         </p>
     *         <p>
     *         Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request.
     *         Use the <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the
     *         OpsItem. To view AWS CLI example commands that use these keys, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
     *         >Creating OpsItems manually</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public Map<String, OpsItemDataValue> operationalData() {
        return operationalData;
    }

    /**
     * Returns true if the OperationalDataToDelete property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasOperationalDataToDelete() {
        return operationalDataToDelete != null && !(operationalDataToDelete instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Keys that you want to remove from the OperationalData map.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOperationalDataToDelete()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Keys that you want to remove from the OperationalData map.
     */
    public List<String> operationalDataToDelete() {
        return operationalDataToDelete;
    }

    /**
     * Returns true if the Notifications property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasNotifications() {
        return notifications != null && !(notifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
     * changed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNotifications()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited
     *         or changed.
     */
    public List<OpsItemNotification> notifications() {
        return notifications;
    }

    /**
     * <p>
     * The importance of this OpsItem in relation to other OpsItems in the system.
     * </p>
     * 
     * @return The importance of this OpsItem in relation to other OpsItems in the system.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * Returns true if the RelatedOpsItems property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasRelatedOpsItems() {
        return relatedOpsItems != null && !(relatedOpsItems instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more OpsItems that share something in common with the current OpsItems. For example, related OpsItems can
     * include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRelatedOpsItems()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more OpsItems that share something in common with the current OpsItems. For example, related
     *         OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
     *         impacted resource.
     */
    public List<RelatedOpsItem> relatedOpsItems() {
        return relatedOpsItems;
    }

    /**
     * <p>
     * The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html#OpsCenter-working-with-OpsItems-editing-details"
     * >Editing OpsItem details</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OpsItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>.
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html#OpsCenter-working-with-OpsItems-editing-details"
     *         >Editing OpsItem details</a> in the <i>AWS Systems Manager User Guide</i>.
     * @see OpsItemStatus
     */
    public OpsItemStatus status() {
        return OpsItemStatus.fromValue(status);
    }

    /**
     * <p>
     * The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html#OpsCenter-working-with-OpsItems-editing-details"
     * >Editing OpsItem details</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OpsItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>.
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html#OpsCenter-working-with-OpsItems-editing-details"
     *         >Editing OpsItem details</a> in the <i>AWS Systems Manager User Guide</i>.
     * @see OpsItemStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The ID of the OpsItem.
     * </p>
     * 
     * @return The ID of the OpsItem.
     */
    public String opsItemId() {
        return opsItemId;
    }

    /**
     * <p>
     * A short heading that describes the nature of the OpsItem and the impacted resource.
     * </p>
     * 
     * @return A short heading that describes the nature of the OpsItem and the impacted resource.
     */
    public String title() {
        return title;
    }

    /**
     * <p>
     * Specify a new category for an OpsItem.
     * </p>
     * 
     * @return Specify a new category for an OpsItem.
     */
    public String category() {
        return category;
    }

    /**
     * <p>
     * Specify a new severity for an OpsItem.
     * </p>
     * 
     * @return Specify a new severity for an OpsItem.
     */
    public String severity() {
        return severity;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(operationalData());
        hashCode = 31 * hashCode + Objects.hashCode(operationalDataToDelete());
        hashCode = 31 * hashCode + Objects.hashCode(notifications());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(relatedOpsItems());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(opsItemId());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(category());
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateOpsItemRequest)) {
            return false;
        }
        UpdateOpsItemRequest other = (UpdateOpsItemRequest) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(operationalData(), other.operationalData())
                && Objects.equals(operationalDataToDelete(), other.operationalDataToDelete())
                && Objects.equals(notifications(), other.notifications()) && Objects.equals(priority(), other.priority())
                && Objects.equals(relatedOpsItems(), other.relatedOpsItems())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(opsItemId(), other.opsItemId())
                && Objects.equals(title(), other.title()) && Objects.equals(category(), other.category())
                && Objects.equals(severity(), other.severity());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("UpdateOpsItemRequest").add("Description", description())
                .add("OperationalData", operationalData()).add("OperationalDataToDelete", operationalDataToDelete())
                .add("Notifications", notifications()).add("Priority", priority()).add("RelatedOpsItems", relatedOpsItems())
                .add("Status", statusAsString()).add("OpsItemId", opsItemId()).add("Title", title()).add("Category", category())
                .add("Severity", severity()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OperationalData":
            return Optional.ofNullable(clazz.cast(operationalData()));
        case "OperationalDataToDelete":
            return Optional.ofNullable(clazz.cast(operationalDataToDelete()));
        case "Notifications":
            return Optional.ofNullable(clazz.cast(notifications()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "RelatedOpsItems":
            return Optional.ofNullable(clazz.cast(relatedOpsItems()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "OpsItemId":
            return Optional.ofNullable(clazz.cast(opsItemId()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Category":
            return Optional.ofNullable(clazz.cast(category()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateOpsItemRequest, T> g) {
        return obj -> g.apply((UpdateOpsItemRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateOpsItemRequest> {
        /**
         * <p>
         * Update the information about the OpsItem. Provide enough information so that users reading this OpsItem for
         * the first time understand the issue.
         * </p>
         * 
         * @param description
         *        Update the information about the OpsItem. Provide enough information so that users reading this
         *        OpsItem for the first time understand the issue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Add new keys or edit existing key-value pairs of the OperationalData map in the OpsItem object.
         * </p>
         * <p>
         * Operational data is custom data that provides useful reference details about the OpsItem. For example, you
         * can specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You enter
         * operational data as key-value pairs. The key has a maximum length of 128 characters. The value has a maximum
         * size of 20 KB.
         * </p>
         * <important>
         * <p>
         * Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws, /amzn,
         * /ssm.
         * </p>
         * </important>
         * <p>
         * You can choose to make the data searchable by other users in the account or you can restrict search access.
         * Searchable data means that all users with access to the OpsItem Overview page (as provided by the
         * <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is not
         * searchable is only viewable by users who have access to the OpsItem (as provided by the <a>GetOpsItem</a> API
         * action).
         * </p>
         * <p>
         * Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request. Use
         * the <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the OpsItem.
         * To view AWS CLI example commands that use these keys, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
         * >Creating OpsItems manually</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param operationalData
         *        Add new keys or edit existing key-value pairs of the OperationalData map in the OpsItem object.</p>
         *        <p>
         *        Operational data is custom data that provides useful reference details about the OpsItem. For example,
         *        you can specify log files, error strings, license keys, troubleshooting tips, or other relevant data.
         *        You enter operational data as key-value pairs. The key has a maximum length of 128 characters. The
         *        value has a maximum size of 20 KB.
         *        </p>
         *        <important>
         *        <p>
         *        Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws,
         *        /amzn, /ssm.
         *        </p>
         *        </important>
         *        <p>
         *        You can choose to make the data searchable by other users in the account or you can restrict search
         *        access. Searchable data means that all users with access to the OpsItem Overview page (as provided by
         *        the <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data
         *        that is not searchable is only viewable by users who have access to the OpsItem (as provided by the
         *        <a>GetOpsItem</a> API action).
         *        </p>
         *        <p>
         *        Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the
         *        request. Use the <code>/aws/automations</code> key in OperationalData to associate an Automation
         *        runbook with the OpsItem. To view AWS CLI example commands that use these keys, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
         *        >Creating OpsItems manually</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationalData(Map<String, OpsItemDataValue> operationalData);

        /**
         * <p>
         * Keys that you want to remove from the OperationalData map.
         * </p>
         * 
         * @param operationalDataToDelete
         *        Keys that you want to remove from the OperationalData map.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationalDataToDelete(Collection<String> operationalDataToDelete);

        /**
         * <p>
         * Keys that you want to remove from the OperationalData map.
         * </p>
         * 
         * @param operationalDataToDelete
         *        Keys that you want to remove from the OperationalData map.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationalDataToDelete(String... operationalDataToDelete);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * 
         * @param notifications
         *        The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is
         *        edited or changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(Collection<OpsItemNotification> notifications);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * 
         * @param notifications
         *        The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is
         *        edited or changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(OpsItemNotification... notifications);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * This is a convenience that creates an instance of the {@link List<OpsItemNotification>.Builder} avoiding the
         * need to create one manually via {@link List<OpsItemNotification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<OpsItemNotification>.Builder#build()} is called immediately
         * and its result is passed to {@link #notifications(List<OpsItemNotification>)}.
         * 
         * @param notifications
         *        a consumer that will call methods on {@link List<OpsItemNotification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notifications(List<OpsItemNotification>)
         */
        Builder notifications(Consumer<OpsItemNotification.Builder>... notifications);

        /**
         * <p>
         * The importance of this OpsItem in relation to other OpsItems in the system.
         * </p>
         * 
         * @param priority
         *        The importance of this OpsItem in relation to other OpsItems in the system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItems. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * 
         * @param relatedOpsItems
         *        One or more OpsItems that share something in common with the current OpsItems. For example, related
         *        OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
         *        impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedOpsItems(Collection<RelatedOpsItem> relatedOpsItems);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItems. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * 
         * @param relatedOpsItems
         *        One or more OpsItems that share something in common with the current OpsItems. For example, related
         *        OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
         *        impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedOpsItems(RelatedOpsItem... relatedOpsItems);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItems. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RelatedOpsItem>.Builder} avoiding the need
         * to create one manually via {@link List<RelatedOpsItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RelatedOpsItem>.Builder#build()} is called immediately and
         * its result is passed to {@link #relatedOpsItems(List<RelatedOpsItem>)}.
         * 
         * @param relatedOpsItems
         *        a consumer that will call methods on {@link List<RelatedOpsItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relatedOpsItems(List<RelatedOpsItem>)
         */
        Builder relatedOpsItems(Consumer<RelatedOpsItem.Builder>... relatedOpsItems);

        /**
         * <p>
         * The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html#OpsCenter-working-with-OpsItems-editing-details"
         * >Editing OpsItem details</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param status
         *        The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or
         *        <code>Resolved</code>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html#OpsCenter-working-with-OpsItems-editing-details"
         *        >Editing OpsItem details</a> in the <i>AWS Systems Manager User Guide</i>.
         * @see OpsItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpsItemStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html#OpsCenter-working-with-OpsItems-editing-details"
         * >Editing OpsItem details</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param status
         *        The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or
         *        <code>Resolved</code>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html#OpsCenter-working-with-OpsItems-editing-details"
         *        >Editing OpsItem details</a> in the <i>AWS Systems Manager User Guide</i>.
         * @see OpsItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpsItemStatus
         */
        Builder status(OpsItemStatus status);

        /**
         * <p>
         * The ID of the OpsItem.
         * </p>
         * 
         * @param opsItemId
         *        The ID of the OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder opsItemId(String opsItemId);

        /**
         * <p>
         * A short heading that describes the nature of the OpsItem and the impacted resource.
         * </p>
         * 
         * @param title
         *        A short heading that describes the nature of the OpsItem and the impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * Specify a new category for an OpsItem.
         * </p>
         * 
         * @param category
         *        Specify a new category for an OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder category(String category);

        /**
         * <p>
         * Specify a new severity for an OpsItem.
         * </p>
         * 
         * @param severity
         *        Specify a new severity for an OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(String severity);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String description;

        private Map<String, OpsItemDataValue> operationalData = DefaultSdkAutoConstructMap.getInstance();

        private List<String> operationalDataToDelete = DefaultSdkAutoConstructList.getInstance();

        private List<OpsItemNotification> notifications = DefaultSdkAutoConstructList.getInstance();

        private Integer priority;

        private List<RelatedOpsItem> relatedOpsItems = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String opsItemId;

        private String title;

        private String category;

        private String severity;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateOpsItemRequest model) {
            super(model);
            description(model.description);
            operationalData(model.operationalData);
            operationalDataToDelete(model.operationalDataToDelete);
            notifications(model.notifications);
            priority(model.priority);
            relatedOpsItems(model.relatedOpsItems);
            status(model.status);
            opsItemId(model.opsItemId);
            title(model.title);
            category(model.category);
            severity(model.severity);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Map<String, OpsItemDataValue.Builder> getOperationalData() {
            return operationalData != null ? CollectionUtils.mapValues(operationalData, OpsItemDataValue::toBuilder) : null;
        }

        @Override
        public final Builder operationalData(Map<String, OpsItemDataValue> operationalData) {
            this.operationalData = OpsItemOperationalDataCopier.copy(operationalData);
            return this;
        }

        public final void setOperationalData(Map<String, OpsItemDataValue.BuilderImpl> operationalData) {
            this.operationalData = OpsItemOperationalDataCopier.copyFromBuilder(operationalData);
        }

        public final Collection<String> getOperationalDataToDelete() {
            return operationalDataToDelete;
        }

        @Override
        public final Builder operationalDataToDelete(Collection<String> operationalDataToDelete) {
            this.operationalDataToDelete = OpsItemOpsDataKeysListCopier.copy(operationalDataToDelete);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operationalDataToDelete(String... operationalDataToDelete) {
            operationalDataToDelete(Arrays.asList(operationalDataToDelete));
            return this;
        }

        public final void setOperationalDataToDelete(Collection<String> operationalDataToDelete) {
            this.operationalDataToDelete = OpsItemOpsDataKeysListCopier.copy(operationalDataToDelete);
        }

        public final Collection<OpsItemNotification.Builder> getNotifications() {
            return notifications != null ? notifications.stream().map(OpsItemNotification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder notifications(Collection<OpsItemNotification> notifications) {
            this.notifications = OpsItemNotificationsCopier.copy(notifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(OpsItemNotification... notifications) {
            notifications(Arrays.asList(notifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(Consumer<OpsItemNotification.Builder>... notifications) {
            notifications(Stream.of(notifications).map(c -> OpsItemNotification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNotifications(Collection<OpsItemNotification.BuilderImpl> notifications) {
            this.notifications = OpsItemNotificationsCopier.copyFromBuilder(notifications);
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final Collection<RelatedOpsItem.Builder> getRelatedOpsItems() {
            return relatedOpsItems != null ? relatedOpsItems.stream().map(RelatedOpsItem::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder relatedOpsItems(Collection<RelatedOpsItem> relatedOpsItems) {
            this.relatedOpsItems = RelatedOpsItemsCopier.copy(relatedOpsItems);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedOpsItems(RelatedOpsItem... relatedOpsItems) {
            relatedOpsItems(Arrays.asList(relatedOpsItems));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedOpsItems(Consumer<RelatedOpsItem.Builder>... relatedOpsItems) {
            relatedOpsItems(Stream.of(relatedOpsItems).map(c -> RelatedOpsItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRelatedOpsItems(Collection<RelatedOpsItem.BuilderImpl> relatedOpsItems) {
            this.relatedOpsItems = RelatedOpsItemsCopier.copyFromBuilder(relatedOpsItems);
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(OpsItemStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getOpsItemId() {
            return opsItemId;
        }

        @Override
        public final Builder opsItemId(String opsItemId) {
            this.opsItemId = opsItemId;
            return this;
        }

        public final void setOpsItemId(String opsItemId) {
            this.opsItemId = opsItemId;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getCategory() {
            return category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        public final String getSeverity() {
            return severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateOpsItemRequest build() {
            return new UpdateOpsItemRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
