/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendCommandRequest extends SsmRequest implements
        ToCopyableBuilder<SendCommandRequest.Builder, SendCommandRequest> {
    private static final SdkField<List<String>> INSTANCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(SendCommandRequest::instanceIds))
            .setter(setter(Builder::instanceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .getter(getter(SendCommandRequest::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOCUMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::documentName)).setter(setter(Builder::documentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentName").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<String> DOCUMENT_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::documentHash)).setter(setter(Builder::documentHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentHash").build()).build();

    private static final SdkField<String> DOCUMENT_HASH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::documentHashTypeAsString)).setter(setter(Builder::documentHashType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentHashType").build()).build();

    private static final SdkField<Integer> TIMEOUT_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(SendCommandRequest::timeoutSeconds)).setter(setter(Builder::timeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutSeconds").build()).build();

    private static final SdkField<String> COMMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::comment)).setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comment").build()).build();

    private static final SdkField<Map<String, List<String>>> PARAMETERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .getter(getter(SendCommandRequest::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> OUTPUT_S3_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::outputS3Region)).setter(setter(Builder::outputS3Region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputS3Region").build()).build();

    private static final SdkField<String> OUTPUT_S3_BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::outputS3BucketName)).setter(setter(Builder::outputS3BucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputS3BucketName").build())
            .build();

    private static final SdkField<String> OUTPUT_S3_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::outputS3KeyPrefix)).setter(setter(Builder::outputS3KeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputS3KeyPrefix").build()).build();

    private static final SdkField<String> MAX_CONCURRENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::maxConcurrency)).setter(setter(Builder::maxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrency").build()).build();

    private static final SdkField<String> MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::maxErrors)).setter(setter(Builder::maxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxErrors").build()).build();

    private static final SdkField<String> SERVICE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendCommandRequest::serviceRoleArn)).setter(setter(Builder::serviceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceRoleArn").build()).build();

    private static final SdkField<NotificationConfig> NOTIFICATION_CONFIG_FIELD = SdkField
            .<NotificationConfig> builder(MarshallingType.SDK_POJO).getter(getter(SendCommandRequest::notificationConfig))
            .setter(setter(Builder::notificationConfig)).constructor(NotificationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationConfig").build())
            .build();

    private static final SdkField<CloudWatchOutputConfig> CLOUD_WATCH_OUTPUT_CONFIG_FIELD = SdkField
            .<CloudWatchOutputConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(SendCommandRequest::cloudWatchOutputConfig)).setter(setter(Builder::cloudWatchOutputConfig))
            .constructor(CloudWatchOutputConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchOutputConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_IDS_FIELD,
            TARGETS_FIELD, DOCUMENT_NAME_FIELD, DOCUMENT_VERSION_FIELD, DOCUMENT_HASH_FIELD, DOCUMENT_HASH_TYPE_FIELD,
            TIMEOUT_SECONDS_FIELD, COMMENT_FIELD, PARAMETERS_FIELD, OUTPUT_S3_REGION_FIELD, OUTPUT_S3_BUCKET_NAME_FIELD,
            OUTPUT_S3_KEY_PREFIX_FIELD, MAX_CONCURRENCY_FIELD, MAX_ERRORS_FIELD, SERVICE_ROLE_ARN_FIELD,
            NOTIFICATION_CONFIG_FIELD, CLOUD_WATCH_OUTPUT_CONFIG_FIELD));

    private final List<String> instanceIds;

    private final List<Target> targets;

    private final String documentName;

    private final String documentVersion;

    private final String documentHash;

    private final String documentHashType;

    private final Integer timeoutSeconds;

    private final String comment;

    private final Map<String, List<String>> parameters;

    private final String outputS3Region;

    private final String outputS3BucketName;

    private final String outputS3KeyPrefix;

    private final String maxConcurrency;

    private final String maxErrors;

    private final String serviceRoleArn;

    private final NotificationConfig notificationConfig;

    private final CloudWatchOutputConfig cloudWatchOutputConfig;

    private SendCommandRequest(BuilderImpl builder) {
        super(builder);
        this.instanceIds = builder.instanceIds;
        this.targets = builder.targets;
        this.documentName = builder.documentName;
        this.documentVersion = builder.documentVersion;
        this.documentHash = builder.documentHash;
        this.documentHashType = builder.documentHashType;
        this.timeoutSeconds = builder.timeoutSeconds;
        this.comment = builder.comment;
        this.parameters = builder.parameters;
        this.outputS3Region = builder.outputS3Region;
        this.outputS3BucketName = builder.outputS3BucketName;
        this.outputS3KeyPrefix = builder.outputS3KeyPrefix;
        this.maxConcurrency = builder.maxConcurrency;
        this.maxErrors = builder.maxErrors;
        this.serviceRoleArn = builder.serviceRoleArn;
        this.notificationConfig = builder.notificationConfig;
        this.cloudWatchOutputConfig = builder.cloudWatchOutputConfig;
    }

    /**
     * Returns true if the InstanceIds property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasInstanceIds() {
        return instanceIds != null && !(instanceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance IDs where the command should run. You can specify a maximum of 50 IDs. If you prefer not to list
     * individual instance IDs, you can instead send commands to a fleet of instances using the Targets parameter, which
     * accepts EC2 tags. For more information about how to use targets, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending Commands
     * to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInstanceIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The instance IDs where the command should run. You can specify a maximum of 50 IDs. If you prefer not to
     *         list individual instance IDs, you can instead send commands to a fleet of instances using the Targets
     *         parameter, which accepts EC2 tags. For more information about how to use targets, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
     *         Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public List<String> instanceIds() {
        return instanceIds;
    }

    /**
     * Returns true if the Targets property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Optional) An array of search criteria that targets instances using a Key,Value combination that you specify.
     * Targets is required if you don't provide one or more instance IDs in the call. For more information about how to
     * use targets, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending Commands
     * to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return (Optional) An array of search criteria that targets instances using a Key,Value combination that you
     *         specify. Targets is required if you don't provide one or more instance IDs in the call. For more
     *         information about how to use targets, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
     *         Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * Required. The name of the Systems Manager document to run. This can be a public document or a custom document.
     * </p>
     * 
     * @return Required. The name of the Systems Manager document to run. This can be a public document or a custom
     *         document.
     */
    public String documentName() {
        return documentName;
    }

    /**
     * <p>
     * The SSM document version to use in the request. You can specify $DEFAULT, $LATEST, or a specific version number.
     * If you run commands by using the AWS CLI, then you must escape the first two options by using a backslash. If you
     * specify a version number, then you don't need to use the backslash. For example:
     * </p>
     * <p>
     * --document-version "\$DEFAULT"
     * </p>
     * <p>
     * --document-version "\$LATEST"
     * </p>
     * <p>
     * --document-version "3"
     * </p>
     * 
     * @return The SSM document version to use in the request. You can specify $DEFAULT, $LATEST, or a specific version
     *         number. If you run commands by using the AWS CLI, then you must escape the first two options by using a
     *         backslash. If you specify a version number, then you don't need to use the backslash. For example:</p>
     *         <p>
     *         --document-version "\$DEFAULT"
     *         </p>
     *         <p>
     *         --document-version "\$LATEST"
     *         </p>
     *         <p>
     *         --document-version "3"
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The Sha256 or Sha1 hash created by the system when the document was created.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * 
     * @return The Sha256 or Sha1 hash created by the system when the document was created. </p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     */
    public String documentHash() {
        return documentHash;
    }

    /**
     * <p>
     * Sha256 or Sha1.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentHashType}
     * will return {@link DocumentHashType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentHashTypeAsString}.
     * </p>
     * 
     * @return Sha256 or Sha1.</p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     * @see DocumentHashType
     */
    public DocumentHashType documentHashType() {
        return DocumentHashType.fromValue(documentHashType);
    }

    /**
     * <p>
     * Sha256 or Sha1.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentHashType}
     * will return {@link DocumentHashType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentHashTypeAsString}.
     * </p>
     * 
     * @return Sha256 or Sha1.</p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     * @see DocumentHashType
     */
    public String documentHashTypeAsString() {
        return documentHashType;
    }

    /**
     * <p>
     * If this time is reached and the command has not already started running, it will not run.
     * </p>
     * 
     * @return If this time is reached and the command has not already started running, it will not run.
     */
    public Integer timeoutSeconds() {
        return timeoutSeconds;
    }

    /**
     * <p>
     * User-specified information about the command, such as a brief description of what the command should do.
     * </p>
     * 
     * @return User-specified information about the command, such as a brief description of what the command should do.
     */
    public String comment() {
        return comment;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The required and optional parameters specified in the document being run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The required and optional parameters specified in the document being run.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
     * automatically determines the Amazon S3 bucket region.
     * </p>
     * 
     * @return (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
     *         automatically determines the Amazon S3 bucket region.
     */
    public String outputS3Region() {
        return outputS3Region;
    }

    /**
     * <p>
     * The name of the S3 bucket where command execution responses should be stored.
     * </p>
     * 
     * @return The name of the S3 bucket where command execution responses should be stored.
     */
    public String outputS3BucketName() {
        return outputS3BucketName;
    }

    /**
     * <p>
     * The directory structure within the S3 bucket where the responses should be stored.
     * </p>
     * 
     * @return The directory structure within the S3 bucket where the responses should be stored.
     */
    public String outputS3KeyPrefix() {
        return outputS3KeyPrefix;
    }

    /**
     * <p>
     * (Optional) The maximum number of instances that are allowed to run the command at the same time. You can specify
     * a number such as 10 or a percentage such as 10%. The default value is 50. For more information about how to use
     * MaxConcurrency, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-velocity"
     * >Using Concurrency Controls</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * 
     * @return (Optional) The maximum number of instances that are allowed to run the command at the same time. You can
     *         specify a number such as 10 or a percentage such as 10%. The default value is 50. For more information
     *         about how to use MaxConcurrency, see <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-velocity"
     *         >Using Concurrency Controls</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The maximum number of errors allowed without the command failing. When the command fails one more time beyond the
     * value of MaxErrors, the systems stops sending the command to additional targets. You can specify a number like 10
     * or a percentage like 10%. The default value is 0. For more information about how to use MaxErrors, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-maxerrors"
     * >Using Error Controls</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * 
     * @return The maximum number of errors allowed without the command failing. When the command fails one more time
     *         beyond the value of MaxErrors, the systems stops sending the command to additional targets. You can
     *         specify a number like 10 or a percentage like 10%. The default value is 0. For more information about how
     *         to use MaxErrors, see <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-maxerrors"
     *         >Using Error Controls</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * The ARN of the IAM service role to use to publish Amazon Simple Notification Service (Amazon SNS) notifications
     * for Run Command commands.
     * </p>
     * 
     * @return The ARN of the IAM service role to use to publish Amazon Simple Notification Service (Amazon SNS)
     *         notifications for Run Command commands.
     */
    public String serviceRoleArn() {
        return serviceRoleArn;
    }

    /**
     * <p>
     * Configurations for sending notifications.
     * </p>
     * 
     * @return Configurations for sending notifications.
     */
    public NotificationConfig notificationConfig() {
        return notificationConfig;
    }

    /**
     * <p>
     * Enables Systems Manager to send Run Command output to Amazon CloudWatch Logs.
     * </p>
     * 
     * @return Enables Systems Manager to send Run Command output to Amazon CloudWatch Logs.
     */
    public CloudWatchOutputConfig cloudWatchOutputConfig() {
        return cloudWatchOutputConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(instanceIds());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentHash());
        hashCode = 31 * hashCode + Objects.hashCode(documentHashTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3Region());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3BucketName());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3KeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(notificationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchOutputConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendCommandRequest)) {
            return false;
        }
        SendCommandRequest other = (SendCommandRequest) obj;
        return Objects.equals(instanceIds(), other.instanceIds()) && Objects.equals(targets(), other.targets())
                && Objects.equals(documentName(), other.documentName())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(documentHash(), other.documentHash())
                && Objects.equals(documentHashTypeAsString(), other.documentHashTypeAsString())
                && Objects.equals(timeoutSeconds(), other.timeoutSeconds()) && Objects.equals(comment(), other.comment())
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(outputS3Region(), other.outputS3Region())
                && Objects.equals(outputS3BucketName(), other.outputS3BucketName())
                && Objects.equals(outputS3KeyPrefix(), other.outputS3KeyPrefix())
                && Objects.equals(maxConcurrency(), other.maxConcurrency()) && Objects.equals(maxErrors(), other.maxErrors())
                && Objects.equals(serviceRoleArn(), other.serviceRoleArn())
                && Objects.equals(notificationConfig(), other.notificationConfig())
                && Objects.equals(cloudWatchOutputConfig(), other.cloudWatchOutputConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SendCommandRequest").add("InstanceIds", instanceIds()).add("Targets", targets())
                .add("DocumentName", documentName()).add("DocumentVersion", documentVersion())
                .add("DocumentHash", documentHash()).add("DocumentHashType", documentHashTypeAsString())
                .add("TimeoutSeconds", timeoutSeconds()).add("Comment", comment()).add("Parameters", parameters())
                .add("OutputS3Region", outputS3Region()).add("OutputS3BucketName", outputS3BucketName())
                .add("OutputS3KeyPrefix", outputS3KeyPrefix()).add("MaxConcurrency", maxConcurrency())
                .add("MaxErrors", maxErrors()).add("ServiceRoleArn", serviceRoleArn())
                .add("NotificationConfig", notificationConfig()).add("CloudWatchOutputConfig", cloudWatchOutputConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceIds":
            return Optional.ofNullable(clazz.cast(instanceIds()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "DocumentName":
            return Optional.ofNullable(clazz.cast(documentName()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "DocumentHash":
            return Optional.ofNullable(clazz.cast(documentHash()));
        case "DocumentHashType":
            return Optional.ofNullable(clazz.cast(documentHashTypeAsString()));
        case "TimeoutSeconds":
            return Optional.ofNullable(clazz.cast(timeoutSeconds()));
        case "Comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "OutputS3Region":
            return Optional.ofNullable(clazz.cast(outputS3Region()));
        case "OutputS3BucketName":
            return Optional.ofNullable(clazz.cast(outputS3BucketName()));
        case "OutputS3KeyPrefix":
            return Optional.ofNullable(clazz.cast(outputS3KeyPrefix()));
        case "MaxConcurrency":
            return Optional.ofNullable(clazz.cast(maxConcurrency()));
        case "MaxErrors":
            return Optional.ofNullable(clazz.cast(maxErrors()));
        case "ServiceRoleArn":
            return Optional.ofNullable(clazz.cast(serviceRoleArn()));
        case "NotificationConfig":
            return Optional.ofNullable(clazz.cast(notificationConfig()));
        case "CloudWatchOutputConfig":
            return Optional.ofNullable(clazz.cast(cloudWatchOutputConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SendCommandRequest, T> g) {
        return obj -> g.apply((SendCommandRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, SendCommandRequest> {
        /**
         * <p>
         * The instance IDs where the command should run. You can specify a maximum of 50 IDs. If you prefer not to list
         * individual instance IDs, you can instead send commands to a fleet of instances using the Targets parameter,
         * which accepts EC2 tags. For more information about how to use targets, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         * Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param instanceIds
         *        The instance IDs where the command should run. You can specify a maximum of 50 IDs. If you prefer not
         *        to list individual instance IDs, you can instead send commands to a fleet of instances using the
         *        Targets parameter, which accepts EC2 tags. For more information about how to use targets, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         *        Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceIds(Collection<String> instanceIds);

        /**
         * <p>
         * The instance IDs where the command should run. You can specify a maximum of 50 IDs. If you prefer not to list
         * individual instance IDs, you can instead send commands to a fleet of instances using the Targets parameter,
         * which accepts EC2 tags. For more information about how to use targets, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         * Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param instanceIds
         *        The instance IDs where the command should run. You can specify a maximum of 50 IDs. If you prefer not
         *        to list individual instance IDs, you can instead send commands to a fleet of instances using the
         *        Targets parameter, which accepts EC2 tags. For more information about how to use targets, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         *        Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceIds(String... instanceIds);

        /**
         * <p>
         * (Optional) An array of search criteria that targets instances using a Key,Value combination that you specify.
         * Targets is required if you don't provide one or more instance IDs in the call. For more information about how
         * to use targets, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         * Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param targets
         *        (Optional) An array of search criteria that targets instances using a Key,Value combination that you
         *        specify. Targets is required if you don't provide one or more instance IDs in the call. For more
         *        information about how to use targets, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         *        Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * (Optional) An array of search criteria that targets instances using a Key,Value combination that you specify.
         * Targets is required if you don't provide one or more instance IDs in the call. For more information about how
         * to use targets, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         * Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param targets
         *        (Optional) An array of search criteria that targets instances using a Key,Value combination that you
         *        specify. Targets is required if you don't provide one or more instance IDs in the call. For more
         *        information about how to use targets, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         *        Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * (Optional) An array of search criteria that targets instances using a Key,Value combination that you specify.
         * Targets is required if you don't provide one or more instance IDs in the call. For more information about how
         * to use targets, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html">Sending
         * Commands to a Fleet</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * Required. The name of the Systems Manager document to run. This can be a public document or a custom
         * document.
         * </p>
         * 
         * @param documentName
         *        Required. The name of the Systems Manager document to run. This can be a public document or a custom
         *        document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * The SSM document version to use in the request. You can specify $DEFAULT, $LATEST, or a specific version
         * number. If you run commands by using the AWS CLI, then you must escape the first two options by using a
         * backslash. If you specify a version number, then you don't need to use the backslash. For example:
         * </p>
         * <p>
         * --document-version "\$DEFAULT"
         * </p>
         * <p>
         * --document-version "\$LATEST"
         * </p>
         * <p>
         * --document-version "3"
         * </p>
         * 
         * @param documentVersion
         *        The SSM document version to use in the request. You can specify $DEFAULT, $LATEST, or a specific
         *        version number. If you run commands by using the AWS CLI, then you must escape the first two options
         *        by using a backslash. If you specify a version number, then you don't need to use the backslash. For
         *        example:</p>
         *        <p>
         *        --document-version "\$DEFAULT"
         *        </p>
         *        <p>
         *        --document-version "\$LATEST"
         *        </p>
         *        <p>
         *        --document-version "3"
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The Sha256 or Sha1 hash created by the system when the document was created.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param documentHash
         *        The Sha256 or Sha1 hash created by the system when the document was created. </p> <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentHash(String documentHash);

        /**
         * <p>
         * Sha256 or Sha1.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param documentHashType
         *        Sha256 or Sha1.</p> <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @see DocumentHashType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentHashType
         */
        Builder documentHashType(String documentHashType);

        /**
         * <p>
         * Sha256 or Sha1.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param documentHashType
         *        Sha256 or Sha1.</p> <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @see DocumentHashType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentHashType
         */
        Builder documentHashType(DocumentHashType documentHashType);

        /**
         * <p>
         * If this time is reached and the command has not already started running, it will not run.
         * </p>
         * 
         * @param timeoutSeconds
         *        If this time is reached and the command has not already started running, it will not run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutSeconds(Integer timeoutSeconds);

        /**
         * <p>
         * User-specified information about the command, such as a brief description of what the command should do.
         * </p>
         * 
         * @param comment
         *        User-specified information about the command, such as a brief description of what the command should
         *        do.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * The required and optional parameters specified in the document being run.
         * </p>
         * 
         * @param parameters
         *        The required and optional parameters specified in the document being run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
         * automatically determines the Amazon S3 bucket region.
         * </p>
         * 
         * @param outputS3Region
         *        (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
         *        automatically determines the Amazon S3 bucket region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3Region(String outputS3Region);

        /**
         * <p>
         * The name of the S3 bucket where command execution responses should be stored.
         * </p>
         * 
         * @param outputS3BucketName
         *        The name of the S3 bucket where command execution responses should be stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3BucketName(String outputS3BucketName);

        /**
         * <p>
         * The directory structure within the S3 bucket where the responses should be stored.
         * </p>
         * 
         * @param outputS3KeyPrefix
         *        The directory structure within the S3 bucket where the responses should be stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3KeyPrefix(String outputS3KeyPrefix);

        /**
         * <p>
         * (Optional) The maximum number of instances that are allowed to run the command at the same time. You can
         * specify a number such as 10 or a percentage such as 10%. The default value is 50. For more information about
         * how to use MaxConcurrency, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-velocity"
         * >Using Concurrency Controls</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param maxConcurrency
         *        (Optional) The maximum number of instances that are allowed to run the command at the same time. You
         *        can specify a number such as 10 or a percentage such as 10%. The default value is 50. For more
         *        information about how to use MaxConcurrency, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-velocity"
         *        >Using Concurrency Controls</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The maximum number of errors allowed without the command failing. When the command fails one more time beyond
         * the value of MaxErrors, the systems stops sending the command to additional targets. You can specify a number
         * like 10 or a percentage like 10%. The default value is 0. For more information about how to use MaxErrors,
         * see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-maxerrors"
         * >Using Error Controls</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param maxErrors
         *        The maximum number of errors allowed without the command failing. When the command fails one more time
         *        beyond the value of MaxErrors, the systems stops sending the command to additional targets. You can
         *        specify a number like 10 or a percentage like 10%. The default value is 0. For more information about
         *        how to use MaxErrors, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-maxerrors"
         *        >Using Error Controls</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * The ARN of the IAM service role to use to publish Amazon Simple Notification Service (Amazon SNS)
         * notifications for Run Command commands.
         * </p>
         * 
         * @param serviceRoleArn
         *        The ARN of the IAM service role to use to publish Amazon Simple Notification Service (Amazon SNS)
         *        notifications for Run Command commands.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRoleArn(String serviceRoleArn);

        /**
         * <p>
         * Configurations for sending notifications.
         * </p>
         * 
         * @param notificationConfig
         *        Configurations for sending notifications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationConfig(NotificationConfig notificationConfig);

        /**
         * <p>
         * Configurations for sending notifications.
         * </p>
         * This is a convenience that creates an instance of the {@link NotificationConfig.Builder} avoiding the need to
         * create one manually via {@link NotificationConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link NotificationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #notificationConfig(NotificationConfig)}.
         * 
         * @param notificationConfig
         *        a consumer that will call methods on {@link NotificationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationConfig(NotificationConfig)
         */
        default Builder notificationConfig(Consumer<NotificationConfig.Builder> notificationConfig) {
            return notificationConfig(NotificationConfig.builder().applyMutation(notificationConfig).build());
        }

        /**
         * <p>
         * Enables Systems Manager to send Run Command output to Amazon CloudWatch Logs.
         * </p>
         * 
         * @param cloudWatchOutputConfig
         *        Enables Systems Manager to send Run Command output to Amazon CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchOutputConfig(CloudWatchOutputConfig cloudWatchOutputConfig);

        /**
         * <p>
         * Enables Systems Manager to send Run Command output to Amazon CloudWatch Logs.
         * </p>
         * This is a convenience that creates an instance of the {@link CloudWatchOutputConfig.Builder} avoiding the
         * need to create one manually via {@link CloudWatchOutputConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CloudWatchOutputConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #cloudWatchOutputConfig(CloudWatchOutputConfig)}.
         * 
         * @param cloudWatchOutputConfig
         *        a consumer that will call methods on {@link CloudWatchOutputConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchOutputConfig(CloudWatchOutputConfig)
         */
        default Builder cloudWatchOutputConfig(Consumer<CloudWatchOutputConfig.Builder> cloudWatchOutputConfig) {
            return cloudWatchOutputConfig(CloudWatchOutputConfig.builder().applyMutation(cloudWatchOutputConfig).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private List<String> instanceIds = DefaultSdkAutoConstructList.getInstance();

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String documentName;

        private String documentVersion;

        private String documentHash;

        private String documentHashType;

        private Integer timeoutSeconds;

        private String comment;

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private String outputS3Region;

        private String outputS3BucketName;

        private String outputS3KeyPrefix;

        private String maxConcurrency;

        private String maxErrors;

        private String serviceRoleArn;

        private NotificationConfig notificationConfig;

        private CloudWatchOutputConfig cloudWatchOutputConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(SendCommandRequest model) {
            super(model);
            instanceIds(model.instanceIds);
            targets(model.targets);
            documentName(model.documentName);
            documentVersion(model.documentVersion);
            documentHash(model.documentHash);
            documentHashType(model.documentHashType);
            timeoutSeconds(model.timeoutSeconds);
            comment(model.comment);
            parameters(model.parameters);
            outputS3Region(model.outputS3Region);
            outputS3BucketName(model.outputS3BucketName);
            outputS3KeyPrefix(model.outputS3KeyPrefix);
            maxConcurrency(model.maxConcurrency);
            maxErrors(model.maxErrors);
            serviceRoleArn(model.serviceRoleArn);
            notificationConfig(model.notificationConfig);
            cloudWatchOutputConfig(model.cloudWatchOutputConfig);
        }

        public final Collection<String> getInstanceIds() {
            return instanceIds;
        }

        @Override
        public final Builder instanceIds(Collection<String> instanceIds) {
            this.instanceIds = InstanceIdListCopier.copy(instanceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceIds(String... instanceIds) {
            instanceIds(Arrays.asList(instanceIds));
            return this;
        }

        public final void setInstanceIds(Collection<String> instanceIds) {
            this.instanceIds = InstanceIdListCopier.copy(instanceIds);
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final String getDocumentName() {
            return documentName;
        }

        @Override
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final String getDocumentHash() {
            return documentHash;
        }

        @Override
        public final Builder documentHash(String documentHash) {
            this.documentHash = documentHash;
            return this;
        }

        public final void setDocumentHash(String documentHash) {
            this.documentHash = documentHash;
        }

        public final String getDocumentHashTypeAsString() {
            return documentHashType;
        }

        @Override
        public final Builder documentHashType(String documentHashType) {
            this.documentHashType = documentHashType;
            return this;
        }

        @Override
        public final Builder documentHashType(DocumentHashType documentHashType) {
            this.documentHashType(documentHashType == null ? null : documentHashType.toString());
            return this;
        }

        public final void setDocumentHashType(String documentHashType) {
            this.documentHashType = documentHashType;
        }

        public final Integer getTimeoutSeconds() {
            return timeoutSeconds;
        }

        @Override
        public final Builder timeoutSeconds(Integer timeoutSeconds) {
            this.timeoutSeconds = timeoutSeconds;
            return this;
        }

        public final void setTimeoutSeconds(Integer timeoutSeconds) {
            this.timeoutSeconds = timeoutSeconds;
        }

        public final String getComment() {
            return comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final String getOutputS3Region() {
            return outputS3Region;
        }

        @Override
        public final Builder outputS3Region(String outputS3Region) {
            this.outputS3Region = outputS3Region;
            return this;
        }

        public final void setOutputS3Region(String outputS3Region) {
            this.outputS3Region = outputS3Region;
        }

        public final String getOutputS3BucketName() {
            return outputS3BucketName;
        }

        @Override
        public final Builder outputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
            return this;
        }

        public final void setOutputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
        }

        public final String getOutputS3KeyPrefix() {
            return outputS3KeyPrefix;
        }

        @Override
        public final Builder outputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
            return this;
        }

        public final void setOutputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        @Override
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        public final String getServiceRoleArn() {
            return serviceRoleArn;
        }

        @Override
        public final Builder serviceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
            return this;
        }

        public final void setServiceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
        }

        public final NotificationConfig.Builder getNotificationConfig() {
            return notificationConfig != null ? notificationConfig.toBuilder() : null;
        }

        @Override
        public final Builder notificationConfig(NotificationConfig notificationConfig) {
            this.notificationConfig = notificationConfig;
            return this;
        }

        public final void setNotificationConfig(NotificationConfig.BuilderImpl notificationConfig) {
            this.notificationConfig = notificationConfig != null ? notificationConfig.build() : null;
        }

        public final CloudWatchOutputConfig.Builder getCloudWatchOutputConfig() {
            return cloudWatchOutputConfig != null ? cloudWatchOutputConfig.toBuilder() : null;
        }

        @Override
        public final Builder cloudWatchOutputConfig(CloudWatchOutputConfig cloudWatchOutputConfig) {
            this.cloudWatchOutputConfig = cloudWatchOutputConfig;
            return this;
        }

        public final void setCloudWatchOutputConfig(CloudWatchOutputConfig.BuilderImpl cloudWatchOutputConfig) {
            this.cloudWatchOutputConfig = cloudWatchOutputConfig != null ? cloudWatchOutputConfig.build() : null;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SendCommandRequest build() {
            return new SendCommandRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
