/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an association of a Systems Manager document and an instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Association implements SdkPojo, Serializable, ToCopyableBuilder<Association.Builder, Association> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Association::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Association::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Association::associationId)).setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationId").build()).build();

    private static final SdkField<String> ASSOCIATION_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Association::associationVersion)).setter(setter(Builder::associationVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationVersion").build())
            .build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Association::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .getter(getter(Association::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_EXECUTION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Association::lastExecutionDate)).setter(setter(Builder::lastExecutionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastExecutionDate").build()).build();

    private static final SdkField<AssociationOverview> OVERVIEW_FIELD = SdkField
            .<AssociationOverview> builder(MarshallingType.SDK_POJO).getter(getter(Association::overview))
            .setter(setter(Builder::overview)).constructor(AssociationOverview::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Overview").build()).build();

    private static final SdkField<String> SCHEDULE_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Association::scheduleExpression)).setter(setter(Builder::scheduleExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleExpression").build())
            .build();

    private static final SdkField<String> ASSOCIATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Association::associationName)).setter(setter(Builder::associationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, INSTANCE_ID_FIELD,
            ASSOCIATION_ID_FIELD, ASSOCIATION_VERSION_FIELD, DOCUMENT_VERSION_FIELD, TARGETS_FIELD, LAST_EXECUTION_DATE_FIELD,
            OVERVIEW_FIELD, SCHEDULE_EXPRESSION_FIELD, ASSOCIATION_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String instanceId;

    private final String associationId;

    private final String associationVersion;

    private final String documentVersion;

    private final List<Target> targets;

    private final Instant lastExecutionDate;

    private final AssociationOverview overview;

    private final String scheduleExpression;

    private final String associationName;

    private Association(BuilderImpl builder) {
        this.name = builder.name;
        this.instanceId = builder.instanceId;
        this.associationId = builder.associationId;
        this.associationVersion = builder.associationVersion;
        this.documentVersion = builder.documentVersion;
        this.targets = builder.targets;
        this.lastExecutionDate = builder.lastExecutionDate;
        this.overview = builder.overview;
        this.scheduleExpression = builder.scheduleExpression;
        this.associationName = builder.associationName;
    }

    /**
     * <p>
     * The name of the Systems Manager document.
     * </p>
     * 
     * @return The name of the Systems Manager document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the instance.
     * </p>
     * 
     * @return The ID of the instance.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The ID created by the system when you create an association. An association is a binding between a document and a
     * set of targets with a schedule.
     * </p>
     * 
     * @return The ID created by the system when you create an association. An association is a binding between a
     *         document and a set of targets with a schedule.
     */
    public String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The association version.
     * </p>
     * 
     * @return The association version.
     */
    public String associationVersion() {
        return associationVersion;
    }

    /**
     * <p>
     * The version of the document used in the association.
     * </p>
     * 
     * @return The version of the document used in the association.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The instances targeted by the request to create an association.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The instances targeted by the request to create an association.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The date on which the association was last run.
     * </p>
     * 
     * @return The date on which the association was last run.
     */
    public Instant lastExecutionDate() {
        return lastExecutionDate;
    }

    /**
     * <p>
     * Information about the association.
     * </p>
     * 
     * @return Information about the association.
     */
    public AssociationOverview overview() {
        return overview;
    }

    /**
     * <p>
     * A cron expression that specifies a schedule when the association runs.
     * </p>
     * 
     * @return A cron expression that specifies a schedule when the association runs.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * The association name.
     * </p>
     * 
     * @return The association name.
     */
    public String associationName() {
        return associationName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(associationVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(lastExecutionDate());
        hashCode = 31 * hashCode + Objects.hashCode(overview());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Association)) {
            return false;
        }
        Association other = (Association) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(associationId(), other.associationId())
                && Objects.equals(associationVersion(), other.associationVersion())
                && Objects.equals(documentVersion(), other.documentVersion()) && Objects.equals(targets(), other.targets())
                && Objects.equals(lastExecutionDate(), other.lastExecutionDate()) && Objects.equals(overview(), other.overview())
                && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(associationName(), other.associationName());
    }

    @Override
    public String toString() {
        return ToString.builder("Association").add("Name", name()).add("InstanceId", instanceId())
                .add("AssociationId", associationId()).add("AssociationVersion", associationVersion())
                .add("DocumentVersion", documentVersion()).add("Targets", targets())
                .add("LastExecutionDate", lastExecutionDate()).add("Overview", overview())
                .add("ScheduleExpression", scheduleExpression()).add("AssociationName", associationName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "AssociationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "AssociationVersion":
            return Optional.ofNullable(clazz.cast(associationVersion()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "LastExecutionDate":
            return Optional.ofNullable(clazz.cast(lastExecutionDate()));
        case "Overview":
            return Optional.ofNullable(clazz.cast(overview()));
        case "ScheduleExpression":
            return Optional.ofNullable(clazz.cast(scheduleExpression()));
        case "AssociationName":
            return Optional.ofNullable(clazz.cast(associationName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Association, T> g) {
        return obj -> g.apply((Association) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Association> {
        /**
         * <p>
         * The name of the Systems Manager document.
         * </p>
         * 
         * @param name
         *        The name of the Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the instance.
         * </p>
         * 
         * @param instanceId
         *        The ID of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The ID created by the system when you create an association. An association is a binding between a document
         * and a set of targets with a schedule.
         * </p>
         * 
         * @param associationId
         *        The ID created by the system when you create an association. An association is a binding between a
         *        document and a set of targets with a schedule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The association version.
         * </p>
         * 
         * @param associationVersion
         *        The association version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationVersion(String associationVersion);

        /**
         * <p>
         * The version of the document used in the association.
         * </p>
         * 
         * @param documentVersion
         *        The version of the document used in the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The instances targeted by the request to create an association.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request to create an association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The instances targeted by the request to create an association.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request to create an association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The instances targeted by the request to create an association.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The date on which the association was last run.
         * </p>
         * 
         * @param lastExecutionDate
         *        The date on which the association was last run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastExecutionDate(Instant lastExecutionDate);

        /**
         * <p>
         * Information about the association.
         * </p>
         * 
         * @param overview
         *        Information about the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overview(AssociationOverview overview);

        /**
         * <p>
         * Information about the association.
         * </p>
         * This is a convenience that creates an instance of the {@link AssociationOverview.Builder} avoiding the need
         * to create one manually via {@link AssociationOverview#builder()}.
         *
         * When the {@link Consumer} completes, {@link AssociationOverview.Builder#build()} is called immediately and
         * its result is passed to {@link #overview(AssociationOverview)}.
         * 
         * @param overview
         *        a consumer that will call methods on {@link AssociationOverview.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overview(AssociationOverview)
         */
        default Builder overview(Consumer<AssociationOverview.Builder> overview) {
            return overview(AssociationOverview.builder().applyMutation(overview).build());
        }

        /**
         * <p>
         * A cron expression that specifies a schedule when the association runs.
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression that specifies a schedule when the association runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * The association name.
         * </p>
         * 
         * @param associationName
         *        The association name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String instanceId;

        private String associationId;

        private String associationVersion;

        private String documentVersion;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private Instant lastExecutionDate;

        private AssociationOverview overview;

        private String scheduleExpression;

        private String associationName;

        private BuilderImpl() {
        }

        private BuilderImpl(Association model) {
            name(model.name);
            instanceId(model.instanceId);
            associationId(model.associationId);
            associationVersion(model.associationVersion);
            documentVersion(model.documentVersion);
            targets(model.targets);
            lastExecutionDate(model.lastExecutionDate);
            overview(model.overview);
            scheduleExpression(model.scheduleExpression);
            associationName(model.associationName);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getAssociationId() {
            return associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        public final String getAssociationVersion() {
            return associationVersion;
        }

        @Override
        public final Builder associationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
            return this;
        }

        public final void setAssociationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final Instant getLastExecutionDate() {
            return lastExecutionDate;
        }

        @Override
        public final Builder lastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
            return this;
        }

        public final void setLastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
        }

        public final AssociationOverview.Builder getOverview() {
            return overview != null ? overview.toBuilder() : null;
        }

        @Override
        public final Builder overview(AssociationOverview overview) {
            this.overview = overview;
            return this;
        }

        public final void setOverview(AssociationOverview.BuilderImpl overview) {
            this.overview = overview != null ? overview.build() : null;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final String getAssociationName() {
            return associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        @Override
        public Association build() {
            return new Association(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
