/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.paginators;

import java.util.Iterator;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.NextPageFetcher;
import software.amazon.awssdk.core.pagination.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.services.ssm.SSMClient;
import software.amazon.awssdk.services.ssm.model.CommandInvocation;
import software.amazon.awssdk.services.ssm.model.ListCommandInvocationsRequest;
import software.amazon.awssdk.services.ssm.model.ListCommandInvocationsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.ssm.SSMClient#listCommandInvocationsIterable(software.amazon.awssdk.services.ssm.model.ListCommandInvocationsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.ssm.model.ListCommandInvocationsResponse} that can be used to iterate through
 * all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ssm.paginators.ListCommandInvocationsPaginator responses = client.listCommandInvocationsIterable(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.ssm.paginators.ListCommandInvocationsPaginator responses = client
 *             .listCommandInvocationsIterable(request);
 *     for (software.amazon.awssdk.services.ssm.model.ListCommandInvocationsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ssm.paginators.ListCommandInvocationsPaginator responses = client.listCommandInvocationsIterable(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listCommandInvocations(software.amazon.awssdk.services.ssm.model.ListCommandInvocationsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListCommandInvocationsPaginator implements SdkIterable<ListCommandInvocationsResponse> {
    private final SSMClient client;

    private final ListCommandInvocationsRequest firstRequest;

    private final NextPageFetcher nextPageFetcher;

    public ListCommandInvocationsPaginator(final SSMClient client, final ListCommandInvocationsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListCommandInvocationsResponseFetcher();
    }

    @Override
    public Iterator<ListCommandInvocationsResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * Returns an iterable to iterate through the paginated {@link ListCommandInvocationsResponse#commandInvocations()}
     * member. The returned iterable is used to iterate through the results across all response pages and not a single
     * page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public SdkIterable<CommandInvocation> commandInvocations() {
        Function<ListCommandInvocationsResponse, Iterator<CommandInvocation>> getIterator = response -> response != null ? response
                .commandInvocations().iterator() : null;

        return new PaginatedItemsIterable(this, getIterator);
    }

    private class ListCommandInvocationsResponseFetcher implements NextPageFetcher<ListCommandInvocationsResponse> {
        @Override
        public boolean hasNextPage(ListCommandInvocationsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public ListCommandInvocationsResponse nextPage(ListCommandInvocationsResponse previousPage) {
            if (previousPage == null) {
                return client.listCommandInvocations(firstRequest);
            }
            return client.listCommandInvocations(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
