/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.MaintenanceWindowTaskMarshaller;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a task defined for a Maintenance Window.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class MaintenanceWindowTask implements StructuredPojo,
        ToCopyableBuilder<MaintenanceWindowTask.Builder, MaintenanceWindowTask> {
    private final String windowId;

    private final String windowTaskId;

    private final String taskArn;

    private final String type;

    private final List<Target> targets;

    private final Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters;

    private final Integer priority;

    private final LoggingInfo loggingInfo;

    private final String serviceRoleArn;

    private final String maxConcurrency;

    private final String maxErrors;

    private final String name;

    private final String description;

    private MaintenanceWindowTask(BuilderImpl builder) {
        this.windowId = builder.windowId;
        this.windowTaskId = builder.windowTaskId;
        this.taskArn = builder.taskArn;
        this.type = builder.type;
        this.targets = builder.targets;
        this.taskParameters = builder.taskParameters;
        this.priority = builder.priority;
        this.loggingInfo = builder.loggingInfo;
        this.serviceRoleArn = builder.serviceRoleArn;
        this.maxConcurrency = builder.maxConcurrency;
        this.maxErrors = builder.maxErrors;
        this.name = builder.name;
        this.description = builder.description;
    }

    /**
     * <p>
     * The Maintenance Window ID where the task is registered.
     * </p>
     * 
     * @return The Maintenance Window ID where the task is registered.
     */
    public String windowId() {
        return windowId;
    }

    /**
     * <p>
     * The task ID.
     * </p>
     * 
     * @return The task ID.
     */
    public String windowTaskId() {
        return windowTaskId;
    }

    /**
     * <p>
     * The resource that the task uses during execution. For RUN_COMMAND and AUTOMATION task types, <code>TaskArn</code>
     * is the Systems Manager document name or ARN. For LAMBDA tasks, it's the function name or ARN. For STEP_FUNCTION
     * tasks, it's the state machine ARN.
     * </p>
     * 
     * @return The resource that the task uses during execution. For RUN_COMMAND and AUTOMATION task types,
     *         <code>TaskArn</code> is the Systems Manager document name or ARN. For LAMBDA tasks, it's the function
     *         name or ARN. For STEP_FUNCTION tasks, it's the state machine ARN.
     */
    public String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or STEP_FUNCTION.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeString}.
     * </p>
     * 
     * @return The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or
     *         STEP_FUNCTION.
     * @see MaintenanceWindowTaskType
     */
    public MaintenanceWindowTaskType type() {
        return MaintenanceWindowTaskType.fromValue(type);
    }

    /**
     * <p>
     * The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or STEP_FUNCTION.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeString}.
     * </p>
     * 
     * @return The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or
     *         STEP_FUNCTION.
     * @see MaintenanceWindowTaskType
     */
    public String typeString() {
        return type;
    }

    /**
     * <p>
     * The targets (either instances or tags). Instances are specified using
     * Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
     * name&gt;,Values=&lt;tag value&gt;.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The targets (either instances or tags). Instances are specified using
     *         Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
     *         name&gt;,Values=&lt;tag value&gt;.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The parameters that should be passed to the task when it is executed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The parameters that should be passed to the task when it is executed.
     */
    public Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters() {
        return taskParameters;
    }

    /**
     * <p>
     * The priority of the task in the Maintenance Window. The lower the number, the higher the priority. Tasks that
     * have the same priority are scheduled in parallel.
     * </p>
     * 
     * @return The priority of the task in the Maintenance Window. The lower the number, the higher the priority. Tasks
     *         that have the same priority are scheduled in parallel.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * <p>
     * Information about an Amazon S3 bucket to write task-level logs to.
     * </p>
     * 
     * @return Information about an Amazon S3 bucket to write task-level logs to.
     */
    public LoggingInfo loggingInfo() {
        return loggingInfo;
    }

    /**
     * <p>
     * The role that should be assumed when executing the task
     * </p>
     * 
     * @return The role that should be assumed when executing the task
     */
    public String serviceRoleArn() {
        return serviceRoleArn;
    }

    /**
     * <p>
     * The maximum number of targets this task can be run for in parallel.
     * </p>
     * 
     * @return The maximum number of targets this task can be run for in parallel.
     */
    public String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The maximum number of errors allowed before this task stops being scheduled.
     * </p>
     * 
     * @return The maximum number of errors allowed before this task stops being scheduled.
     */
    public String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * The task name.
     * </p>
     * 
     * @return The task name.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A description of the task.
     * </p>
     * 
     * @return A description of the task.
     */
    public String description() {
        return description;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(windowId());
        hashCode = 31 * hashCode + Objects.hashCode(windowTaskId());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(typeString());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(taskParameters());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(loggingInfo());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MaintenanceWindowTask)) {
            return false;
        }
        MaintenanceWindowTask other = (MaintenanceWindowTask) obj;
        return Objects.equals(windowId(), other.windowId()) && Objects.equals(windowTaskId(), other.windowTaskId())
                && Objects.equals(taskArn(), other.taskArn()) && Objects.equals(typeString(), other.typeString())
                && Objects.equals(targets(), other.targets()) && Objects.equals(taskParameters(), other.taskParameters())
                && Objects.equals(priority(), other.priority()) && Objects.equals(loggingInfo(), other.loggingInfo())
                && Objects.equals(serviceRoleArn(), other.serviceRoleArn())
                && Objects.equals(maxConcurrency(), other.maxConcurrency()) && Objects.equals(maxErrors(), other.maxErrors())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description());
    }

    @Override
    public String toString() {
        return ToString.builder("MaintenanceWindowTask").add("WindowId", windowId()).add("WindowTaskId", windowTaskId())
                .add("TaskArn", taskArn()).add("Type", typeString()).add("Targets", targets())
                .add("TaskParameters", taskParameters()).add("Priority", priority()).add("LoggingInfo", loggingInfo())
                .add("ServiceRoleArn", serviceRoleArn()).add("MaxConcurrency", maxConcurrency()).add("MaxErrors", maxErrors())
                .add("Name", name()).add("Description", description()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WindowId":
            return Optional.of(clazz.cast(windowId()));
        case "WindowTaskId":
            return Optional.of(clazz.cast(windowTaskId()));
        case "TaskArn":
            return Optional.of(clazz.cast(taskArn()));
        case "Type":
            return Optional.of(clazz.cast(typeString()));
        case "Targets":
            return Optional.of(clazz.cast(targets()));
        case "TaskParameters":
            return Optional.of(clazz.cast(taskParameters()));
        case "Priority":
            return Optional.of(clazz.cast(priority()));
        case "LoggingInfo":
            return Optional.of(clazz.cast(loggingInfo()));
        case "ServiceRoleArn":
            return Optional.of(clazz.cast(serviceRoleArn()));
        case "MaxConcurrency":
            return Optional.of(clazz.cast(maxConcurrency()));
        case "MaxErrors":
            return Optional.of(clazz.cast(maxErrors()));
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        MaintenanceWindowTaskMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, MaintenanceWindowTask> {
        /**
         * <p>
         * The Maintenance Window ID where the task is registered.
         * </p>
         * 
         * @param windowId
         *        The Maintenance Window ID where the task is registered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowId(String windowId);

        /**
         * <p>
         * The task ID.
         * </p>
         * 
         * @param windowTaskId
         *        The task ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowTaskId(String windowTaskId);

        /**
         * <p>
         * The resource that the task uses during execution. For RUN_COMMAND and AUTOMATION task types,
         * <code>TaskArn</code> is the Systems Manager document name or ARN. For LAMBDA tasks, it's the function name or
         * ARN. For STEP_FUNCTION tasks, it's the state machine ARN.
         * </p>
         * 
         * @param taskArn
         *        The resource that the task uses during execution. For RUN_COMMAND and AUTOMATION task types,
         *        <code>TaskArn</code> is the Systems Manager document name or ARN. For LAMBDA tasks, it's the function
         *        name or ARN. For STEP_FUNCTION tasks, it's the state machine ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or STEP_FUNCTION.
         * </p>
         * 
         * @param type
         *        The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or
         *        STEP_FUNCTION.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or STEP_FUNCTION.
         * </p>
         * 
         * @param type
         *        The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or
         *        STEP_FUNCTION.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder type(MaintenanceWindowTaskType type);

        /**
         * <p>
         * The targets (either instances or tags). Instances are specified using
         * Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         * name&gt;,Values=&lt;tag value&gt;.
         * </p>
         * 
         * @param targets
         *        The targets (either instances or tags). Instances are specified using
         *        Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         *        name&gt;,Values=&lt;tag value&gt;.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets (either instances or tags). Instances are specified using
         * Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         * name&gt;,Values=&lt;tag value&gt;.
         * </p>
         * 
         * @param targets
         *        The targets (either instances or tags). Instances are specified using
         *        Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         *        name&gt;,Values=&lt;tag value&gt;.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The parameters that should be passed to the task when it is executed.
         * </p>
         * 
         * @param taskParameters
         *        The parameters that should be passed to the task when it is executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters);

        /**
         * <p>
         * The priority of the task in the Maintenance Window. The lower the number, the higher the priority. Tasks that
         * have the same priority are scheduled in parallel.
         * </p>
         * 
         * @param priority
         *        The priority of the task in the Maintenance Window. The lower the number, the higher the priority.
         *        Tasks that have the same priority are scheduled in parallel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * Information about an Amazon S3 bucket to write task-level logs to.
         * </p>
         * 
         * @param loggingInfo
         *        Information about an Amazon S3 bucket to write task-level logs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingInfo(LoggingInfo loggingInfo);

        /**
         * <p>
         * Information about an Amazon S3 bucket to write task-level logs to.
         * </p>
         * This is a convenience that creates an instance of the {@link LoggingInfo.Builder} avoiding the need to create
         * one manually via {@link LoggingInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link LoggingInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #loggingInfo(LoggingInfo)}.
         * 
         * @param loggingInfo
         *        a consumer that will call methods on {@link LoggingInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loggingInfo(LoggingInfo)
         */
        default Builder loggingInfo(Consumer<LoggingInfo.Builder> loggingInfo) {
            return loggingInfo(LoggingInfo.builder().apply(loggingInfo).build());
        }

        /**
         * <p>
         * The role that should be assumed when executing the task
         * </p>
         * 
         * @param serviceRoleArn
         *        The role that should be assumed when executing the task
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRoleArn(String serviceRoleArn);

        /**
         * <p>
         * The maximum number of targets this task can be run for in parallel.
         * </p>
         * 
         * @param maxConcurrency
         *        The maximum number of targets this task can be run for in parallel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The maximum number of errors allowed before this task stops being scheduled.
         * </p>
         * 
         * @param maxErrors
         *        The maximum number of errors allowed before this task stops being scheduled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * The task name.
         * </p>
         * 
         * @param name
         *        The task name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the task.
         * </p>
         * 
         * @param description
         *        A description of the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);
    }

    static final class BuilderImpl implements Builder {
        private String windowId;

        private String windowTaskId;

        private String taskArn;

        private String type;

        private List<Target> targets;

        private Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters;

        private Integer priority;

        private LoggingInfo loggingInfo;

        private String serviceRoleArn;

        private String maxConcurrency;

        private String maxErrors;

        private String name;

        private String description;

        private BuilderImpl() {
        }

        private BuilderImpl(MaintenanceWindowTask model) {
            windowId(model.windowId);
            windowTaskId(model.windowTaskId);
            taskArn(model.taskArn);
            type(model.type);
            targets(model.targets);
            taskParameters(model.taskParameters);
            priority(model.priority);
            loggingInfo(model.loggingInfo);
            serviceRoleArn(model.serviceRoleArn);
            maxConcurrency(model.maxConcurrency);
            maxErrors(model.maxErrors);
            name(model.name);
            description(model.description);
        }

        public final String getWindowId() {
            return windowId;
        }

        @Override
        public final Builder windowId(String windowId) {
            this.windowId = windowId;
            return this;
        }

        public final void setWindowId(String windowId) {
            this.windowId = windowId;
        }

        public final String getWindowTaskId() {
            return windowTaskId;
        }

        @Override
        public final Builder windowTaskId(String windowTaskId) {
            this.windowTaskId = windowTaskId;
            return this;
        }

        public final void setWindowTaskId(String windowTaskId) {
            this.windowTaskId = windowTaskId;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(MaintenanceWindowTaskType type) {
            this.type(type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final Map<String, MaintenanceWindowTaskParameterValueExpression.Builder> getTaskParameters() {
            return taskParameters != null ? CollectionUtils.mapValues(taskParameters,
                    MaintenanceWindowTaskParameterValueExpression::toBuilder) : null;
        }

        @Override
        public final Builder taskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters) {
            this.taskParameters = MaintenanceWindowTaskParametersCopier.copy(taskParameters);
            return this;
        }

        public final void setTaskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression.BuilderImpl> taskParameters) {
            this.taskParameters = MaintenanceWindowTaskParametersCopier.copyFromBuilder(taskParameters);
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final LoggingInfo.Builder getLoggingInfo() {
            return loggingInfo != null ? loggingInfo.toBuilder() : null;
        }

        @Override
        public final Builder loggingInfo(LoggingInfo loggingInfo) {
            this.loggingInfo = loggingInfo;
            return this;
        }

        public final void setLoggingInfo(LoggingInfo.BuilderImpl loggingInfo) {
            this.loggingInfo = loggingInfo != null ? loggingInfo.build() : null;
        }

        public final String getServiceRoleArn() {
            return serviceRoleArn;
        }

        @Override
        public final Builder serviceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
            return this;
        }

        public final void setServiceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        @Override
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public MaintenanceWindowTask build() {
            return new MaintenanceWindowTask(this);
        }
    }
}
