/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.MaintenanceWindowExecutionTaskInvocationIdentityMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the information about a task invocation for a particular target as part of a task execution performed as
 * part of a Maintenance Window execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class MaintenanceWindowExecutionTaskInvocationIdentity
        implements
        StructuredPojo,
        ToCopyableBuilder<MaintenanceWindowExecutionTaskInvocationIdentity.Builder, MaintenanceWindowExecutionTaskInvocationIdentity> {
    private final String windowExecutionId;

    private final String taskExecutionId;

    private final String invocationId;

    private final String executionId;

    private final String taskType;

    private final String parameters;

    private final String status;

    private final String statusDetails;

    private final Instant startTime;

    private final Instant endTime;

    private final String ownerInformation;

    private final String windowTargetId;

    private MaintenanceWindowExecutionTaskInvocationIdentity(BuilderImpl builder) {
        this.windowExecutionId = builder.windowExecutionId;
        this.taskExecutionId = builder.taskExecutionId;
        this.invocationId = builder.invocationId;
        this.executionId = builder.executionId;
        this.taskType = builder.taskType;
        this.parameters = builder.parameters;
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.ownerInformation = builder.ownerInformation;
        this.windowTargetId = builder.windowTargetId;
    }

    /**
     * <p>
     * The ID of the Maintenance Window execution that ran the task.
     * </p>
     * 
     * @return The ID of the Maintenance Window execution that ran the task.
     */
    public String windowExecutionId() {
        return windowExecutionId;
    }

    /**
     * <p>
     * The ID of the specific task execution in the Maintenance Window execution.
     * </p>
     * 
     * @return The ID of the specific task execution in the Maintenance Window execution.
     */
    public String taskExecutionId() {
        return taskExecutionId;
    }

    /**
     * <p>
     * The ID of the task invocation.
     * </p>
     * 
     * @return The ID of the task invocation.
     */
    public String invocationId() {
        return invocationId;
    }

    /**
     * <p>
     * The ID of the action performed in the service that actually handled the task invocation. If the task type is
     * RUN_COMMAND, this value is the command ID.
     * </p>
     * 
     * @return The ID of the action performed in the service that actually handled the task invocation. If the task type
     *         is RUN_COMMAND, this value is the command ID.
     */
    public String executionId() {
        return executionId;
    }

    /**
     * <p>
     * The task type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskTypeString}.
     * </p>
     * 
     * @return The task type.
     * @see MaintenanceWindowTaskType
     */
    public MaintenanceWindowTaskType taskType() {
        return MaintenanceWindowTaskType.fromValue(taskType);
    }

    /**
     * <p>
     * The task type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskTypeString}.
     * </p>
     * 
     * @return The task type.
     * @see MaintenanceWindowTaskType
     */
    public String taskTypeString() {
        return taskType;
    }

    /**
     * <p>
     * The parameters that were provided for the invocation when it was executed.
     * </p>
     * 
     * @return The parameters that were provided for the invocation when it was executed.
     */
    public String parameters() {
        return parameters;
    }

    /**
     * <p>
     * The status of the task invocation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MaintenanceWindowExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusString}.
     * </p>
     * 
     * @return The status of the task invocation.
     * @see MaintenanceWindowExecutionStatus
     */
    public MaintenanceWindowExecutionStatus status() {
        return MaintenanceWindowExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the task invocation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MaintenanceWindowExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusString}.
     * </p>
     * 
     * @return The status of the task invocation.
     * @see MaintenanceWindowExecutionStatus
     */
    public String statusString() {
        return status;
    }

    /**
     * <p>
     * The details explaining the status of the task invocation. Only available for certain Status values.
     * </p>
     * 
     * @return The details explaining the status of the task invocation. Only available for certain Status values.
     */
    public String statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * The time the invocation started.
     * </p>
     * 
     * @return The time the invocation started.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time the invocation finished.
     * </p>
     * 
     * @return The time the invocation finished.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * User-provided value that was specified when the target was registered with the Maintenance Window. This was also
     * included in any CloudWatch events raised during the task invocation.
     * </p>
     * 
     * @return User-provided value that was specified when the target was registered with the Maintenance Window. This
     *         was also included in any CloudWatch events raised during the task invocation.
     */
    public String ownerInformation() {
        return ownerInformation;
    }

    /**
     * <p>
     * The ID of the target definition in this Maintenance Window the invocation was performed for.
     * </p>
     * 
     * @return The ID of the target definition in this Maintenance Window the invocation was performed for.
     */
    public String windowTargetId() {
        return windowTargetId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(windowExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(taskExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(invocationId());
        hashCode = 31 * hashCode + Objects.hashCode(executionId());
        hashCode = 31 * hashCode + Objects.hashCode(taskTypeString());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(statusString());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(ownerInformation());
        hashCode = 31 * hashCode + Objects.hashCode(windowTargetId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MaintenanceWindowExecutionTaskInvocationIdentity)) {
            return false;
        }
        MaintenanceWindowExecutionTaskInvocationIdentity other = (MaintenanceWindowExecutionTaskInvocationIdentity) obj;
        return Objects.equals(windowExecutionId(), other.windowExecutionId())
                && Objects.equals(taskExecutionId(), other.taskExecutionId())
                && Objects.equals(invocationId(), other.invocationId()) && Objects.equals(executionId(), other.executionId())
                && Objects.equals(taskTypeString(), other.taskTypeString()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(statusString(), other.statusString()) && Objects.equals(statusDetails(), other.statusDetails())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(ownerInformation(), other.ownerInformation())
                && Objects.equals(windowTargetId(), other.windowTargetId());
    }

    @Override
    public String toString() {
        return ToString.builder("MaintenanceWindowExecutionTaskInvocationIdentity").add("WindowExecutionId", windowExecutionId())
                .add("TaskExecutionId", taskExecutionId()).add("InvocationId", invocationId()).add("ExecutionId", executionId())
                .add("TaskType", taskTypeString()).add("Parameters", parameters()).add("Status", statusString())
                .add("StatusDetails", statusDetails()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("OwnerInformation", ownerInformation()).add("WindowTargetId", windowTargetId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WindowExecutionId":
            return Optional.of(clazz.cast(windowExecutionId()));
        case "TaskExecutionId":
            return Optional.of(clazz.cast(taskExecutionId()));
        case "InvocationId":
            return Optional.of(clazz.cast(invocationId()));
        case "ExecutionId":
            return Optional.of(clazz.cast(executionId()));
        case "TaskType":
            return Optional.of(clazz.cast(taskTypeString()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "Status":
            return Optional.of(clazz.cast(statusString()));
        case "StatusDetails":
            return Optional.of(clazz.cast(statusDetails()));
        case "StartTime":
            return Optional.of(clazz.cast(startTime()));
        case "EndTime":
            return Optional.of(clazz.cast(endTime()));
        case "OwnerInformation":
            return Optional.of(clazz.cast(ownerInformation()));
        case "WindowTargetId":
            return Optional.of(clazz.cast(windowTargetId()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        MaintenanceWindowExecutionTaskInvocationIdentityMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, MaintenanceWindowExecutionTaskInvocationIdentity> {
        /**
         * <p>
         * The ID of the Maintenance Window execution that ran the task.
         * </p>
         * 
         * @param windowExecutionId
         *        The ID of the Maintenance Window execution that ran the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowExecutionId(String windowExecutionId);

        /**
         * <p>
         * The ID of the specific task execution in the Maintenance Window execution.
         * </p>
         * 
         * @param taskExecutionId
         *        The ID of the specific task execution in the Maintenance Window execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskExecutionId(String taskExecutionId);

        /**
         * <p>
         * The ID of the task invocation.
         * </p>
         * 
         * @param invocationId
         *        The ID of the task invocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invocationId(String invocationId);

        /**
         * <p>
         * The ID of the action performed in the service that actually handled the task invocation. If the task type is
         * RUN_COMMAND, this value is the command ID.
         * </p>
         * 
         * @param executionId
         *        The ID of the action performed in the service that actually handled the task invocation. If the task
         *        type is RUN_COMMAND, this value is the command ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionId(String executionId);

        /**
         * <p>
         * The task type.
         * </p>
         * 
         * @param taskType
         *        The task type.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder taskType(String taskType);

        /**
         * <p>
         * The task type.
         * </p>
         * 
         * @param taskType
         *        The task type.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder taskType(MaintenanceWindowTaskType taskType);

        /**
         * <p>
         * The parameters that were provided for the invocation when it was executed.
         * </p>
         * 
         * @param parameters
         *        The parameters that were provided for the invocation when it was executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(String parameters);

        /**
         * <p>
         * The status of the task invocation.
         * </p>
         * 
         * @param status
         *        The status of the task invocation.
         * @see MaintenanceWindowExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the task invocation.
         * </p>
         * 
         * @param status
         *        The status of the task invocation.
         * @see MaintenanceWindowExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowExecutionStatus
         */
        Builder status(MaintenanceWindowExecutionStatus status);

        /**
         * <p>
         * The details explaining the status of the task invocation. Only available for certain Status values.
         * </p>
         * 
         * @param statusDetails
         *        The details explaining the status of the task invocation. Only available for certain Status values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(String statusDetails);

        /**
         * <p>
         * The time the invocation started.
         * </p>
         * 
         * @param startTime
         *        The time the invocation started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The time the invocation finished.
         * </p>
         * 
         * @param endTime
         *        The time the invocation finished.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * User-provided value that was specified when the target was registered with the Maintenance Window. This was
         * also included in any CloudWatch events raised during the task invocation.
         * </p>
         * 
         * @param ownerInformation
         *        User-provided value that was specified when the target was registered with the Maintenance Window.
         *        This was also included in any CloudWatch events raised during the task invocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerInformation(String ownerInformation);

        /**
         * <p>
         * The ID of the target definition in this Maintenance Window the invocation was performed for.
         * </p>
         * 
         * @param windowTargetId
         *        The ID of the target definition in this Maintenance Window the invocation was performed for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowTargetId(String windowTargetId);
    }

    static final class BuilderImpl implements Builder {
        private String windowExecutionId;

        private String taskExecutionId;

        private String invocationId;

        private String executionId;

        private String taskType;

        private String parameters;

        private String status;

        private String statusDetails;

        private Instant startTime;

        private Instant endTime;

        private String ownerInformation;

        private String windowTargetId;

        private BuilderImpl() {
        }

        private BuilderImpl(MaintenanceWindowExecutionTaskInvocationIdentity model) {
            windowExecutionId(model.windowExecutionId);
            taskExecutionId(model.taskExecutionId);
            invocationId(model.invocationId);
            executionId(model.executionId);
            taskType(model.taskType);
            parameters(model.parameters);
            status(model.status);
            statusDetails(model.statusDetails);
            startTime(model.startTime);
            endTime(model.endTime);
            ownerInformation(model.ownerInformation);
            windowTargetId(model.windowTargetId);
        }

        public final String getWindowExecutionId() {
            return windowExecutionId;
        }

        @Override
        public final Builder windowExecutionId(String windowExecutionId) {
            this.windowExecutionId = windowExecutionId;
            return this;
        }

        public final void setWindowExecutionId(String windowExecutionId) {
            this.windowExecutionId = windowExecutionId;
        }

        public final String getTaskExecutionId() {
            return taskExecutionId;
        }

        @Override
        public final Builder taskExecutionId(String taskExecutionId) {
            this.taskExecutionId = taskExecutionId;
            return this;
        }

        public final void setTaskExecutionId(String taskExecutionId) {
            this.taskExecutionId = taskExecutionId;
        }

        public final String getInvocationId() {
            return invocationId;
        }

        @Override
        public final Builder invocationId(String invocationId) {
            this.invocationId = invocationId;
            return this;
        }

        public final void setInvocationId(String invocationId) {
            this.invocationId = invocationId;
        }

        public final String getExecutionId() {
            return executionId;
        }

        @Override
        public final Builder executionId(String executionId) {
            this.executionId = executionId;
            return this;
        }

        public final void setExecutionId(String executionId) {
            this.executionId = executionId;
        }

        public final String getTaskType() {
            return taskType;
        }

        @Override
        public final Builder taskType(String taskType) {
            this.taskType = taskType;
            return this;
        }

        @Override
        public final Builder taskType(MaintenanceWindowTaskType taskType) {
            this.taskType(taskType.toString());
            return this;
        }

        public final void setTaskType(String taskType) {
            this.taskType = taskType;
        }

        public final String getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(String parameters) {
            this.parameters = parameters;
            return this;
        }

        public final void setParameters(String parameters) {
            this.parameters = parameters;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MaintenanceWindowExecutionStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusDetails() {
            return statusDetails;
        }

        @Override
        public final Builder statusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final void setStatusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getOwnerInformation() {
            return ownerInformation;
        }

        @Override
        public final Builder ownerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
            return this;
        }

        public final void setOwnerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
        }

        public final String getWindowTargetId() {
            return windowTargetId;
        }

        @Override
        public final Builder windowTargetId(String windowTargetId) {
            this.windowTargetId = windowTargetId;
            return this;
        }

        public final void setWindowTargetId(String windowTargetId) {
            this.windowTargetId = windowTargetId;
        }

        @Override
        public MaintenanceWindowExecutionTaskInvocationIdentity build() {
            return new MaintenanceWindowExecutionTaskInvocationIdentity(this);
        }
    }
}
