/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.InventoryResultItemMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The inventory result item.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class InventoryResultItem implements StructuredPojo, ToCopyableBuilder<InventoryResultItem.Builder, InventoryResultItem> {
    private final String typeName;

    private final String schemaVersion;

    private final String captureTime;

    private final String contentHash;

    private final List<Map<String, String>> content;

    private InventoryResultItem(BuilderImpl builder) {
        this.typeName = builder.typeName;
        this.schemaVersion = builder.schemaVersion;
        this.captureTime = builder.captureTime;
        this.contentHash = builder.contentHash;
        this.content = builder.content;
    }

    /**
     * <p>
     * The name of the inventory result item type.
     * </p>
     * 
     * @return The name of the inventory result item type.
     */
    public String typeName() {
        return typeName;
    }

    /**
     * <p>
     * The schema version for the inventory result item/
     * </p>
     * 
     * @return The schema version for the inventory result item/
     */
    public String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The time inventory item data was captured.
     * </p>
     * 
     * @return The time inventory item data was captured.
     */
    public String captureTime() {
        return captureTime;
    }

    /**
     * <p>
     * MD5 hash of the inventory item type contents. The content hash is used to determine whether to update inventory
     * information. The PutInventory API does not update the inventory item type contents if the MD5 hash has not
     * changed since last update.
     * </p>
     * 
     * @return MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
     *         inventory information. The PutInventory API does not update the inventory item type contents if the MD5
     *         hash has not changed since last update.
     */
    public String contentHash() {
        return contentHash;
    }

    /**
     * <p>
     * Contains all the inventory data of the item type. Results include attribute names and values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Contains all the inventory data of the item type. Results include attribute names and values.
     */
    public List<Map<String, String>> content() {
        return content;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeName());
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(captureTime());
        hashCode = 31 * hashCode + Objects.hashCode(contentHash());
        hashCode = 31 * hashCode + Objects.hashCode(content());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InventoryResultItem)) {
            return false;
        }
        InventoryResultItem other = (InventoryResultItem) obj;
        return Objects.equals(typeName(), other.typeName()) && Objects.equals(schemaVersion(), other.schemaVersion())
                && Objects.equals(captureTime(), other.captureTime()) && Objects.equals(contentHash(), other.contentHash())
                && Objects.equals(content(), other.content());
    }

    @Override
    public String toString() {
        return ToString.builder("InventoryResultItem").add("TypeName", typeName()).add("SchemaVersion", schemaVersion())
                .add("CaptureTime", captureTime()).add("ContentHash", contentHash()).add("Content", content()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TypeName":
            return Optional.of(clazz.cast(typeName()));
        case "SchemaVersion":
            return Optional.of(clazz.cast(schemaVersion()));
        case "CaptureTime":
            return Optional.of(clazz.cast(captureTime()));
        case "ContentHash":
            return Optional.of(clazz.cast(contentHash()));
        case "Content":
            return Optional.of(clazz.cast(content()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        InventoryResultItemMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, InventoryResultItem> {
        /**
         * <p>
         * The name of the inventory result item type.
         * </p>
         * 
         * @param typeName
         *        The name of the inventory result item type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typeName(String typeName);

        /**
         * <p>
         * The schema version for the inventory result item/
         * </p>
         * 
         * @param schemaVersion
         *        The schema version for the inventory result item/
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The time inventory item data was captured.
         * </p>
         * 
         * @param captureTime
         *        The time inventory item data was captured.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureTime(String captureTime);

        /**
         * <p>
         * MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
         * inventory information. The PutInventory API does not update the inventory item type contents if the MD5 hash
         * has not changed since last update.
         * </p>
         * 
         * @param contentHash
         *        MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
         *        inventory information. The PutInventory API does not update the inventory item type contents if the
         *        MD5 hash has not changed since last update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentHash(String contentHash);

        /**
         * <p>
         * Contains all the inventory data of the item type. Results include attribute names and values.
         * </p>
         * 
         * @param content
         *        Contains all the inventory data of the item type. Results include attribute names and values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(Collection<Map<String, String>> content);

        /**
         * <p>
         * Contains all the inventory data of the item type. Results include attribute names and values.
         * </p>
         * 
         * @param content
         *        Contains all the inventory data of the item type. Results include attribute names and values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(Map<String, String>... content);
    }

    static final class BuilderImpl implements Builder {
        private String typeName;

        private String schemaVersion;

        private String captureTime;

        private String contentHash;

        private List<Map<String, String>> content;

        private BuilderImpl() {
        }

        private BuilderImpl(InventoryResultItem model) {
            typeName(model.typeName);
            schemaVersion(model.schemaVersion);
            captureTime(model.captureTime);
            contentHash(model.contentHash);
            content(model.content);
        }

        public final String getTypeName() {
            return typeName;
        }

        @Override
        public final Builder typeName(String typeName) {
            this.typeName = typeName;
            return this;
        }

        public final void setTypeName(String typeName) {
            this.typeName = typeName;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        public final String getCaptureTime() {
            return captureTime;
        }

        @Override
        public final Builder captureTime(String captureTime) {
            this.captureTime = captureTime;
            return this;
        }

        public final void setCaptureTime(String captureTime) {
            this.captureTime = captureTime;
        }

        public final String getContentHash() {
            return contentHash;
        }

        @Override
        public final Builder contentHash(String contentHash) {
            this.contentHash = contentHash;
            return this;
        }

        public final void setContentHash(String contentHash) {
            this.contentHash = contentHash;
        }

        public final Collection<Map<String, String>> getContent() {
            return content;
        }

        @Override
        public final Builder content(Collection<Map<String, String>> content) {
            this.content = InventoryItemEntryListCopier.copy(content);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder content(Map<String, String>... content) {
            content(Arrays.asList(content));
            return this;
        }

        public final void setContent(Collection<Map<String, String>> content) {
            this.content = InventoryItemEntryListCopier.copy(content);
        }

        @Override
        public InventoryResultItem build() {
            return new InventoryResultItem(this);
        }
    }
}
