/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.InventoryItemSchemaMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The inventory item schema definition. Users can use this to compose inventory query filters.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class InventoryItemSchema implements StructuredPojo, ToCopyableBuilder<InventoryItemSchema.Builder, InventoryItemSchema> {
    private final String typeName;

    private final String version;

    private final List<InventoryItemAttribute> attributes;

    private final String displayName;

    private InventoryItemSchema(BuilderImpl builder) {
        this.typeName = builder.typeName;
        this.version = builder.version;
        this.attributes = builder.attributes;
        this.displayName = builder.displayName;
    }

    /**
     * <p>
     * The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type names
     * will start with Custom. Default inventory item types include the following: AWS:AWSComponent, AWS:Application,
     * AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
     * </p>
     * 
     * @return The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type
     *         names will start with Custom. Default inventory item types include the following: AWS:AWSComponent,
     *         AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
     */
    public String typeName() {
        return typeName;
    }

    /**
     * <p>
     * The schema version for the inventory item.
     * </p>
     * 
     * @return The schema version for the inventory item.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * The schema attributes for inventory. This contains data type and attribute name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The schema attributes for inventory. This contains data type and attribute name.
     */
    public List<InventoryItemAttribute> attributes() {
        return attributes;
    }

    /**
     * <p>
     * The alias name of the inventory type. The alias name is used for display purposes.
     * </p>
     * 
     * @return The alias name of the inventory type. The alias name is used for display purposes.
     */
    public String displayName() {
        return displayName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeName());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(attributes());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InventoryItemSchema)) {
            return false;
        }
        InventoryItemSchema other = (InventoryItemSchema) obj;
        return Objects.equals(typeName(), other.typeName()) && Objects.equals(version(), other.version())
                && Objects.equals(attributes(), other.attributes()) && Objects.equals(displayName(), other.displayName());
    }

    @Override
    public String toString() {
        return ToString.builder("InventoryItemSchema").add("TypeName", typeName()).add("Version", version())
                .add("Attributes", attributes()).add("DisplayName", displayName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TypeName":
            return Optional.of(clazz.cast(typeName()));
        case "Version":
            return Optional.of(clazz.cast(version()));
        case "Attributes":
            return Optional.of(clazz.cast(attributes()));
        case "DisplayName":
            return Optional.of(clazz.cast(displayName()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        InventoryItemSchemaMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, InventoryItemSchema> {
        /**
         * <p>
         * The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type names
         * will start with Custom. Default inventory item types include the following: AWS:AWSComponent,
         * AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
         * </p>
         * 
         * @param typeName
         *        The name of the inventory type. Default inventory item type names start with AWS. Custom inventory
         *        type names will start with Custom. Default inventory item types include the following:
         *        AWS:AWSComponent, AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typeName(String typeName);

        /**
         * <p>
         * The schema version for the inventory item.
         * </p>
         * 
         * @param version
         *        The schema version for the inventory item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The schema attributes for inventory. This contains data type and attribute name.
         * </p>
         * 
         * @param attributes
         *        The schema attributes for inventory. This contains data type and attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Collection<InventoryItemAttribute> attributes);

        /**
         * <p>
         * The schema attributes for inventory. This contains data type and attribute name.
         * </p>
         * 
         * @param attributes
         *        The schema attributes for inventory. This contains data type and attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(InventoryItemAttribute... attributes);

        /**
         * <p>
         * The alias name of the inventory type. The alias name is used for display purposes.
         * </p>
         * 
         * @param displayName
         *        The alias name of the inventory type. The alias name is used for display purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);
    }

    static final class BuilderImpl implements Builder {
        private String typeName;

        private String version;

        private List<InventoryItemAttribute> attributes;

        private String displayName;

        private BuilderImpl() {
        }

        private BuilderImpl(InventoryItemSchema model) {
            typeName(model.typeName);
            version(model.version);
            attributes(model.attributes);
            displayName(model.displayName);
        }

        public final String getTypeName() {
            return typeName;
        }

        @Override
        public final Builder typeName(String typeName) {
            this.typeName = typeName;
            return this;
        }

        public final void setTypeName(String typeName) {
            this.typeName = typeName;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final Collection<InventoryItemAttribute.Builder> getAttributes() {
            return attributes != null ? attributes.stream().map(InventoryItemAttribute::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder attributes(Collection<InventoryItemAttribute> attributes) {
            this.attributes = InventoryItemAttributeListCopier.copy(attributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(InventoryItemAttribute... attributes) {
            attributes(Arrays.asList(attributes));
            return this;
        }

        public final void setAttributes(Collection<InventoryItemAttribute.BuilderImpl> attributes) {
            this.attributes = InventoryItemAttributeListCopier.copyFromBuilder(attributes);
        }

        public final String getDisplayName() {
            return displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public InventoryItemSchema build() {
            return new InventoryItemSchema(this);
        }
    }
}
