/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.services.ssm.transform.DocumentIdentifierMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the name of a Systems Manager document.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DocumentIdentifier implements StructuredPojo, ToCopyableBuilder<DocumentIdentifier.Builder, DocumentIdentifier> {
    private final String name;

    private final String owner;

    private final List<String> platformTypes;

    private final String documentVersion;

    private final String documentType;

    private final String schemaVersion;

    private final List<Tag> tags;

    private DocumentIdentifier(BuilderImpl builder) {
        this.name = builder.name;
        this.owner = builder.owner;
        this.platformTypes = builder.platformTypes;
        this.documentVersion = builder.documentVersion;
        this.documentType = builder.documentType;
        this.schemaVersion = builder.schemaVersion;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the Systems Manager document.
     * </p>
     * 
     * @return The name of the Systems Manager document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The AWS user account that created the document.
     * </p>
     * 
     * @return The AWS user account that created the document.
     */
    public String owner() {
        return owner;
    }

    /**
     * <p>
     * The operating system platform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The operating system platform.
     */
    public List<PlatformType> platformTypes() {
        return TypeConverter.convert(platformTypes, PlatformType::fromValue);
    }

    /**
     * <p>
     * The operating system platform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The operating system platform.
     */
    public List<String> platformTypesStrings() {
        return platformTypes;
    }

    /**
     * <p>
     * The document version.
     * </p>
     * 
     * @return The document version.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The document type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeString}.
     * </p>
     * 
     * @return The document type.
     * @see DocumentType
     */
    public DocumentType documentType() {
        return DocumentType.fromValue(documentType);
    }

    /**
     * <p>
     * The document type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeString}.
     * </p>
     * 
     * @return The document type.
     * @see DocumentType
     */
    public String documentTypeString() {
        return documentType;
    }

    /**
     * <p>
     * The schema version.
     * </p>
     * 
     * @return The schema version.
     */
    public String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The tags, or metadata, that have been applied to the document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The tags, or metadata, that have been applied to the document.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(platformTypesStrings());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentTypeString());
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentIdentifier)) {
            return false;
        }
        DocumentIdentifier other = (DocumentIdentifier) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(owner(), other.owner())
                && Objects.equals(platformTypesStrings(), other.platformTypesStrings())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(documentTypeString(), other.documentTypeString())
                && Objects.equals(schemaVersion(), other.schemaVersion()) && Objects.equals(tags(), other.tags());
    }

    @Override
    public String toString() {
        return ToString.builder("DocumentIdentifier").add("Name", name()).add("Owner", owner())
                .add("PlatformTypes", platformTypesStrings()).add("DocumentVersion", documentVersion())
                .add("DocumentType", documentTypeString()).add("SchemaVersion", schemaVersion()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "Owner":
            return Optional.of(clazz.cast(owner()));
        case "PlatformTypes":
            return Optional.of(clazz.cast(platformTypesStrings()));
        case "DocumentVersion":
            return Optional.of(clazz.cast(documentVersion()));
        case "DocumentType":
            return Optional.of(clazz.cast(documentTypeString()));
        case "SchemaVersion":
            return Optional.of(clazz.cast(schemaVersion()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DocumentIdentifierMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, DocumentIdentifier> {
        /**
         * <p>
         * The name of the Systems Manager document.
         * </p>
         * 
         * @param name
         *        The name of the Systems Manager document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The AWS user account that created the document.
         * </p>
         * 
         * @param owner
         *        The AWS user account that created the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(Collection<String> platformTypes);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(String... platformTypes);

        /**
         * <p>
         * The document version.
         * </p>
         * 
         * @param documentVersion
         *        The document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The document type.
         * </p>
         * 
         * @param documentType
         *        The document type.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(String documentType);

        /**
         * <p>
         * The document type.
         * </p>
         * 
         * @param documentType
         *        The document type.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(DocumentType documentType);

        /**
         * <p>
         * The schema version.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String owner;

        private List<String> platformTypes;

        private String documentVersion;

        private String documentType;

        private String schemaVersion;

        private List<Tag> tags;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentIdentifier model) {
            name(model.name);
            owner(model.owner);
            platformTypes(model.platformTypes);
            documentVersion(model.documentVersion);
            documentType(model.documentType);
            schemaVersion(model.schemaVersion);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getOwner() {
            return owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        public final Collection<String> getPlatformTypes() {
            return platformTypes;
        }

        @Override
        public final Builder platformTypes(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypes(String... platformTypes) {
            platformTypes(Arrays.asList(platformTypes));
            return this;
        }

        public final void setPlatformTypes(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final String getDocumentType() {
            return documentType;
        }

        @Override
        public final Builder documentType(String documentType) {
            this.documentType = documentType;
            return this;
        }

        @Override
        public final Builder documentType(DocumentType documentType) {
            this.documentType(documentType.toString());
            return this;
        }

        public final void setDocumentType(String documentType) {
            this.documentType = documentType;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public DocumentIdentifier build() {
            return new DocumentIdentifier(this);
        }
    }
}
