/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.CreateAssociationBatchRequestEntryMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the association of a Systems Manager document and an instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class CreateAssociationBatchRequestEntry implements StructuredPojo,
        ToCopyableBuilder<CreateAssociationBatchRequestEntry.Builder, CreateAssociationBatchRequestEntry> {
    private final String name;

    private final String instanceId;

    private final Map<String, List<String>> parameters;

    private final String documentVersion;

    private final List<Target> targets;

    private final String scheduleExpression;

    private final InstanceAssociationOutputLocation outputLocation;

    private final String associationName;

    private CreateAssociationBatchRequestEntry(BuilderImpl builder) {
        this.name = builder.name;
        this.instanceId = builder.instanceId;
        this.parameters = builder.parameters;
        this.documentVersion = builder.documentVersion;
        this.targets = builder.targets;
        this.scheduleExpression = builder.scheduleExpression;
        this.outputLocation = builder.outputLocation;
        this.associationName = builder.associationName;
    }

    /**
     * <p>
     * The name of the configuration document.
     * </p>
     * 
     * @return The name of the configuration document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the instance.
     * </p>
     * 
     * @return The ID of the instance.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * A description of the parameters for a document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A description of the parameters for a document.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The document version.
     * </p>
     * 
     * @return The document version.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The instances targeted by the request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The instances targeted by the request.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * A cron expression that specifies a schedule when the association runs.
     * </p>
     * 
     * @return A cron expression that specifies a schedule when the association runs.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * An Amazon S3 bucket where you want to store the results of this request.
     * </p>
     * 
     * @return An Amazon S3 bucket where you want to store the results of this request.
     */
    public InstanceAssociationOutputLocation outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * Specify a descriptive name for the association.
     * </p>
     * 
     * @return Specify a descriptive name for the association.
     */
    public String associationName() {
        return associationName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAssociationBatchRequestEntry)) {
            return false;
        }
        CreateAssociationBatchRequestEntry other = (CreateAssociationBatchRequestEntry) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(targets(), other.targets()) && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(associationName(), other.associationName());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateAssociationBatchRequestEntry").add("Name", name()).add("InstanceId", instanceId())
                .add("Parameters", parameters()).add("DocumentVersion", documentVersion()).add("Targets", targets())
                .add("ScheduleExpression", scheduleExpression()).add("OutputLocation", outputLocation())
                .add("AssociationName", associationName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "InstanceId":
            return Optional.of(clazz.cast(instanceId()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "DocumentVersion":
            return Optional.of(clazz.cast(documentVersion()));
        case "Targets":
            return Optional.of(clazz.cast(targets()));
        case "ScheduleExpression":
            return Optional.of(clazz.cast(scheduleExpression()));
        case "OutputLocation":
            return Optional.of(clazz.cast(outputLocation()));
        case "AssociationName":
            return Optional.of(clazz.cast(associationName()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        CreateAssociationBatchRequestEntryMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, CreateAssociationBatchRequestEntry> {
        /**
         * <p>
         * The name of the configuration document.
         * </p>
         * 
         * @param name
         *        The name of the configuration document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the instance.
         * </p>
         * 
         * @param instanceId
         *        The ID of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * 
         * @param parameters
         *        A description of the parameters for a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The document version.
         * </p>
         * 
         * @param documentVersion
         *        The document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The instances targeted by the request.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The instances targeted by the request.
         * </p>
         * 
         * @param targets
         *        The instances targeted by the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * A cron expression that specifies a schedule when the association runs.
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression that specifies a schedule when the association runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the results of this request.
         * </p>
         * 
         * @param outputLocation
         *        An Amazon S3 bucket where you want to store the results of this request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(InstanceAssociationOutputLocation outputLocation);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the results of this request.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceAssociationOutputLocation.Builder}
         * avoiding the need to create one manually via {@link InstanceAssociationOutputLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceAssociationOutputLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #outputLocation(InstanceAssociationOutputLocation)}.
         * 
         * @param outputLocation
         *        a consumer that will call methods on {@link InstanceAssociationOutputLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputLocation(InstanceAssociationOutputLocation)
         */
        default Builder outputLocation(Consumer<InstanceAssociationOutputLocation.Builder> outputLocation) {
            return outputLocation(InstanceAssociationOutputLocation.builder().apply(outputLocation).build());
        }

        /**
         * <p>
         * Specify a descriptive name for the association.
         * </p>
         * 
         * @param associationName
         *        Specify a descriptive name for the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String instanceId;

        private Map<String, List<String>> parameters;

        private String documentVersion;

        private List<Target> targets;

        private String scheduleExpression;

        private InstanceAssociationOutputLocation outputLocation;

        private String associationName;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAssociationBatchRequestEntry model) {
            name(model.name);
            instanceId(model.instanceId);
            parameters(model.parameters);
            documentVersion(model.documentVersion);
            targets(model.targets);
            scheduleExpression(model.scheduleExpression);
            outputLocation(model.outputLocation);
            associationName(model.associationName);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final InstanceAssociationOutputLocation.Builder getOutputLocation() {
            return outputLocation != null ? outputLocation.toBuilder() : null;
        }

        @Override
        public final Builder outputLocation(InstanceAssociationOutputLocation outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final void setOutputLocation(InstanceAssociationOutputLocation.BuilderImpl outputLocation) {
            this.outputLocation = outputLocation != null ? outputLocation.build() : null;
        }

        public final String getAssociationName() {
            return associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        @Override
        public CreateAssociationBatchRequestEntry build() {
            return new CreateAssociationBatchRequestEntry(this);
        }
    }
}
