/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.CommandMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a command request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Command implements StructuredPojo, ToCopyableBuilder<Command.Builder, Command> {
    private final String commandId;

    private final String documentName;

    private final String comment;

    private final Instant expiresAfter;

    private final Map<String, List<String>> parameters;

    private final List<String> instanceIds;

    private final List<Target> targets;

    private final Instant requestedDateTime;

    private final String status;

    private final String statusDetails;

    private final String outputS3Region;

    private final String outputS3BucketName;

    private final String outputS3KeyPrefix;

    private final String maxConcurrency;

    private final String maxErrors;

    private final Integer targetCount;

    private final Integer completedCount;

    private final Integer errorCount;

    private final String serviceRole;

    private final NotificationConfig notificationConfig;

    private Command(BuilderImpl builder) {
        this.commandId = builder.commandId;
        this.documentName = builder.documentName;
        this.comment = builder.comment;
        this.expiresAfter = builder.expiresAfter;
        this.parameters = builder.parameters;
        this.instanceIds = builder.instanceIds;
        this.targets = builder.targets;
        this.requestedDateTime = builder.requestedDateTime;
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.outputS3Region = builder.outputS3Region;
        this.outputS3BucketName = builder.outputS3BucketName;
        this.outputS3KeyPrefix = builder.outputS3KeyPrefix;
        this.maxConcurrency = builder.maxConcurrency;
        this.maxErrors = builder.maxErrors;
        this.targetCount = builder.targetCount;
        this.completedCount = builder.completedCount;
        this.errorCount = builder.errorCount;
        this.serviceRole = builder.serviceRole;
        this.notificationConfig = builder.notificationConfig;
    }

    /**
     * <p>
     * A unique identifier for this command.
     * </p>
     * 
     * @return A unique identifier for this command.
     */
    public String commandId() {
        return commandId;
    }

    /**
     * <p>
     * The name of the document requested for execution.
     * </p>
     * 
     * @return The name of the document requested for execution.
     */
    public String documentName() {
        return documentName;
    }

    /**
     * <p>
     * User-specified information about the command, such as a brief description of what the command should do.
     * </p>
     * 
     * @return User-specified information about the command, such as a brief description of what the command should do.
     */
    public String comment() {
        return comment;
    }

    /**
     * <p>
     * If this time is reached and the command has not already started executing, it will not execute. Calculated based
     * on the ExpiresAfter user input provided as part of the SendCommand API.
     * </p>
     * 
     * @return If this time is reached and the command has not already started executing, it will not execute.
     *         Calculated based on the ExpiresAfter user input provided as part of the SendCommand API.
     */
    public Instant expiresAfter() {
        return expiresAfter;
    }

    /**
     * <p>
     * The parameter values to be inserted in the document when executing the command.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The parameter values to be inserted in the document when executing the command.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The instance IDs against which this command was requested.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The instance IDs against which this command was requested.
     */
    public List<String> instanceIds() {
        return instanceIds;
    }

    /**
     * <p>
     * An array of search criteria that targets instances using a Key,Value combination that you specify. Targets is
     * required if you don't provide one or more instance IDs in the call.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of search criteria that targets instances using a Key,Value combination that you specify.
     *         Targets is required if you don't provide one or more instance IDs in the call.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The date and time the command was requested.
     * </p>
     * 
     * @return The date and time the command was requested.
     */
    public Instant requestedDateTime() {
        return requestedDateTime;
    }

    /**
     * <p>
     * The status of the command.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CommandStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusString}.
     * </p>
     * 
     * @return The status of the command.
     * @see CommandStatus
     */
    public CommandStatus status() {
        return CommandStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the command.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CommandStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusString}.
     * </p>
     * 
     * @return The status of the command.
     * @see CommandStatus
     */
    public String statusString() {
        return status;
    }

    /**
     * <p>
     * A detailed status of the command execution. StatusDetails includes more information than Status because it
     * includes states resulting from error and concurrency control parameters. StatusDetails can show different results
     * than Status. For more information about these statuses, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-about-status.html">Run Command
     * Status</a>. StatusDetails can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pending: The command has not been sent to any instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * In Progress: The command has been sent to at least one instance but has not reached a final state on all
     * instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Success: The command successfully executed on all invocations. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delivery Timed Out: The value of MaxErrors or more command invocations shows a status of Delivery Timed Out. This
     * is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Execution Timed Out: The value of MaxErrors or more command invocations shows a status of Execution Timed Out.
     * This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The value of MaxErrors or more command invocations shows a status of Failed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Incomplete: The command was attempted on all instances and one or more invocations does not have a value of
     * Success but not enough invocations failed for the status to be Failed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Canceled: The command was terminated before it was completed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Rate Exceeded: The number of instances targeted by the command exceeded the account limit for pending
     * invocations. The system has canceled the command before executing it on any instance. This is a terminal state.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A detailed status of the command execution. StatusDetails includes more information than Status because
     *         it includes states resulting from error and concurrency control parameters. StatusDetails can show
     *         different results than Status. For more information about these statuses, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-about-status.html">Run Command
     *         Status</a>. StatusDetails can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pending: The command has not been sent to any instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In Progress: The command has been sent to at least one instance but has not reached a final state on all
     *         instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Success: The command successfully executed on all invocations. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delivery Timed Out: The value of MaxErrors or more command invocations shows a status of Delivery Timed
     *         Out. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Execution Timed Out: The value of MaxErrors or more command invocations shows a status of Execution Timed
     *         Out. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The value of MaxErrors or more command invocations shows a status of Failed. This is a terminal
     *         state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Incomplete: The command was attempted on all instances and one or more invocations does not have a value
     *         of Success but not enough invocations failed for the status to be Failed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Canceled: The command was terminated before it was completed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Rate Exceeded: The number of instances targeted by the command exceeded the account limit for pending
     *         invocations. The system has canceled the command before executing it on any instance. This is a terminal
     *         state.
     *         </p>
     *         </li>
     */
    public String statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
     * automatically determines the Amazon S3 bucket region.
     * </p>
     * 
     * @return (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
     *         automatically determines the Amazon S3 bucket region.
     */
    public String outputS3Region() {
        return outputS3Region;
    }

    /**
     * <p>
     * The S3 bucket where the responses to the command executions should be stored. This was requested when issuing the
     * command.
     * </p>
     * 
     * @return The S3 bucket where the responses to the command executions should be stored. This was requested when
     *         issuing the command.
     */
    public String outputS3BucketName() {
        return outputS3BucketName;
    }

    /**
     * <p>
     * The S3 directory path inside the bucket where the responses to the command executions should be stored. This was
     * requested when issuing the command.
     * </p>
     * 
     * @return The S3 directory path inside the bucket where the responses to the command executions should be stored.
     *         This was requested when issuing the command.
     */
    public String outputS3KeyPrefix() {
        return outputS3KeyPrefix;
    }

    /**
     * <p>
     * The maximum number of instances that are allowed to execute the command at the same time. You can specify a
     * number of instances, such as 10, or a percentage of instances, such as 10%. The default value is 50. For more
     * information about how to use MaxConcurrency, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command Using
     * Systems Manager Run Command</a>.
     * </p>
     * 
     * @return The maximum number of instances that are allowed to execute the command at the same time. You can specify
     *         a number of instances, such as 10, or a percentage of instances, such as 10%. The default value is 50.
     *         For more information about how to use MaxConcurrency, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command
     *         Using Systems Manager Run Command</a>.
     */
    public String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The maximum number of errors allowed before the system stops sending the command to additional targets. You can
     * specify a number of errors, such as 10, or a percentage or errors, such as 10%. The default value is 50. For more
     * information about how to use MaxErrors, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command Using
     * Systems Manager Run Command</a>.
     * </p>
     * 
     * @return The maximum number of errors allowed before the system stops sending the command to additional targets.
     *         You can specify a number of errors, such as 10, or a percentage or errors, such as 10%. The default value
     *         is 50. For more information about how to use MaxErrors, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command
     *         Using Systems Manager Run Command</a>.
     */
    public String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * The number of targets for the command.
     * </p>
     * 
     * @return The number of targets for the command.
     */
    public Integer targetCount() {
        return targetCount;
    }

    /**
     * <p>
     * The number of targets for which the command invocation reached a terminal state. Terminal states include the
     * following: Success, Failed, Execution Timed Out, Delivery Timed Out, Canceled, Terminated, or Undeliverable.
     * </p>
     * 
     * @return The number of targets for which the command invocation reached a terminal state. Terminal states include
     *         the following: Success, Failed, Execution Timed Out, Delivery Timed Out, Canceled, Terminated, or
     *         Undeliverable.
     */
    public Integer completedCount() {
        return completedCount;
    }

    /**
     * <p>
     * The number of targets for which the status is Failed or Execution Timed Out.
     * </p>
     * 
     * @return The number of targets for which the status is Failed or Execution Timed Out.
     */
    public Integer errorCount() {
        return errorCount;
    }

    /**
     * <p>
     * The IAM service role that Run Command uses to act on your behalf when sending notifications about command status
     * changes.
     * </p>
     * 
     * @return The IAM service role that Run Command uses to act on your behalf when sending notifications about command
     *         status changes.
     */
    public String serviceRole() {
        return serviceRole;
    }

    /**
     * <p>
     * Configurations for sending notifications about command status changes.
     * </p>
     * 
     * @return Configurations for sending notifications about command status changes.
     */
    public NotificationConfig notificationConfig() {
        return notificationConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(commandId());
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(expiresAfter());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(instanceIds());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(requestedDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusString());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3Region());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3BucketName());
        hashCode = 31 * hashCode + Objects.hashCode(outputS3KeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(targetCount());
        hashCode = 31 * hashCode + Objects.hashCode(completedCount());
        hashCode = 31 * hashCode + Objects.hashCode(errorCount());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRole());
        hashCode = 31 * hashCode + Objects.hashCode(notificationConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Command)) {
            return false;
        }
        Command other = (Command) obj;
        return Objects.equals(commandId(), other.commandId()) && Objects.equals(documentName(), other.documentName())
                && Objects.equals(comment(), other.comment()) && Objects.equals(expiresAfter(), other.expiresAfter())
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(instanceIds(), other.instanceIds())
                && Objects.equals(targets(), other.targets()) && Objects.equals(requestedDateTime(), other.requestedDateTime())
                && Objects.equals(statusString(), other.statusString()) && Objects.equals(statusDetails(), other.statusDetails())
                && Objects.equals(outputS3Region(), other.outputS3Region())
                && Objects.equals(outputS3BucketName(), other.outputS3BucketName())
                && Objects.equals(outputS3KeyPrefix(), other.outputS3KeyPrefix())
                && Objects.equals(maxConcurrency(), other.maxConcurrency()) && Objects.equals(maxErrors(), other.maxErrors())
                && Objects.equals(targetCount(), other.targetCount()) && Objects.equals(completedCount(), other.completedCount())
                && Objects.equals(errorCount(), other.errorCount()) && Objects.equals(serviceRole(), other.serviceRole())
                && Objects.equals(notificationConfig(), other.notificationConfig());
    }

    @Override
    public String toString() {
        return ToString.builder("Command").add("CommandId", commandId()).add("DocumentName", documentName())
                .add("Comment", comment()).add("ExpiresAfter", expiresAfter()).add("Parameters", parameters())
                .add("InstanceIds", instanceIds()).add("Targets", targets()).add("RequestedDateTime", requestedDateTime())
                .add("Status", statusString()).add("StatusDetails", statusDetails()).add("OutputS3Region", outputS3Region())
                .add("OutputS3BucketName", outputS3BucketName()).add("OutputS3KeyPrefix", outputS3KeyPrefix())
                .add("MaxConcurrency", maxConcurrency()).add("MaxErrors", maxErrors()).add("TargetCount", targetCount())
                .add("CompletedCount", completedCount()).add("ErrorCount", errorCount()).add("ServiceRole", serviceRole())
                .add("NotificationConfig", notificationConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CommandId":
            return Optional.of(clazz.cast(commandId()));
        case "DocumentName":
            return Optional.of(clazz.cast(documentName()));
        case "Comment":
            return Optional.of(clazz.cast(comment()));
        case "ExpiresAfter":
            return Optional.of(clazz.cast(expiresAfter()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "InstanceIds":
            return Optional.of(clazz.cast(instanceIds()));
        case "Targets":
            return Optional.of(clazz.cast(targets()));
        case "RequestedDateTime":
            return Optional.of(clazz.cast(requestedDateTime()));
        case "Status":
            return Optional.of(clazz.cast(statusString()));
        case "StatusDetails":
            return Optional.of(clazz.cast(statusDetails()));
        case "OutputS3Region":
            return Optional.of(clazz.cast(outputS3Region()));
        case "OutputS3BucketName":
            return Optional.of(clazz.cast(outputS3BucketName()));
        case "OutputS3KeyPrefix":
            return Optional.of(clazz.cast(outputS3KeyPrefix()));
        case "MaxConcurrency":
            return Optional.of(clazz.cast(maxConcurrency()));
        case "MaxErrors":
            return Optional.of(clazz.cast(maxErrors()));
        case "TargetCount":
            return Optional.of(clazz.cast(targetCount()));
        case "CompletedCount":
            return Optional.of(clazz.cast(completedCount()));
        case "ErrorCount":
            return Optional.of(clazz.cast(errorCount()));
        case "ServiceRole":
            return Optional.of(clazz.cast(serviceRole()));
        case "NotificationConfig":
            return Optional.of(clazz.cast(notificationConfig()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        CommandMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Command> {
        /**
         * <p>
         * A unique identifier for this command.
         * </p>
         * 
         * @param commandId
         *        A unique identifier for this command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commandId(String commandId);

        /**
         * <p>
         * The name of the document requested for execution.
         * </p>
         * 
         * @param documentName
         *        The name of the document requested for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * User-specified information about the command, such as a brief description of what the command should do.
         * </p>
         * 
         * @param comment
         *        User-specified information about the command, such as a brief description of what the command should
         *        do.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * If this time is reached and the command has not already started executing, it will not execute. Calculated
         * based on the ExpiresAfter user input provided as part of the SendCommand API.
         * </p>
         * 
         * @param expiresAfter
         *        If this time is reached and the command has not already started executing, it will not execute.
         *        Calculated based on the ExpiresAfter user input provided as part of the SendCommand API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiresAfter(Instant expiresAfter);

        /**
         * <p>
         * The parameter values to be inserted in the document when executing the command.
         * </p>
         * 
         * @param parameters
         *        The parameter values to be inserted in the document when executing the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The instance IDs against which this command was requested.
         * </p>
         * 
         * @param instanceIds
         *        The instance IDs against which this command was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceIds(Collection<String> instanceIds);

        /**
         * <p>
         * The instance IDs against which this command was requested.
         * </p>
         * 
         * @param instanceIds
         *        The instance IDs against which this command was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceIds(String... instanceIds);

        /**
         * <p>
         * An array of search criteria that targets instances using a Key,Value combination that you specify. Targets is
         * required if you don't provide one or more instance IDs in the call.
         * </p>
         * 
         * @param targets
         *        An array of search criteria that targets instances using a Key,Value combination that you specify.
         *        Targets is required if you don't provide one or more instance IDs in the call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * An array of search criteria that targets instances using a Key,Value combination that you specify. Targets is
         * required if you don't provide one or more instance IDs in the call.
         * </p>
         * 
         * @param targets
         *        An array of search criteria that targets instances using a Key,Value combination that you specify.
         *        Targets is required if you don't provide one or more instance IDs in the call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The date and time the command was requested.
         * </p>
         * 
         * @param requestedDateTime
         *        The date and time the command was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedDateTime(Instant requestedDateTime);

        /**
         * <p>
         * The status of the command.
         * </p>
         * 
         * @param status
         *        The status of the command.
         * @see CommandStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the command.
         * </p>
         * 
         * @param status
         *        The status of the command.
         * @see CommandStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandStatus
         */
        Builder status(CommandStatus status);

        /**
         * <p>
         * A detailed status of the command execution. StatusDetails includes more information than Status because it
         * includes states resulting from error and concurrency control parameters. StatusDetails can show different
         * results than Status. For more information about these statuses, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-about-status.html">Run Command
         * Status</a>. StatusDetails can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pending: The command has not been sent to any instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * In Progress: The command has been sent to at least one instance but has not reached a final state on all
         * instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Success: The command successfully executed on all invocations. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Delivery Timed Out: The value of MaxErrors or more command invocations shows a status of Delivery Timed Out.
         * This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Execution Timed Out: The value of MaxErrors or more command invocations shows a status of Execution Timed
         * Out. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The value of MaxErrors or more command invocations shows a status of Failed. This is a terminal
         * state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Incomplete: The command was attempted on all instances and one or more invocations does not have a value of
         * Success but not enough invocations failed for the status to be Failed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Canceled: The command was terminated before it was completed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Rate Exceeded: The number of instances targeted by the command exceeded the account limit for pending
         * invocations. The system has canceled the command before executing it on any instance. This is a terminal
         * state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusDetails
         *        A detailed status of the command execution. StatusDetails includes more information than Status
         *        because it includes states resulting from error and concurrency control parameters. StatusDetails can
         *        show different results than Status. For more information about these statuses, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-about-status.html">Run
         *        Command Status</a>. StatusDetails can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pending: The command has not been sent to any instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In Progress: The command has been sent to at least one instance but has not reached a final state on
         *        all instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Success: The command successfully executed on all invocations. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delivery Timed Out: The value of MaxErrors or more command invocations shows a status of Delivery
         *        Timed Out. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Execution Timed Out: The value of MaxErrors or more command invocations shows a status of Execution
         *        Timed Out. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The value of MaxErrors or more command invocations shows a status of Failed. This is a
         *        terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Incomplete: The command was attempted on all instances and one or more invocations does not have a
         *        value of Success but not enough invocations failed for the status to be Failed. This is a terminal
         *        state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Canceled: The command was terminated before it was completed. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Rate Exceeded: The number of instances targeted by the command exceeded the account limit for pending
         *        invocations. The system has canceled the command before executing it on any instance. This is a
         *        terminal state.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(String statusDetails);

        /**
         * <p>
         * (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
         * automatically determines the Amazon S3 bucket region.
         * </p>
         * 
         * @param outputS3Region
         *        (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
         *        automatically determines the Amazon S3 bucket region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3Region(String outputS3Region);

        /**
         * <p>
         * The S3 bucket where the responses to the command executions should be stored. This was requested when issuing
         * the command.
         * </p>
         * 
         * @param outputS3BucketName
         *        The S3 bucket where the responses to the command executions should be stored. This was requested when
         *        issuing the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3BucketName(String outputS3BucketName);

        /**
         * <p>
         * The S3 directory path inside the bucket where the responses to the command executions should be stored. This
         * was requested when issuing the command.
         * </p>
         * 
         * @param outputS3KeyPrefix
         *        The S3 directory path inside the bucket where the responses to the command executions should be
         *        stored. This was requested when issuing the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3KeyPrefix(String outputS3KeyPrefix);

        /**
         * <p>
         * The maximum number of instances that are allowed to execute the command at the same time. You can specify a
         * number of instances, such as 10, or a percentage of instances, such as 10%. The default value is 50. For more
         * information about how to use MaxConcurrency, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command Using
         * Systems Manager Run Command</a>.
         * </p>
         * 
         * @param maxConcurrency
         *        The maximum number of instances that are allowed to execute the command at the same time. You can
         *        specify a number of instances, such as 10, or a percentage of instances, such as 10%. The default
         *        value is 50. For more information about how to use MaxConcurrency, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a
         *        Command Using Systems Manager Run Command</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The maximum number of errors allowed before the system stops sending the command to additional targets. You
         * can specify a number of errors, such as 10, or a percentage or errors, such as 10%. The default value is 50.
         * For more information about how to use MaxErrors, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command Using
         * Systems Manager Run Command</a>.
         * </p>
         * 
         * @param maxErrors
         *        The maximum number of errors allowed before the system stops sending the command to additional
         *        targets. You can specify a number of errors, such as 10, or a percentage or errors, such as 10%. The
         *        default value is 50. For more information about how to use MaxErrors, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a
         *        Command Using Systems Manager Run Command</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * The number of targets for the command.
         * </p>
         * 
         * @param targetCount
         *        The number of targets for the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetCount(Integer targetCount);

        /**
         * <p>
         * The number of targets for which the command invocation reached a terminal state. Terminal states include the
         * following: Success, Failed, Execution Timed Out, Delivery Timed Out, Canceled, Terminated, or Undeliverable.
         * </p>
         * 
         * @param completedCount
         *        The number of targets for which the command invocation reached a terminal state. Terminal states
         *        include the following: Success, Failed, Execution Timed Out, Delivery Timed Out, Canceled, Terminated,
         *        or Undeliverable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedCount(Integer completedCount);

        /**
         * <p>
         * The number of targets for which the status is Failed or Execution Timed Out.
         * </p>
         * 
         * @param errorCount
         *        The number of targets for which the status is Failed or Execution Timed Out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorCount(Integer errorCount);

        /**
         * <p>
         * The IAM service role that Run Command uses to act on your behalf when sending notifications about command
         * status changes.
         * </p>
         * 
         * @param serviceRole
         *        The IAM service role that Run Command uses to act on your behalf when sending notifications about
         *        command status changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * Configurations for sending notifications about command status changes.
         * </p>
         * 
         * @param notificationConfig
         *        Configurations for sending notifications about command status changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationConfig(NotificationConfig notificationConfig);

        /**
         * <p>
         * Configurations for sending notifications about command status changes.
         * </p>
         * This is a convenience that creates an instance of the {@link NotificationConfig.Builder} avoiding the need to
         * create one manually via {@link NotificationConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link NotificationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #notificationConfig(NotificationConfig)}.
         * 
         * @param notificationConfig
         *        a consumer that will call methods on {@link NotificationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationConfig(NotificationConfig)
         */
        default Builder notificationConfig(Consumer<NotificationConfig.Builder> notificationConfig) {
            return notificationConfig(NotificationConfig.builder().apply(notificationConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String commandId;

        private String documentName;

        private String comment;

        private Instant expiresAfter;

        private Map<String, List<String>> parameters;

        private List<String> instanceIds;

        private List<Target> targets;

        private Instant requestedDateTime;

        private String status;

        private String statusDetails;

        private String outputS3Region;

        private String outputS3BucketName;

        private String outputS3KeyPrefix;

        private String maxConcurrency;

        private String maxErrors;

        private Integer targetCount;

        private Integer completedCount;

        private Integer errorCount;

        private String serviceRole;

        private NotificationConfig notificationConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Command model) {
            commandId(model.commandId);
            documentName(model.documentName);
            comment(model.comment);
            expiresAfter(model.expiresAfter);
            parameters(model.parameters);
            instanceIds(model.instanceIds);
            targets(model.targets);
            requestedDateTime(model.requestedDateTime);
            status(model.status);
            statusDetails(model.statusDetails);
            outputS3Region(model.outputS3Region);
            outputS3BucketName(model.outputS3BucketName);
            outputS3KeyPrefix(model.outputS3KeyPrefix);
            maxConcurrency(model.maxConcurrency);
            maxErrors(model.maxErrors);
            targetCount(model.targetCount);
            completedCount(model.completedCount);
            errorCount(model.errorCount);
            serviceRole(model.serviceRole);
            notificationConfig(model.notificationConfig);
        }

        public final String getCommandId() {
            return commandId;
        }

        @Override
        public final Builder commandId(String commandId) {
            this.commandId = commandId;
            return this;
        }

        public final void setCommandId(String commandId) {
            this.commandId = commandId;
        }

        public final String getDocumentName() {
            return documentName;
        }

        @Override
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        public final String getComment() {
            return comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        public final Instant getExpiresAfter() {
            return expiresAfter;
        }

        @Override
        public final Builder expiresAfter(Instant expiresAfter) {
            this.expiresAfter = expiresAfter;
            return this;
        }

        public final void setExpiresAfter(Instant expiresAfter) {
            this.expiresAfter = expiresAfter;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final Collection<String> getInstanceIds() {
            return instanceIds;
        }

        @Override
        public final Builder instanceIds(Collection<String> instanceIds) {
            this.instanceIds = InstanceIdListCopier.copy(instanceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceIds(String... instanceIds) {
            instanceIds(Arrays.asList(instanceIds));
            return this;
        }

        public final void setInstanceIds(Collection<String> instanceIds) {
            this.instanceIds = InstanceIdListCopier.copy(instanceIds);
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final Instant getRequestedDateTime() {
            return requestedDateTime;
        }

        @Override
        public final Builder requestedDateTime(Instant requestedDateTime) {
            this.requestedDateTime = requestedDateTime;
            return this;
        }

        public final void setRequestedDateTime(Instant requestedDateTime) {
            this.requestedDateTime = requestedDateTime;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CommandStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusDetails() {
            return statusDetails;
        }

        @Override
        public final Builder statusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final void setStatusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
        }

        public final String getOutputS3Region() {
            return outputS3Region;
        }

        @Override
        public final Builder outputS3Region(String outputS3Region) {
            this.outputS3Region = outputS3Region;
            return this;
        }

        public final void setOutputS3Region(String outputS3Region) {
            this.outputS3Region = outputS3Region;
        }

        public final String getOutputS3BucketName() {
            return outputS3BucketName;
        }

        @Override
        public final Builder outputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
            return this;
        }

        public final void setOutputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
        }

        public final String getOutputS3KeyPrefix() {
            return outputS3KeyPrefix;
        }

        @Override
        public final Builder outputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
            return this;
        }

        public final void setOutputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        @Override
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        public final Integer getTargetCount() {
            return targetCount;
        }

        @Override
        public final Builder targetCount(Integer targetCount) {
            this.targetCount = targetCount;
            return this;
        }

        public final void setTargetCount(Integer targetCount) {
            this.targetCount = targetCount;
        }

        public final Integer getCompletedCount() {
            return completedCount;
        }

        @Override
        public final Builder completedCount(Integer completedCount) {
            this.completedCount = completedCount;
            return this;
        }

        public final void setCompletedCount(Integer completedCount) {
            this.completedCount = completedCount;
        }

        public final Integer getErrorCount() {
            return errorCount;
        }

        @Override
        public final Builder errorCount(Integer errorCount) {
            this.errorCount = errorCount;
            return this;
        }

        public final void setErrorCount(Integer errorCount) {
            this.errorCount = errorCount;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        public final NotificationConfig.Builder getNotificationConfig() {
            return notificationConfig != null ? notificationConfig.toBuilder() : null;
        }

        @Override
        public final Builder notificationConfig(NotificationConfig notificationConfig) {
            this.notificationConfig = notificationConfig;
            return this;
        }

        public final void setNotificationConfig(NotificationConfig.BuilderImpl notificationConfig) {
            this.notificationConfig = notificationConfig != null ? notificationConfig.build() : null;
        }

        @Override
        public Command build() {
            return new Command(this);
        }
    }
}
