/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about an the execution state of an Automation step.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StepExecution implements SdkPojo, Serializable, ToCopyableBuilder<StepExecution.Builder, StepExecution> {
    private static final SdkField<String> STEP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StepExecution::stepName)).setter(setter(Builder::stepName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepName").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StepExecution::action)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<Instant> EXECUTION_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(StepExecution::executionStartTime)).setter(setter(Builder::executionStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionStartTime").build())
            .build();

    private static final SdkField<Instant> EXECUTION_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(StepExecution::executionEndTime)).setter(setter(Builder::executionEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionEndTime").build()).build();

    private static final SdkField<String> STEP_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StepExecution::stepStatusAsString)).setter(setter(Builder::stepStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepStatus").build()).build();

    private static final SdkField<String> RESPONSE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StepExecution::responseCode)).setter(setter(Builder::responseCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseCode").build()).build();

    private static final SdkField<Map<String, String>> INPUTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(StepExecution::inputs))
            .setter(setter(Builder::inputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Inputs").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, List<String>>> OUTPUTS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .getter(getter(StepExecution::outputs))
            .setter(setter(Builder::outputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Outputs").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> RESPONSE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StepExecution::response)).setter(setter(Builder::response))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Response").build()).build();

    private static final SdkField<String> FAILURE_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StepExecution::failureMessage)).setter(setter(Builder::failureMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureMessage").build()).build();

    private static final SdkField<FailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<FailureDetails> builder(MarshallingType.SDK_POJO).getter(getter(StepExecution::failureDetails))
            .setter(setter(Builder::failureDetails)).constructor(FailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STEP_NAME_FIELD, ACTION_FIELD,
            EXECUTION_START_TIME_FIELD, EXECUTION_END_TIME_FIELD, STEP_STATUS_FIELD, RESPONSE_CODE_FIELD, INPUTS_FIELD,
            OUTPUTS_FIELD, RESPONSE_FIELD, FAILURE_MESSAGE_FIELD, FAILURE_DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stepName;

    private final String action;

    private final Instant executionStartTime;

    private final Instant executionEndTime;

    private final String stepStatus;

    private final String responseCode;

    private final Map<String, String> inputs;

    private final Map<String, List<String>> outputs;

    private final String response;

    private final String failureMessage;

    private final FailureDetails failureDetails;

    private StepExecution(BuilderImpl builder) {
        this.stepName = builder.stepName;
        this.action = builder.action;
        this.executionStartTime = builder.executionStartTime;
        this.executionEndTime = builder.executionEndTime;
        this.stepStatus = builder.stepStatus;
        this.responseCode = builder.responseCode;
        this.inputs = builder.inputs;
        this.outputs = builder.outputs;
        this.response = builder.response;
        this.failureMessage = builder.failureMessage;
        this.failureDetails = builder.failureDetails;
    }

    /**
     * <p>
     * The name of this execution step.
     * </p>
     * 
     * @return The name of this execution step.
     */
    public String stepName() {
        return stepName;
    }

    /**
     * <p>
     * The action this step performs. The action determines the behavior of the step.
     * </p>
     * 
     * @return The action this step performs. The action determines the behavior of the step.
     */
    public String action() {
        return action;
    }

    /**
     * <p>
     * If a step has begun execution, this contains the time the step started. If the step is in Pending status, this
     * field is not populated.
     * </p>
     * 
     * @return If a step has begun execution, this contains the time the step started. If the step is in Pending status,
     *         this field is not populated.
     */
    public Instant executionStartTime() {
        return executionStartTime;
    }

    /**
     * <p>
     * If a step has finished execution, this contains the time the execution ended. If the step has not yet concluded,
     * this field is not populated.
     * </p>
     * 
     * @return If a step has finished execution, this contains the time the execution ended. If the step has not yet
     *         concluded, this field is not populated.
     */
    public Instant executionEndTime() {
        return executionEndTime;
    }

    /**
     * <p>
     * The execution status for this step. Valid values include: Pending, InProgress, Success, Cancelled, Failed, and
     * TimedOut.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stepStatus} will
     * return {@link AutomationExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stepStatusAsString}.
     * </p>
     * 
     * @return The execution status for this step. Valid values include: Pending, InProgress, Success, Cancelled,
     *         Failed, and TimedOut.
     * @see AutomationExecutionStatus
     */
    public AutomationExecutionStatus stepStatus() {
        return AutomationExecutionStatus.fromValue(stepStatus);
    }

    /**
     * <p>
     * The execution status for this step. Valid values include: Pending, InProgress, Success, Cancelled, Failed, and
     * TimedOut.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stepStatus} will
     * return {@link AutomationExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stepStatusAsString}.
     * </p>
     * 
     * @return The execution status for this step. Valid values include: Pending, InProgress, Success, Cancelled,
     *         Failed, and TimedOut.
     * @see AutomationExecutionStatus
     */
    public String stepStatusAsString() {
        return stepStatus;
    }

    /**
     * <p>
     * The response code returned by the execution of the step.
     * </p>
     * 
     * @return The response code returned by the execution of the step.
     */
    public String responseCode() {
        return responseCode;
    }

    /**
     * <p>
     * Fully-resolved values passed into the step before execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Fully-resolved values passed into the step before execution.
     */
    public Map<String, String> inputs() {
        return inputs;
    }

    /**
     * <p>
     * Returned values from the execution of the step.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Returned values from the execution of the step.
     */
    public Map<String, List<String>> outputs() {
        return outputs;
    }

    /**
     * <p>
     * A message associated with the response code for an execution.
     * </p>
     * 
     * @return A message associated with the response code for an execution.
     */
    public String response() {
        return response;
    }

    /**
     * <p>
     * If a step failed, this message explains why the execution failed.
     * </p>
     * 
     * @return If a step failed, this message explains why the execution failed.
     */
    public String failureMessage() {
        return failureMessage;
    }

    /**
     * <p>
     * Information about the Automation failure.
     * </p>
     * 
     * @return Information about the Automation failure.
     */
    public FailureDetails failureDetails() {
        return failureDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stepName());
        hashCode = 31 * hashCode + Objects.hashCode(action());
        hashCode = 31 * hashCode + Objects.hashCode(executionStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(executionEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(stepStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(responseCode());
        hashCode = 31 * hashCode + Objects.hashCode(inputs());
        hashCode = 31 * hashCode + Objects.hashCode(outputs());
        hashCode = 31 * hashCode + Objects.hashCode(response());
        hashCode = 31 * hashCode + Objects.hashCode(failureMessage());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StepExecution)) {
            return false;
        }
        StepExecution other = (StepExecution) obj;
        return Objects.equals(stepName(), other.stepName()) && Objects.equals(action(), other.action())
                && Objects.equals(executionStartTime(), other.executionStartTime())
                && Objects.equals(executionEndTime(), other.executionEndTime())
                && Objects.equals(stepStatusAsString(), other.stepStatusAsString())
                && Objects.equals(responseCode(), other.responseCode()) && Objects.equals(inputs(), other.inputs())
                && Objects.equals(outputs(), other.outputs()) && Objects.equals(response(), other.response())
                && Objects.equals(failureMessage(), other.failureMessage())
                && Objects.equals(failureDetails(), other.failureDetails());
    }

    @Override
    public String toString() {
        return ToString.builder("StepExecution").add("StepName", stepName()).add("Action", action())
                .add("ExecutionStartTime", executionStartTime()).add("ExecutionEndTime", executionEndTime())
                .add("StepStatus", stepStatusAsString()).add("ResponseCode", responseCode()).add("Inputs", inputs())
                .add("Outputs", outputs()).add("Response", response()).add("FailureMessage", failureMessage())
                .add("FailureDetails", failureDetails()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StepName":
            return Optional.ofNullable(clazz.cast(stepName()));
        case "Action":
            return Optional.ofNullable(clazz.cast(action()));
        case "ExecutionStartTime":
            return Optional.ofNullable(clazz.cast(executionStartTime()));
        case "ExecutionEndTime":
            return Optional.ofNullable(clazz.cast(executionEndTime()));
        case "StepStatus":
            return Optional.ofNullable(clazz.cast(stepStatusAsString()));
        case "ResponseCode":
            return Optional.ofNullable(clazz.cast(responseCode()));
        case "Inputs":
            return Optional.ofNullable(clazz.cast(inputs()));
        case "Outputs":
            return Optional.ofNullable(clazz.cast(outputs()));
        case "Response":
            return Optional.ofNullable(clazz.cast(response()));
        case "FailureMessage":
            return Optional.ofNullable(clazz.cast(failureMessage()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StepExecution, T> g) {
        return obj -> g.apply((StepExecution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StepExecution> {
        /**
         * <p>
         * The name of this execution step.
         * </p>
         * 
         * @param stepName
         *        The name of this execution step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepName(String stepName);

        /**
         * <p>
         * The action this step performs. The action determines the behavior of the step.
         * </p>
         * 
         * @param action
         *        The action this step performs. The action determines the behavior of the step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder action(String action);

        /**
         * <p>
         * If a step has begun execution, this contains the time the step started. If the step is in Pending status,
         * this field is not populated.
         * </p>
         * 
         * @param executionStartTime
         *        If a step has begun execution, this contains the time the step started. If the step is in Pending
         *        status, this field is not populated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionStartTime(Instant executionStartTime);

        /**
         * <p>
         * If a step has finished execution, this contains the time the execution ended. If the step has not yet
         * concluded, this field is not populated.
         * </p>
         * 
         * @param executionEndTime
         *        If a step has finished execution, this contains the time the execution ended. If the step has not yet
         *        concluded, this field is not populated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionEndTime(Instant executionEndTime);

        /**
         * <p>
         * The execution status for this step. Valid values include: Pending, InProgress, Success, Cancelled, Failed,
         * and TimedOut.
         * </p>
         * 
         * @param stepStatus
         *        The execution status for this step. Valid values include: Pending, InProgress, Success, Cancelled,
         *        Failed, and TimedOut.
         * @see AutomationExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationExecutionStatus
         */
        Builder stepStatus(String stepStatus);

        /**
         * <p>
         * The execution status for this step. Valid values include: Pending, InProgress, Success, Cancelled, Failed,
         * and TimedOut.
         * </p>
         * 
         * @param stepStatus
         *        The execution status for this step. Valid values include: Pending, InProgress, Success, Cancelled,
         *        Failed, and TimedOut.
         * @see AutomationExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationExecutionStatus
         */
        Builder stepStatus(AutomationExecutionStatus stepStatus);

        /**
         * <p>
         * The response code returned by the execution of the step.
         * </p>
         * 
         * @param responseCode
         *        The response code returned by the execution of the step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCode(String responseCode);

        /**
         * <p>
         * Fully-resolved values passed into the step before execution.
         * </p>
         * 
         * @param inputs
         *        Fully-resolved values passed into the step before execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputs(Map<String, String> inputs);

        /**
         * <p>
         * Returned values from the execution of the step.
         * </p>
         * 
         * @param outputs
         *        Returned values from the execution of the step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Map<String, ? extends Collection<String>> outputs);

        /**
         * <p>
         * A message associated with the response code for an execution.
         * </p>
         * 
         * @param response
         *        A message associated with the response code for an execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder response(String response);

        /**
         * <p>
         * If a step failed, this message explains why the execution failed.
         * </p>
         * 
         * @param failureMessage
         *        If a step failed, this message explains why the execution failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessage(String failureMessage);

        /**
         * <p>
         * Information about the Automation failure.
         * </p>
         * 
         * @param failureDetails
         *        Information about the Automation failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(FailureDetails failureDetails);

        /**
         * <p>
         * Information about the Automation failure.
         * </p>
         * This is a convenience that creates an instance of the {@link FailureDetails.Builder} avoiding the need to
         * create one manually via {@link FailureDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link FailureDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #failureDetails(FailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link FailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(FailureDetails)
         */
        default Builder failureDetails(Consumer<FailureDetails.Builder> failureDetails) {
            return failureDetails(FailureDetails.builder().applyMutation(failureDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String stepName;

        private String action;

        private Instant executionStartTime;

        private Instant executionEndTime;

        private String stepStatus;

        private String responseCode;

        private Map<String, String> inputs = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, List<String>> outputs = DefaultSdkAutoConstructMap.getInstance();

        private String response;

        private String failureMessage;

        private FailureDetails failureDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(StepExecution model) {
            stepName(model.stepName);
            action(model.action);
            executionStartTime(model.executionStartTime);
            executionEndTime(model.executionEndTime);
            stepStatus(model.stepStatus);
            responseCode(model.responseCode);
            inputs(model.inputs);
            outputs(model.outputs);
            response(model.response);
            failureMessage(model.failureMessage);
            failureDetails(model.failureDetails);
        }

        public final String getStepName() {
            return stepName;
        }

        @Override
        public final Builder stepName(String stepName) {
            this.stepName = stepName;
            return this;
        }

        public final void setStepName(String stepName) {
            this.stepName = stepName;
        }

        public final String getAction() {
            return action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        public final Instant getExecutionStartTime() {
            return executionStartTime;
        }

        @Override
        public final Builder executionStartTime(Instant executionStartTime) {
            this.executionStartTime = executionStartTime;
            return this;
        }

        public final void setExecutionStartTime(Instant executionStartTime) {
            this.executionStartTime = executionStartTime;
        }

        public final Instant getExecutionEndTime() {
            return executionEndTime;
        }

        @Override
        public final Builder executionEndTime(Instant executionEndTime) {
            this.executionEndTime = executionEndTime;
            return this;
        }

        public final void setExecutionEndTime(Instant executionEndTime) {
            this.executionEndTime = executionEndTime;
        }

        public final String getStepStatus() {
            return stepStatus;
        }

        @Override
        public final Builder stepStatus(String stepStatus) {
            this.stepStatus = stepStatus;
            return this;
        }

        @Override
        public final Builder stepStatus(AutomationExecutionStatus stepStatus) {
            this.stepStatus(stepStatus.toString());
            return this;
        }

        public final void setStepStatus(String stepStatus) {
            this.stepStatus = stepStatus;
        }

        public final String getResponseCode() {
            return responseCode;
        }

        @Override
        public final Builder responseCode(String responseCode) {
            this.responseCode = responseCode;
            return this;
        }

        public final void setResponseCode(String responseCode) {
            this.responseCode = responseCode;
        }

        public final Map<String, String> getInputs() {
            return inputs;
        }

        @Override
        public final Builder inputs(Map<String, String> inputs) {
            this.inputs = NormalStringMapCopier.copy(inputs);
            return this;
        }

        public final void setInputs(Map<String, String> inputs) {
            this.inputs = NormalStringMapCopier.copy(inputs);
        }

        public final Map<String, ? extends Collection<String>> getOutputs() {
            return outputs;
        }

        @Override
        public final Builder outputs(Map<String, ? extends Collection<String>> outputs) {
            this.outputs = AutomationParameterMapCopier.copy(outputs);
            return this;
        }

        public final void setOutputs(Map<String, ? extends Collection<String>> outputs) {
            this.outputs = AutomationParameterMapCopier.copy(outputs);
        }

        public final String getResponse() {
            return response;
        }

        @Override
        public final Builder response(String response) {
            this.response = response;
            return this;
        }

        public final void setResponse(String response) {
            this.response = response;
        }

        public final String getFailureMessage() {
            return failureMessage;
        }

        @Override
        public final Builder failureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
            return this;
        }

        public final void setFailureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
        }

        public final FailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        @Override
        public final Builder failureDetails(FailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final void setFailureDetails(FailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        @Override
        public StepExecution build() {
            return new StepExecution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
