/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.ssm.transform.ComplianceItemEntryMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a compliance item.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComplianceItemEntry implements StructuredPojo,
        ToCopyableBuilder<ComplianceItemEntry.Builder, ComplianceItemEntry> {
    private final String id;

    private final String title;

    private final String severity;

    private final String status;

    private final Map<String, String> details;

    private ComplianceItemEntry(BuilderImpl builder) {
        this.id = builder.id;
        this.title = builder.title;
        this.severity = builder.severity;
        this.status = builder.status;
        this.details = builder.details;
    }

    /**
     * <p>
     * The compliance item ID. For example, if the compliance item is a Windows patch, the ID could be the number of the
     * KB article.
     * </p>
     * 
     * @return The compliance item ID. For example, if the compliance item is a Windows patch, the ID could be the
     *         number of the KB article.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The title of the compliance item. For example, if the compliance item is a Windows patch, the title could be the
     * title of the KB article for the patch. Here's an example: Security Update for Active Directory Federation
     * Services.
     * </p>
     * 
     * @return The title of the compliance item. For example, if the compliance item is a Windows patch, the title could
     *         be the title of the KB article for the patch. Here's an example: Security Update for Active Directory
     *         Federation Services.
     */
    public String title() {
        return title;
    }

    /**
     * <p>
     * The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
     * Informational, Unspecified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link ComplianceSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
     *         Informational, Unspecified.
     * @see ComplianceSeverity
     */
    public ComplianceSeverity severity() {
        return ComplianceSeverity.fromValue(severity);
    }

    /**
     * <p>
     * The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
     * Informational, Unspecified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link ComplianceSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
     *         Informational, Unspecified.
     * @see ComplianceSeverity
     */
    public String severityAsString() {
        return severity;
    }

    /**
     * <p>
     * The status of the compliance item. An item is either COMPLIANT or NON_COMPLIANT.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComplianceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the compliance item. An item is either COMPLIANT or NON_COMPLIANT.
     * @see ComplianceStatus
     */
    public ComplianceStatus status() {
        return ComplianceStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the compliance item. An item is either COMPLIANT or NON_COMPLIANT.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComplianceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the compliance item. An item is either COMPLIANT or NON_COMPLIANT.
     * @see ComplianceStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A "Key": "Value" tag combination for the compliance item.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A "Key": "Value" tag combination for the compliance item.
     */
    public Map<String, String> details() {
        return details;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(severityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(details());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComplianceItemEntry)) {
            return false;
        }
        ComplianceItemEntry other = (ComplianceItemEntry) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(title(), other.title())
                && Objects.equals(severityAsString(), other.severityAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(details(), other.details());
    }

    @Override
    public String toString() {
        return ToString.builder("ComplianceItemEntry").add("Id", id()).add("Title", title()).add("Severity", severityAsString())
                .add("Status", statusAsString()).add("Details", details()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severityAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Details":
            return Optional.ofNullable(clazz.cast(details()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ComplianceItemEntryMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ComplianceItemEntry> {
        /**
         * <p>
         * The compliance item ID. For example, if the compliance item is a Windows patch, the ID could be the number of
         * the KB article.
         * </p>
         * 
         * @param id
         *        The compliance item ID. For example, if the compliance item is a Windows patch, the ID could be the
         *        number of the KB article.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The title of the compliance item. For example, if the compliance item is a Windows patch, the title could be
         * the title of the KB article for the patch. Here's an example: Security Update for Active Directory Federation
         * Services.
         * </p>
         * 
         * @param title
         *        The title of the compliance item. For example, if the compliance item is a Windows patch, the title
         *        could be the title of the KB article for the patch. Here's an example: Security Update for Active
         *        Directory Federation Services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
         * Informational, Unspecified.
         * </p>
         * 
         * @param severity
         *        The severity of the compliance status. Severity can be one of the following: Critical, High, Medium,
         *        Low, Informational, Unspecified.
         * @see ComplianceSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceSeverity
         */
        Builder severity(String severity);

        /**
         * <p>
         * The severity of the compliance status. Severity can be one of the following: Critical, High, Medium, Low,
         * Informational, Unspecified.
         * </p>
         * 
         * @param severity
         *        The severity of the compliance status. Severity can be one of the following: Critical, High, Medium,
         *        Low, Informational, Unspecified.
         * @see ComplianceSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceSeverity
         */
        Builder severity(ComplianceSeverity severity);

        /**
         * <p>
         * The status of the compliance item. An item is either COMPLIANT or NON_COMPLIANT.
         * </p>
         * 
         * @param status
         *        The status of the compliance item. An item is either COMPLIANT or NON_COMPLIANT.
         * @see ComplianceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the compliance item. An item is either COMPLIANT or NON_COMPLIANT.
         * </p>
         * 
         * @param status
         *        The status of the compliance item. An item is either COMPLIANT or NON_COMPLIANT.
         * @see ComplianceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceStatus
         */
        Builder status(ComplianceStatus status);

        /**
         * <p>
         * A "Key": "Value" tag combination for the compliance item.
         * </p>
         * 
         * @param details
         *        A "Key": "Value" tag combination for the compliance item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(Map<String, String> details);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String title;

        private String severity;

        private String status;

        private Map<String, String> details = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComplianceItemEntry model) {
            id(model.id);
            title(model.title);
            severity(model.severity);
            status(model.status);
            details(model.details);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getSeverity() {
            return severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        @Override
        public final Builder severity(ComplianceSeverity severity) {
            this.severity(severity.toString());
            return this;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ComplianceStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Map<String, String> getDetails() {
            return details;
        }

        @Override
        public final Builder details(Map<String, String> details) {
            this.details = ComplianceItemDetailsCopier.copy(details);
            return this;
        }

        public final void setDetails(Map<String, String> details) {
            this.details = ComplianceItemDetailsCopier.copy(details);
        }

        @Override
        public ComplianceItemEntry build() {
            return new ComplianceItemEntry(this);
        }
    }
}
