/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.ssm.transform.AutomationExecutionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about the current state of an individual Automation execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutomationExecution implements StructuredPojo,
        ToCopyableBuilder<AutomationExecution.Builder, AutomationExecution> {
    private final String automationExecutionId;

    private final String documentName;

    private final String documentVersion;

    private final Instant executionStartTime;

    private final Instant executionEndTime;

    private final String automationExecutionStatus;

    private final List<StepExecution> stepExecutions;

    private final Map<String, List<String>> parameters;

    private final Map<String, List<String>> outputs;

    private final String failureMessage;

    private AutomationExecution(BuilderImpl builder) {
        this.automationExecutionId = builder.automationExecutionId;
        this.documentName = builder.documentName;
        this.documentVersion = builder.documentVersion;
        this.executionStartTime = builder.executionStartTime;
        this.executionEndTime = builder.executionEndTime;
        this.automationExecutionStatus = builder.automationExecutionStatus;
        this.stepExecutions = builder.stepExecutions;
        this.parameters = builder.parameters;
        this.outputs = builder.outputs;
        this.failureMessage = builder.failureMessage;
    }

    /**
     * <p>
     * The execution ID.
     * </p>
     * 
     * @return The execution ID.
     */
    public String automationExecutionId() {
        return automationExecutionId;
    }

    /**
     * <p>
     * The name of the Automation document used during the execution.
     * </p>
     * 
     * @return The name of the Automation document used during the execution.
     */
    public String documentName() {
        return documentName;
    }

    /**
     * <p>
     * The version of the document to use during execution.
     * </p>
     * 
     * @return The version of the document to use during execution.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The time the execution started.
     * </p>
     * 
     * @return The time the execution started.
     */
    public Instant executionStartTime() {
        return executionStartTime;
    }

    /**
     * <p>
     * The time the execution finished.
     * </p>
     * 
     * @return The time the execution finished.
     */
    public Instant executionEndTime() {
        return executionEndTime;
    }

    /**
     * <p>
     * The execution status of the Automation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #automationExecutionStatus} will return {@link AutomationExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #automationExecutionStatusAsString}.
     * </p>
     * 
     * @return The execution status of the Automation.
     * @see AutomationExecutionStatus
     */
    public AutomationExecutionStatus automationExecutionStatus() {
        return AutomationExecutionStatus.fromValue(automationExecutionStatus);
    }

    /**
     * <p>
     * The execution status of the Automation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #automationExecutionStatus} will return {@link AutomationExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #automationExecutionStatusAsString}.
     * </p>
     * 
     * @return The execution status of the Automation.
     * @see AutomationExecutionStatus
     */
    public String automationExecutionStatusAsString() {
        return automationExecutionStatus;
    }

    /**
     * <p>
     * A list of details about the current state of all steps that comprise an execution. An Automation document
     * contains a list of steps that are executed in order.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of details about the current state of all steps that comprise an execution. An Automation document
     *         contains a list of steps that are executed in order.
     */
    public List<StepExecution> stepExecutions() {
        return stepExecutions;
    }

    /**
     * <p>
     * The key-value map of execution parameters, which were supplied when calling StartAutomationExecution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The key-value map of execution parameters, which were supplied when calling StartAutomationExecution.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The list of execution outputs as defined in the automation document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The list of execution outputs as defined in the automation document.
     */
    public Map<String, List<String>> outputs() {
        return outputs;
    }

    /**
     * <p>
     * A message describing why an execution has failed, if the status is set to Failed.
     * </p>
     * 
     * @return A message describing why an execution has failed, if the status is set to Failed.
     */
    public String failureMessage() {
        return failureMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(automationExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(executionStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(executionEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(automationExecutionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stepExecutions());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(outputs());
        hashCode = 31 * hashCode + Objects.hashCode(failureMessage());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutomationExecution)) {
            return false;
        }
        AutomationExecution other = (AutomationExecution) obj;
        return Objects.equals(automationExecutionId(), other.automationExecutionId())
                && Objects.equals(documentName(), other.documentName())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(executionStartTime(), other.executionStartTime())
                && Objects.equals(executionEndTime(), other.executionEndTime())
                && Objects.equals(automationExecutionStatusAsString(), other.automationExecutionStatusAsString())
                && Objects.equals(stepExecutions(), other.stepExecutions()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(outputs(), other.outputs()) && Objects.equals(failureMessage(), other.failureMessage());
    }

    @Override
    public String toString() {
        return ToString.builder("AutomationExecution").add("AutomationExecutionId", automationExecutionId())
                .add("DocumentName", documentName()).add("DocumentVersion", documentVersion())
                .add("ExecutionStartTime", executionStartTime()).add("ExecutionEndTime", executionEndTime())
                .add("AutomationExecutionStatus", automationExecutionStatusAsString()).add("StepExecutions", stepExecutions())
                .add("Parameters", parameters()).add("Outputs", outputs()).add("FailureMessage", failureMessage()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutomationExecutionId":
            return Optional.ofNullable(clazz.cast(automationExecutionId()));
        case "DocumentName":
            return Optional.ofNullable(clazz.cast(documentName()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "ExecutionStartTime":
            return Optional.ofNullable(clazz.cast(executionStartTime()));
        case "ExecutionEndTime":
            return Optional.ofNullable(clazz.cast(executionEndTime()));
        case "AutomationExecutionStatus":
            return Optional.ofNullable(clazz.cast(automationExecutionStatusAsString()));
        case "StepExecutions":
            return Optional.ofNullable(clazz.cast(stepExecutions()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "Outputs":
            return Optional.ofNullable(clazz.cast(outputs()));
        case "FailureMessage":
            return Optional.ofNullable(clazz.cast(failureMessage()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AutomationExecutionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AutomationExecution> {
        /**
         * <p>
         * The execution ID.
         * </p>
         * 
         * @param automationExecutionId
         *        The execution ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automationExecutionId(String automationExecutionId);

        /**
         * <p>
         * The name of the Automation document used during the execution.
         * </p>
         * 
         * @param documentName
         *        The name of the Automation document used during the execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * The version of the document to use during execution.
         * </p>
         * 
         * @param documentVersion
         *        The version of the document to use during execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The time the execution started.
         * </p>
         * 
         * @param executionStartTime
         *        The time the execution started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionStartTime(Instant executionStartTime);

        /**
         * <p>
         * The time the execution finished.
         * </p>
         * 
         * @param executionEndTime
         *        The time the execution finished.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionEndTime(Instant executionEndTime);

        /**
         * <p>
         * The execution status of the Automation.
         * </p>
         * 
         * @param automationExecutionStatus
         *        The execution status of the Automation.
         * @see AutomationExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationExecutionStatus
         */
        Builder automationExecutionStatus(String automationExecutionStatus);

        /**
         * <p>
         * The execution status of the Automation.
         * </p>
         * 
         * @param automationExecutionStatus
         *        The execution status of the Automation.
         * @see AutomationExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationExecutionStatus
         */
        Builder automationExecutionStatus(AutomationExecutionStatus automationExecutionStatus);

        /**
         * <p>
         * A list of details about the current state of all steps that comprise an execution. An Automation document
         * contains a list of steps that are executed in order.
         * </p>
         * 
         * @param stepExecutions
         *        A list of details about the current state of all steps that comprise an execution. An Automation
         *        document contains a list of steps that are executed in order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepExecutions(Collection<StepExecution> stepExecutions);

        /**
         * <p>
         * A list of details about the current state of all steps that comprise an execution. An Automation document
         * contains a list of steps that are executed in order.
         * </p>
         * 
         * @param stepExecutions
         *        A list of details about the current state of all steps that comprise an execution. An Automation
         *        document contains a list of steps that are executed in order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepExecutions(StepExecution... stepExecutions);

        /**
         * <p>
         * A list of details about the current state of all steps that comprise an execution. An Automation document
         * contains a list of steps that are executed in order.
         * </p>
         * This is a convenience that creates an instance of the {@link List<StepExecution>.Builder} avoiding the need
         * to create one manually via {@link List<StepExecution>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<StepExecution>.Builder#build()} is called immediately and
         * its result is passed to {@link #stepExecutions(List<StepExecution>)}.
         * 
         * @param stepExecutions
         *        a consumer that will call methods on {@link List<StepExecution>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stepExecutions(List<StepExecution>)
         */
        Builder stepExecutions(Consumer<StepExecution.Builder>... stepExecutions);

        /**
         * <p>
         * The key-value map of execution parameters, which were supplied when calling StartAutomationExecution.
         * </p>
         * 
         * @param parameters
         *        The key-value map of execution parameters, which were supplied when calling StartAutomationExecution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The list of execution outputs as defined in the automation document.
         * </p>
         * 
         * @param outputs
         *        The list of execution outputs as defined in the automation document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Map<String, ? extends Collection<String>> outputs);

        /**
         * <p>
         * A message describing why an execution has failed, if the status is set to Failed.
         * </p>
         * 
         * @param failureMessage
         *        A message describing why an execution has failed, if the status is set to Failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessage(String failureMessage);
    }

    static final class BuilderImpl implements Builder {
        private String automationExecutionId;

        private String documentName;

        private String documentVersion;

        private Instant executionStartTime;

        private Instant executionEndTime;

        private String automationExecutionStatus;

        private List<StepExecution> stepExecutions = DefaultSdkAutoConstructList.getInstance();

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, List<String>> outputs = DefaultSdkAutoConstructMap.getInstance();

        private String failureMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(AutomationExecution model) {
            automationExecutionId(model.automationExecutionId);
            documentName(model.documentName);
            documentVersion(model.documentVersion);
            executionStartTime(model.executionStartTime);
            executionEndTime(model.executionEndTime);
            automationExecutionStatus(model.automationExecutionStatus);
            stepExecutions(model.stepExecutions);
            parameters(model.parameters);
            outputs(model.outputs);
            failureMessage(model.failureMessage);
        }

        public final String getAutomationExecutionId() {
            return automationExecutionId;
        }

        @Override
        public final Builder automationExecutionId(String automationExecutionId) {
            this.automationExecutionId = automationExecutionId;
            return this;
        }

        public final void setAutomationExecutionId(String automationExecutionId) {
            this.automationExecutionId = automationExecutionId;
        }

        public final String getDocumentName() {
            return documentName;
        }

        @Override
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Instant getExecutionStartTime() {
            return executionStartTime;
        }

        @Override
        public final Builder executionStartTime(Instant executionStartTime) {
            this.executionStartTime = executionStartTime;
            return this;
        }

        public final void setExecutionStartTime(Instant executionStartTime) {
            this.executionStartTime = executionStartTime;
        }

        public final Instant getExecutionEndTime() {
            return executionEndTime;
        }

        @Override
        public final Builder executionEndTime(Instant executionEndTime) {
            this.executionEndTime = executionEndTime;
            return this;
        }

        public final void setExecutionEndTime(Instant executionEndTime) {
            this.executionEndTime = executionEndTime;
        }

        public final String getAutomationExecutionStatus() {
            return automationExecutionStatus;
        }

        @Override
        public final Builder automationExecutionStatus(String automationExecutionStatus) {
            this.automationExecutionStatus = automationExecutionStatus;
            return this;
        }

        @Override
        public final Builder automationExecutionStatus(AutomationExecutionStatus automationExecutionStatus) {
            this.automationExecutionStatus(automationExecutionStatus.toString());
            return this;
        }

        public final void setAutomationExecutionStatus(String automationExecutionStatus) {
            this.automationExecutionStatus = automationExecutionStatus;
        }

        public final Collection<StepExecution.Builder> getStepExecutions() {
            return stepExecutions != null ? stepExecutions.stream().map(StepExecution::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder stepExecutions(Collection<StepExecution> stepExecutions) {
            this.stepExecutions = StepExecutionListCopier.copy(stepExecutions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepExecutions(StepExecution... stepExecutions) {
            stepExecutions(Arrays.asList(stepExecutions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepExecutions(Consumer<StepExecution.Builder>... stepExecutions) {
            stepExecutions(Stream.of(stepExecutions).map(c -> StepExecution.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setStepExecutions(Collection<StepExecution.BuilderImpl> stepExecutions) {
            this.stepExecutions = StepExecutionListCopier.copyFromBuilder(stepExecutions);
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = AutomationParameterMapCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = AutomationParameterMapCopier.copy(parameters);
        }

        public final Map<String, ? extends Collection<String>> getOutputs() {
            return outputs;
        }

        @Override
        public final Builder outputs(Map<String, ? extends Collection<String>> outputs) {
            this.outputs = AutomationParameterMapCopier.copy(outputs);
            return this;
        }

        public final void setOutputs(Map<String, ? extends Collection<String>> outputs) {
            this.outputs = AutomationParameterMapCopier.copy(outputs);
        }

        public final String getFailureMessage() {
            return failureMessage;
        }

        @Override
        public final Builder failureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
            return this;
        }

        public final void setFailureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
        }

        @Override
        public AutomationExecution build() {
            return new AutomationExecution(this);
        }
    }
}
