/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.MaintenanceWindowExecutionTaskIdentityMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a task execution performed as part of a Maintenance Window execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MaintenanceWindowExecutionTaskIdentity implements StructuredPojo,
        ToCopyableBuilder<MaintenanceWindowExecutionTaskIdentity.Builder, MaintenanceWindowExecutionTaskIdentity> {
    private final String windowExecutionId;

    private final String taskExecutionId;

    private final String status;

    private final String statusDetails;

    private final Instant startTime;

    private final Instant endTime;

    private final String taskArn;

    private final String taskType;

    private MaintenanceWindowExecutionTaskIdentity(BuilderImpl builder) {
        this.windowExecutionId = builder.windowExecutionId;
        this.taskExecutionId = builder.taskExecutionId;
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.taskArn = builder.taskArn;
        this.taskType = builder.taskType;
    }

    /**
     * <p>
     * The ID of the Maintenance Window execution that ran the task.
     * </p>
     * 
     * @return The ID of the Maintenance Window execution that ran the task.
     */
    public String windowExecutionId() {
        return windowExecutionId;
    }

    /**
     * <p>
     * The ID of the specific task execution in the Maintenance Window execution.
     * </p>
     * 
     * @return The ID of the specific task execution in the Maintenance Window execution.
     */
    public String taskExecutionId() {
        return taskExecutionId;
    }

    /**
     * <p>
     * The status of the task execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MaintenanceWindowExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the task execution.
     * @see MaintenanceWindowExecutionStatus
     */
    public MaintenanceWindowExecutionStatus status() {
        return MaintenanceWindowExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the task execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MaintenanceWindowExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the task execution.
     * @see MaintenanceWindowExecutionStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The details explaining the status of the task execution. Only available for certain status values.
     * </p>
     * 
     * @return The details explaining the status of the task execution. Only available for certain status values.
     */
    public String statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * The time the task execution started.
     * </p>
     * 
     * @return The time the task execution started.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time the task execution finished.
     * </p>
     * 
     * @return The time the task execution finished.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The ARN of the executed task.
     * </p>
     * 
     * @return The ARN of the executed task.
     */
    public String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The type of executed task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskTypeAsString}.
     * </p>
     * 
     * @return The type of executed task.
     * @see MaintenanceWindowTaskType
     */
    public MaintenanceWindowTaskType taskType() {
        return MaintenanceWindowTaskType.fromValue(taskType);
    }

    /**
     * <p>
     * The type of executed task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskTypeAsString}.
     * </p>
     * 
     * @return The type of executed task.
     * @see MaintenanceWindowTaskType
     */
    public String taskTypeAsString() {
        return taskType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(windowExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(taskExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MaintenanceWindowExecutionTaskIdentity)) {
            return false;
        }
        MaintenanceWindowExecutionTaskIdentity other = (MaintenanceWindowExecutionTaskIdentity) obj;
        return Objects.equals(windowExecutionId(), other.windowExecutionId())
                && Objects.equals(taskExecutionId(), other.taskExecutionId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusDetails(), other.statusDetails()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(taskArn(), other.taskArn())
                && Objects.equals(taskTypeAsString(), other.taskTypeAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("MaintenanceWindowExecutionTaskIdentity").add("WindowExecutionId", windowExecutionId())
                .add("TaskExecutionId", taskExecutionId()).add("Status", statusAsString()).add("StatusDetails", statusDetails())
                .add("StartTime", startTime()).add("EndTime", endTime()).add("TaskArn", taskArn())
                .add("TaskType", taskTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WindowExecutionId":
            return Optional.ofNullable(clazz.cast(windowExecutionId()));
        case "TaskExecutionId":
            return Optional.ofNullable(clazz.cast(taskExecutionId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusDetails":
            return Optional.ofNullable(clazz.cast(statusDetails()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "TaskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "TaskType":
            return Optional.ofNullable(clazz.cast(taskTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        MaintenanceWindowExecutionTaskIdentityMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, MaintenanceWindowExecutionTaskIdentity> {
        /**
         * <p>
         * The ID of the Maintenance Window execution that ran the task.
         * </p>
         * 
         * @param windowExecutionId
         *        The ID of the Maintenance Window execution that ran the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowExecutionId(String windowExecutionId);

        /**
         * <p>
         * The ID of the specific task execution in the Maintenance Window execution.
         * </p>
         * 
         * @param taskExecutionId
         *        The ID of the specific task execution in the Maintenance Window execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskExecutionId(String taskExecutionId);

        /**
         * <p>
         * The status of the task execution.
         * </p>
         * 
         * @param status
         *        The status of the task execution.
         * @see MaintenanceWindowExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the task execution.
         * </p>
         * 
         * @param status
         *        The status of the task execution.
         * @see MaintenanceWindowExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowExecutionStatus
         */
        Builder status(MaintenanceWindowExecutionStatus status);

        /**
         * <p>
         * The details explaining the status of the task execution. Only available for certain status values.
         * </p>
         * 
         * @param statusDetails
         *        The details explaining the status of the task execution. Only available for certain status values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(String statusDetails);

        /**
         * <p>
         * The time the task execution started.
         * </p>
         * 
         * @param startTime
         *        The time the task execution started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The time the task execution finished.
         * </p>
         * 
         * @param endTime
         *        The time the task execution finished.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The ARN of the executed task.
         * </p>
         * 
         * @param taskArn
         *        The ARN of the executed task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The type of executed task.
         * </p>
         * 
         * @param taskType
         *        The type of executed task.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder taskType(String taskType);

        /**
         * <p>
         * The type of executed task.
         * </p>
         * 
         * @param taskType
         *        The type of executed task.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder taskType(MaintenanceWindowTaskType taskType);
    }

    static final class BuilderImpl implements Builder {
        private String windowExecutionId;

        private String taskExecutionId;

        private String status;

        private String statusDetails;

        private Instant startTime;

        private Instant endTime;

        private String taskArn;

        private String taskType;

        private BuilderImpl() {
        }

        private BuilderImpl(MaintenanceWindowExecutionTaskIdentity model) {
            windowExecutionId(model.windowExecutionId);
            taskExecutionId(model.taskExecutionId);
            status(model.status);
            statusDetails(model.statusDetails);
            startTime(model.startTime);
            endTime(model.endTime);
            taskArn(model.taskArn);
            taskType(model.taskType);
        }

        public final String getWindowExecutionId() {
            return windowExecutionId;
        }

        @Override
        public final Builder windowExecutionId(String windowExecutionId) {
            this.windowExecutionId = windowExecutionId;
            return this;
        }

        public final void setWindowExecutionId(String windowExecutionId) {
            this.windowExecutionId = windowExecutionId;
        }

        public final String getTaskExecutionId() {
            return taskExecutionId;
        }

        @Override
        public final Builder taskExecutionId(String taskExecutionId) {
            this.taskExecutionId = taskExecutionId;
            return this;
        }

        public final void setTaskExecutionId(String taskExecutionId) {
            this.taskExecutionId = taskExecutionId;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MaintenanceWindowExecutionStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusDetails() {
            return statusDetails;
        }

        @Override
        public final Builder statusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final void setStatusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        public final String getTaskType() {
            return taskType;
        }

        @Override
        public final Builder taskType(String taskType) {
            this.taskType = taskType;
            return this;
        }

        @Override
        public final Builder taskType(MaintenanceWindowTaskType taskType) {
            this.taskType(taskType.toString());
            return this;
        }

        public final void setTaskType(String taskType) {
            this.taskType = taskType;
        }

        @Override
        public MaintenanceWindowExecutionTaskIdentity build() {
            return new MaintenanceWindowExecutionTaskIdentity(this);
        }
    }
}
