/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfig;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public class UpdateAssociationRequest extends SSMRequest implements
        ToCopyableBuilder<UpdateAssociationRequest.Builder, UpdateAssociationRequest> {
    private final String associationId;

    private final Map<String, List<String>> parameters;

    private final String documentVersion;

    private final String scheduleExpression;

    private final InstanceAssociationOutputLocation outputLocation;

    private final String name;

    private final List<Target> targets;

    private final String associationName;

    private final String associationVersion;

    private UpdateAssociationRequest(BuilderImpl builder) {
        super(builder);
        this.associationId = builder.associationId;
        this.parameters = builder.parameters;
        this.documentVersion = builder.documentVersion;
        this.scheduleExpression = builder.scheduleExpression;
        this.outputLocation = builder.outputLocation;
        this.name = builder.name;
        this.targets = builder.targets;
        this.associationName = builder.associationName;
        this.associationVersion = builder.associationVersion;
    }

    /**
     * <p>
     * The ID of the association you want to update.
     * </p>
     * 
     * @return The ID of the association you want to update.
     */
    public String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The parameters you want to update for the association. If you create a parameter using Parameter Store, you can
     * reference the parameter using {{ssm:parameter-name}}
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The parameters you want to update for the association. If you create a parameter using Parameter Store,
     *         you can reference the parameter using {{ssm:parameter-name}}
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The document version you want update for the association.
     * </p>
     * 
     * @return The document version you want update for the association.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The cron expression used to schedule the association that you want to update.
     * </p>
     * 
     * @return The cron expression used to schedule the association that you want to update.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * An Amazon S3 bucket where you want to store the results of this request.
     * </p>
     * 
     * @return An Amazon S3 bucket where you want to store the results of this request.
     */
    public InstanceAssociationOutputLocation outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * The name of the association document.
     * </p>
     * 
     * @return The name of the association document.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The targets of the association.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The targets of the association.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The name of the association that you want to update.
     * </p>
     * 
     * @return The name of the association that you want to update.
     */
    public String associationName() {
        return associationName;
    }

    /**
     * <p>
     * This parameter is provided for concurrency control purposes. You must specify the latest association version in
     * the service. If you want to ensure that this request succeeds, either specify <code>$LATEST</code>, or omit this
     * parameter.
     * </p>
     * 
     * @return This parameter is provided for concurrency control purposes. You must specify the latest association
     *         version in the service. If you want to ensure that this request succeeds, either specify
     *         <code>$LATEST</code>, or omit this parameter.
     */
    public String associationVersion() {
        return associationVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        hashCode = 31 * hashCode + Objects.hashCode(associationVersion());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateAssociationRequest)) {
            return false;
        }
        UpdateAssociationRequest other = (UpdateAssociationRequest) obj;
        return Objects.equals(associationId(), other.associationId()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(outputLocation(), other.outputLocation()) && Objects.equals(name(), other.name())
                && Objects.equals(targets(), other.targets()) && Objects.equals(associationName(), other.associationName())
                && Objects.equals(associationVersion(), other.associationVersion());
    }

    @Override
    public String toString() {
        return ToString.builder("UpdateAssociationRequest").add("AssociationId", associationId()).add("Parameters", parameters())
                .add("DocumentVersion", documentVersion()).add("ScheduleExpression", scheduleExpression())
                .add("OutputLocation", outputLocation()).add("Name", name()).add("Targets", targets())
                .add("AssociationName", associationName()).add("AssociationVersion", associationVersion()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociationId":
            return Optional.of(clazz.cast(associationId()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "DocumentVersion":
            return Optional.of(clazz.cast(documentVersion()));
        case "ScheduleExpression":
            return Optional.of(clazz.cast(scheduleExpression()));
        case "OutputLocation":
            return Optional.of(clazz.cast(outputLocation()));
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "Targets":
            return Optional.of(clazz.cast(targets()));
        case "AssociationName":
            return Optional.of(clazz.cast(associationName()));
        case "AssociationVersion":
            return Optional.of(clazz.cast(associationVersion()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SSMRequest.Builder, CopyableBuilder<Builder, UpdateAssociationRequest> {
        /**
         * <p>
         * The ID of the association you want to update.
         * </p>
         * 
         * @param associationId
         *        The ID of the association you want to update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The parameters you want to update for the association. If you create a parameter using Parameter Store, you
         * can reference the parameter using {{ssm:parameter-name}}
         * </p>
         * 
         * @param parameters
         *        The parameters you want to update for the association. If you create a parameter using Parameter
         *        Store, you can reference the parameter using {{ssm:parameter-name}}
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The document version you want update for the association.
         * </p>
         * 
         * @param documentVersion
         *        The document version you want update for the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The cron expression used to schedule the association that you want to update.
         * </p>
         * 
         * @param scheduleExpression
         *        The cron expression used to schedule the association that you want to update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the results of this request.
         * </p>
         * 
         * @param outputLocation
         *        An Amazon S3 bucket where you want to store the results of this request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(InstanceAssociationOutputLocation outputLocation);

        /**
         * <p>
         * An Amazon S3 bucket where you want to store the results of this request.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceAssociationOutputLocation.Builder}
         * avoiding the need to create one manually via {@link InstanceAssociationOutputLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceAssociationOutputLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #outputLocation(InstanceAssociationOutputLocation)}.
         * 
         * @param outputLocation
         *        a consumer that will call methods on {@link InstanceAssociationOutputLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputLocation(InstanceAssociationOutputLocation)
         */
        default Builder outputLocation(Consumer<InstanceAssociationOutputLocation.Builder> outputLocation) {
            return outputLocation(InstanceAssociationOutputLocation.builder().apply(outputLocation).build());
        }

        /**
         * <p>
         * The name of the association document.
         * </p>
         * 
         * @param name
         *        The name of the association document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The targets of the association.
         * </p>
         * 
         * @param targets
         *        The targets of the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets of the association.
         * </p>
         * 
         * @param targets
         *        The targets of the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets of the association.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The name of the association that you want to update.
         * </p>
         * 
         * @param associationName
         *        The name of the association that you want to update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);

        /**
         * <p>
         * This parameter is provided for concurrency control purposes. You must specify the latest association version
         * in the service. If you want to ensure that this request succeeds, either specify <code>$LATEST</code>, or
         * omit this parameter.
         * </p>
         * 
         * @param associationVersion
         *        This parameter is provided for concurrency control purposes. You must specify the latest association
         *        version in the service. If you want to ensure that this request succeeds, either specify
         *        <code>$LATEST</code>, or omit this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationVersion(String associationVersion);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);

        @Override
        Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SSMRequest.BuilderImpl implements Builder {
        private String associationId;

        private Map<String, List<String>> parameters;

        private String documentVersion;

        private String scheduleExpression;

        private InstanceAssociationOutputLocation outputLocation;

        private String name;

        private List<Target> targets;

        private String associationName;

        private String associationVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateAssociationRequest model) {
            super(model);
            associationId(model.associationId);
            parameters(model.parameters);
            documentVersion(model.documentVersion);
            scheduleExpression(model.scheduleExpression);
            outputLocation(model.outputLocation);
            name(model.name);
            targets(model.targets);
            associationName(model.associationName);
            associationVersion(model.associationVersion);
        }

        public final String getAssociationId() {
            return associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final InstanceAssociationOutputLocation.Builder getOutputLocation() {
            return outputLocation != null ? outputLocation.toBuilder() : null;
        }

        @Override
        public final Builder outputLocation(InstanceAssociationOutputLocation outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final void setOutputLocation(InstanceAssociationOutputLocation.BuilderImpl outputLocation) {
            this.outputLocation = outputLocation != null ? outputLocation.build() : null;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final String getAssociationName() {
            return associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        public final String getAssociationVersion() {
            return associationVersion;
        }

        @Override
        public final Builder associationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
            return this;
        }

        public final void setAssociationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public UpdateAssociationRequest build() {
            return new UpdateAssociationRequest(this);
        }
    }
}
