/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.InventoryAggregatorMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the inventory type and attribute for the aggregation execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class InventoryAggregator implements StructuredPojo, ToCopyableBuilder<InventoryAggregator.Builder, InventoryAggregator> {
    private final String expression;

    private final List<InventoryAggregator> aggregators;

    private InventoryAggregator(BuilderImpl builder) {
        this.expression = builder.expression;
        this.aggregators = builder.aggregators;
    }

    /**
     * <p>
     * The inventory type and attribute name for aggregation.
     * </p>
     * 
     * @return The inventory type and attribute name for aggregation.
     */
    public String expression() {
        return expression;
    }

    /**
     * <p>
     * Nested aggregators to further refine aggregation for an inventory type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Nested aggregators to further refine aggregation for an inventory type.
     */
    public List<InventoryAggregator> aggregators() {
        return aggregators;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expression());
        hashCode = 31 * hashCode + Objects.hashCode(aggregators());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InventoryAggregator)) {
            return false;
        }
        InventoryAggregator other = (InventoryAggregator) obj;
        return Objects.equals(expression(), other.expression()) && Objects.equals(aggregators(), other.aggregators());
    }

    @Override
    public String toString() {
        return ToString.builder("InventoryAggregator").add("Expression", expression()).add("Aggregators", aggregators()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Expression":
            return Optional.of(clazz.cast(expression()));
        case "Aggregators":
            return Optional.of(clazz.cast(aggregators()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        InventoryAggregatorMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, InventoryAggregator> {
        /**
         * <p>
         * The inventory type and attribute name for aggregation.
         * </p>
         * 
         * @param expression
         *        The inventory type and attribute name for aggregation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expression(String expression);

        /**
         * <p>
         * Nested aggregators to further refine aggregation for an inventory type.
         * </p>
         * 
         * @param aggregators
         *        Nested aggregators to further refine aggregation for an inventory type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aggregators(Collection<InventoryAggregator> aggregators);

        /**
         * <p>
         * Nested aggregators to further refine aggregation for an inventory type.
         * </p>
         * 
         * @param aggregators
         *        Nested aggregators to further refine aggregation for an inventory type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aggregators(InventoryAggregator... aggregators);

        /**
         * <p>
         * Nested aggregators to further refine aggregation for an inventory type.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InventoryAggregator>.Builder} avoiding the
         * need to create one manually via {@link List<InventoryAggregator>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InventoryAggregator>.Builder#build()} is called immediately
         * and its result is passed to {@link #aggregators(List<InventoryAggregator>)}.
         * 
         * @param aggregators
         *        a consumer that will call methods on {@link List<InventoryAggregator>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #aggregators(List<InventoryAggregator>)
         */
        Builder aggregators(Consumer<Builder>... aggregators);
    }

    static final class BuilderImpl implements Builder {
        private String expression;

        private List<InventoryAggregator> aggregators;

        private BuilderImpl() {
        }

        private BuilderImpl(InventoryAggregator model) {
            expression(model.expression);
            aggregators(model.aggregators);
        }

        public final String getExpression() {
            return expression;
        }

        @Override
        public final Builder expression(String expression) {
            this.expression = expression;
            return this;
        }

        public final void setExpression(String expression) {
            this.expression = expression;
        }

        public final Collection<Builder> getAggregators() {
            return aggregators != null ? aggregators.stream().map(InventoryAggregator::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder aggregators(Collection<InventoryAggregator> aggregators) {
            this.aggregators = InventoryAggregatorListCopier.copy(aggregators);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aggregators(InventoryAggregator... aggregators) {
            aggregators(Arrays.asList(aggregators));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aggregators(Consumer<Builder>... aggregators) {
            aggregators(Stream.of(aggregators).map(c -> InventoryAggregator.builder().apply(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAggregators(Collection<BuilderImpl> aggregators) {
            this.aggregators = InventoryAggregatorListCopier.copyFromBuilder(aggregators);
        }

        @Override
        public InventoryAggregator build() {
            return new InventoryAggregator(this);
        }
    }
}
