/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.InstancePatchStateMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the high-level patch compliance state for a managed instance, providing information about the number of
 * installed, missing, not applicable, and failed patches along with metadata about the operation when this information
 * was gathered for the instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class InstancePatchState implements StructuredPojo, ToCopyableBuilder<InstancePatchState.Builder, InstancePatchState> {
    private final String instanceId;

    private final String patchGroup;

    private final String baselineId;

    private final String snapshotId;

    private final String ownerInformation;

    private final Integer installedCount;

    private final Integer installedOtherCount;

    private final Integer missingCount;

    private final Integer failedCount;

    private final Integer notApplicableCount;

    private final Instant operationStartTime;

    private final Instant operationEndTime;

    private final String operation;

    private InstancePatchState(BuilderImpl builder) {
        this.instanceId = builder.instanceId;
        this.patchGroup = builder.patchGroup;
        this.baselineId = builder.baselineId;
        this.snapshotId = builder.snapshotId;
        this.ownerInformation = builder.ownerInformation;
        this.installedCount = builder.installedCount;
        this.installedOtherCount = builder.installedOtherCount;
        this.missingCount = builder.missingCount;
        this.failedCount = builder.failedCount;
        this.notApplicableCount = builder.notApplicableCount;
        this.operationStartTime = builder.operationStartTime;
        this.operationEndTime = builder.operationEndTime;
        this.operation = builder.operation;
    }

    /**
     * <p>
     * The ID of the managed instance the high-level patch compliance information was collected for.
     * </p>
     * 
     * @return The ID of the managed instance the high-level patch compliance information was collected for.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The name of the patch group the managed instance belongs to.
     * </p>
     * 
     * @return The name of the patch group the managed instance belongs to.
     */
    public String patchGroup() {
        return patchGroup;
    }

    /**
     * <p>
     * The ID of the patch baseline used to patch the instance.
     * </p>
     * 
     * @return The ID of the patch baseline used to patch the instance.
     */
    public String baselineId() {
        return baselineId;
    }

    /**
     * <p>
     * The ID of the patch baseline snapshot used during the patching operation when this compliance data was collected.
     * </p>
     * 
     * @return The ID of the patch baseline snapshot used during the patching operation when this compliance data was
     *         collected.
     */
    public String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * Placeholder information, this field will always be empty in the current release of the service.
     * </p>
     * 
     * @return Placeholder information, this field will always be empty in the current release of the service.
     */
    public String ownerInformation() {
        return ownerInformation;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that are installed on the instance.
     * </p>
     * 
     * @return The number of patches from the patch baseline that are installed on the instance.
     */
    public Integer installedCount() {
        return installedCount;
    }

    /**
     * <p>
     * The number of patches not specified in the patch baseline that are installed on the instance.
     * </p>
     * 
     * @return The number of patches not specified in the patch baseline that are installed on the instance.
     */
    public Integer installedOtherCount() {
        return installedOtherCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that are applicable for the instance but aren't currently
     * installed.
     * </p>
     * 
     * @return The number of patches from the patch baseline that are applicable for the instance but aren't currently
     *         installed.
     */
    public Integer missingCount() {
        return missingCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that were attempted to be installed during the last patching
     * operation, but failed to install.
     * </p>
     * 
     * @return The number of patches from the patch baseline that were attempted to be installed during the last
     *         patching operation, but failed to install.
     */
    public Integer failedCount() {
        return failedCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that aren't applicable for the instance and hence aren't installed
     * on the instance.
     * </p>
     * 
     * @return The number of patches from the patch baseline that aren't applicable for the instance and hence aren't
     *         installed on the instance.
     */
    public Integer notApplicableCount() {
        return notApplicableCount;
    }

    /**
     * <p>
     * The time the most recent patching operation was started on the instance.
     * </p>
     * 
     * @return The time the most recent patching operation was started on the instance.
     */
    public Instant operationStartTime() {
        return operationStartTime;
    }

    /**
     * <p>
     * The time the most recent patching operation completed on the instance.
     * </p>
     * 
     * @return The time the most recent patching operation completed on the instance.
     */
    public Instant operationEndTime() {
        return operationEndTime;
    }

    /**
     * <p>
     * The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL (install
     * missing patches).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link PatchOperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL
     *         (install missing patches).
     * @see PatchOperationType
     */
    public PatchOperationType operation() {
        return PatchOperationType.fromValue(operation);
    }

    /**
     * <p>
     * The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL (install
     * missing patches).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link PatchOperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL
     *         (install missing patches).
     * @see PatchOperationType
     */
    public String operationAsString() {
        return operation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(patchGroup());
        hashCode = 31 * hashCode + Objects.hashCode(baselineId());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(ownerInformation());
        hashCode = 31 * hashCode + Objects.hashCode(installedCount());
        hashCode = 31 * hashCode + Objects.hashCode(installedOtherCount());
        hashCode = 31 * hashCode + Objects.hashCode(missingCount());
        hashCode = 31 * hashCode + Objects.hashCode(failedCount());
        hashCode = 31 * hashCode + Objects.hashCode(notApplicableCount());
        hashCode = 31 * hashCode + Objects.hashCode(operationStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(operationEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(operationAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstancePatchState)) {
            return false;
        }
        InstancePatchState other = (InstancePatchState) obj;
        return Objects.equals(instanceId(), other.instanceId()) && Objects.equals(patchGroup(), other.patchGroup())
                && Objects.equals(baselineId(), other.baselineId()) && Objects.equals(snapshotId(), other.snapshotId())
                && Objects.equals(ownerInformation(), other.ownerInformation())
                && Objects.equals(installedCount(), other.installedCount())
                && Objects.equals(installedOtherCount(), other.installedOtherCount())
                && Objects.equals(missingCount(), other.missingCount()) && Objects.equals(failedCount(), other.failedCount())
                && Objects.equals(notApplicableCount(), other.notApplicableCount())
                && Objects.equals(operationStartTime(), other.operationStartTime())
                && Objects.equals(operationEndTime(), other.operationEndTime())
                && Objects.equals(operationAsString(), other.operationAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("InstancePatchState").add("InstanceId", instanceId()).add("PatchGroup", patchGroup())
                .add("BaselineId", baselineId()).add("SnapshotId", snapshotId()).add("OwnerInformation", ownerInformation())
                .add("InstalledCount", installedCount()).add("InstalledOtherCount", installedOtherCount())
                .add("MissingCount", missingCount()).add("FailedCount", failedCount())
                .add("NotApplicableCount", notApplicableCount()).add("OperationStartTime", operationStartTime())
                .add("OperationEndTime", operationEndTime()).add("Operation", operationAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceId":
            return Optional.of(clazz.cast(instanceId()));
        case "PatchGroup":
            return Optional.of(clazz.cast(patchGroup()));
        case "BaselineId":
            return Optional.of(clazz.cast(baselineId()));
        case "SnapshotId":
            return Optional.of(clazz.cast(snapshotId()));
        case "OwnerInformation":
            return Optional.of(clazz.cast(ownerInformation()));
        case "InstalledCount":
            return Optional.of(clazz.cast(installedCount()));
        case "InstalledOtherCount":
            return Optional.of(clazz.cast(installedOtherCount()));
        case "MissingCount":
            return Optional.of(clazz.cast(missingCount()));
        case "FailedCount":
            return Optional.of(clazz.cast(failedCount()));
        case "NotApplicableCount":
            return Optional.of(clazz.cast(notApplicableCount()));
        case "OperationStartTime":
            return Optional.of(clazz.cast(operationStartTime()));
        case "OperationEndTime":
            return Optional.of(clazz.cast(operationEndTime()));
        case "Operation":
            return Optional.of(clazz.cast(operationAsString()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        InstancePatchStateMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, InstancePatchState> {
        /**
         * <p>
         * The ID of the managed instance the high-level patch compliance information was collected for.
         * </p>
         * 
         * @param instanceId
         *        The ID of the managed instance the high-level patch compliance information was collected for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The name of the patch group the managed instance belongs to.
         * </p>
         * 
         * @param patchGroup
         *        The name of the patch group the managed instance belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patchGroup(String patchGroup);

        /**
         * <p>
         * The ID of the patch baseline used to patch the instance.
         * </p>
         * 
         * @param baselineId
         *        The ID of the patch baseline used to patch the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineId(String baselineId);

        /**
         * <p>
         * The ID of the patch baseline snapshot used during the patching operation when this compliance data was
         * collected.
         * </p>
         * 
         * @param snapshotId
         *        The ID of the patch baseline snapshot used during the patching operation when this compliance data was
         *        collected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * Placeholder information, this field will always be empty in the current release of the service.
         * </p>
         * 
         * @param ownerInformation
         *        Placeholder information, this field will always be empty in the current release of the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerInformation(String ownerInformation);

        /**
         * <p>
         * The number of patches from the patch baseline that are installed on the instance.
         * </p>
         * 
         * @param installedCount
         *        The number of patches from the patch baseline that are installed on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedCount(Integer installedCount);

        /**
         * <p>
         * The number of patches not specified in the patch baseline that are installed on the instance.
         * </p>
         * 
         * @param installedOtherCount
         *        The number of patches not specified in the patch baseline that are installed on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedOtherCount(Integer installedOtherCount);

        /**
         * <p>
         * The number of patches from the patch baseline that are applicable for the instance but aren't currently
         * installed.
         * </p>
         * 
         * @param missingCount
         *        The number of patches from the patch baseline that are applicable for the instance but aren't
         *        currently installed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder missingCount(Integer missingCount);

        /**
         * <p>
         * The number of patches from the patch baseline that were attempted to be installed during the last patching
         * operation, but failed to install.
         * </p>
         * 
         * @param failedCount
         *        The number of patches from the patch baseline that were attempted to be installed during the last
         *        patching operation, but failed to install.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedCount(Integer failedCount);

        /**
         * <p>
         * The number of patches from the patch baseline that aren't applicable for the instance and hence aren't
         * installed on the instance.
         * </p>
         * 
         * @param notApplicableCount
         *        The number of patches from the patch baseline that aren't applicable for the instance and hence aren't
         *        installed on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notApplicableCount(Integer notApplicableCount);

        /**
         * <p>
         * The time the most recent patching operation was started on the instance.
         * </p>
         * 
         * @param operationStartTime
         *        The time the most recent patching operation was started on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationStartTime(Instant operationStartTime);

        /**
         * <p>
         * The time the most recent patching operation completed on the instance.
         * </p>
         * 
         * @param operationEndTime
         *        The time the most recent patching operation completed on the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationEndTime(Instant operationEndTime);

        /**
         * <p>
         * The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL (install
         * missing patches).
         * </p>
         * 
         * @param operation
         *        The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL
         *        (install missing patches).
         * @see PatchOperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchOperationType
         */
        Builder operation(String operation);

        /**
         * <p>
         * The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL (install
         * missing patches).
         * </p>
         * 
         * @param operation
         *        The type of patching operation that was performed: SCAN (assess patch compliance state) or INSTALL
         *        (install missing patches).
         * @see PatchOperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchOperationType
         */
        Builder operation(PatchOperationType operation);
    }

    static final class BuilderImpl implements Builder {
        private String instanceId;

        private String patchGroup;

        private String baselineId;

        private String snapshotId;

        private String ownerInformation;

        private Integer installedCount;

        private Integer installedOtherCount;

        private Integer missingCount;

        private Integer failedCount;

        private Integer notApplicableCount;

        private Instant operationStartTime;

        private Instant operationEndTime;

        private String operation;

        private BuilderImpl() {
        }

        private BuilderImpl(InstancePatchState model) {
            instanceId(model.instanceId);
            patchGroup(model.patchGroup);
            baselineId(model.baselineId);
            snapshotId(model.snapshotId);
            ownerInformation(model.ownerInformation);
            installedCount(model.installedCount);
            installedOtherCount(model.installedOtherCount);
            missingCount(model.missingCount);
            failedCount(model.failedCount);
            notApplicableCount(model.notApplicableCount);
            operationStartTime(model.operationStartTime);
            operationEndTime(model.operationEndTime);
            operation(model.operation);
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getPatchGroup() {
            return patchGroup;
        }

        @Override
        public final Builder patchGroup(String patchGroup) {
            this.patchGroup = patchGroup;
            return this;
        }

        public final void setPatchGroup(String patchGroup) {
            this.patchGroup = patchGroup;
        }

        public final String getBaselineId() {
            return baselineId;
        }

        @Override
        public final Builder baselineId(String baselineId) {
            this.baselineId = baselineId;
            return this;
        }

        public final void setBaselineId(String baselineId) {
            this.baselineId = baselineId;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        public final String getOwnerInformation() {
            return ownerInformation;
        }

        @Override
        public final Builder ownerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
            return this;
        }

        public final void setOwnerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
        }

        public final Integer getInstalledCount() {
            return installedCount;
        }

        @Override
        public final Builder installedCount(Integer installedCount) {
            this.installedCount = installedCount;
            return this;
        }

        public final void setInstalledCount(Integer installedCount) {
            this.installedCount = installedCount;
        }

        public final Integer getInstalledOtherCount() {
            return installedOtherCount;
        }

        @Override
        public final Builder installedOtherCount(Integer installedOtherCount) {
            this.installedOtherCount = installedOtherCount;
            return this;
        }

        public final void setInstalledOtherCount(Integer installedOtherCount) {
            this.installedOtherCount = installedOtherCount;
        }

        public final Integer getMissingCount() {
            return missingCount;
        }

        @Override
        public final Builder missingCount(Integer missingCount) {
            this.missingCount = missingCount;
            return this;
        }

        public final void setMissingCount(Integer missingCount) {
            this.missingCount = missingCount;
        }

        public final Integer getFailedCount() {
            return failedCount;
        }

        @Override
        public final Builder failedCount(Integer failedCount) {
            this.failedCount = failedCount;
            return this;
        }

        public final void setFailedCount(Integer failedCount) {
            this.failedCount = failedCount;
        }

        public final Integer getNotApplicableCount() {
            return notApplicableCount;
        }

        @Override
        public final Builder notApplicableCount(Integer notApplicableCount) {
            this.notApplicableCount = notApplicableCount;
            return this;
        }

        public final void setNotApplicableCount(Integer notApplicableCount) {
            this.notApplicableCount = notApplicableCount;
        }

        public final Instant getOperationStartTime() {
            return operationStartTime;
        }

        @Override
        public final Builder operationStartTime(Instant operationStartTime) {
            this.operationStartTime = operationStartTime;
            return this;
        }

        public final void setOperationStartTime(Instant operationStartTime) {
            this.operationStartTime = operationStartTime;
        }

        public final Instant getOperationEndTime() {
            return operationEndTime;
        }

        @Override
        public final Builder operationEndTime(Instant operationEndTime) {
            this.operationEndTime = operationEndTime;
            return this;
        }

        public final void setOperationEndTime(Instant operationEndTime) {
            this.operationEndTime = operationEndTime;
        }

        public final String getOperation() {
            return operation;
        }

        @Override
        public final Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        @Override
        public final Builder operation(PatchOperationType operation) {
            this.operation(operation.toString());
            return this;
        }

        public final void setOperation(String operation) {
            this.operation = operation;
        }

        @Override
        public InstancePatchState build() {
            return new InstancePatchState(this);
        }
    }
}
