/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.ssm.transform.AssociationVersionInfoMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the association version.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class AssociationVersionInfo implements StructuredPojo,
        ToCopyableBuilder<AssociationVersionInfo.Builder, AssociationVersionInfo> {
    private final String associationId;

    private final String associationVersion;

    private final Instant createdDate;

    private final String name;

    private final String documentVersion;

    private final Map<String, List<String>> parameters;

    private final List<Target> targets;

    private final String scheduleExpression;

    private final InstanceAssociationOutputLocation outputLocation;

    private final String associationName;

    private AssociationVersionInfo(BuilderImpl builder) {
        this.associationId = builder.associationId;
        this.associationVersion = builder.associationVersion;
        this.createdDate = builder.createdDate;
        this.name = builder.name;
        this.documentVersion = builder.documentVersion;
        this.parameters = builder.parameters;
        this.targets = builder.targets;
        this.scheduleExpression = builder.scheduleExpression;
        this.outputLocation = builder.outputLocation;
        this.associationName = builder.associationName;
    }

    /**
     * <p>
     * The ID created by the system when the association was created.
     * </p>
     * 
     * @return The ID created by the system when the association was created.
     */
    public String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The association version.
     * </p>
     * 
     * @return The association version.
     */
    public String associationVersion() {
        return associationVersion;
    }

    /**
     * <p>
     * The date the association version was created.
     * </p>
     * 
     * @return The date the association version was created.
     */
    public Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * The name specified when the association was created.
     * </p>
     * 
     * @return The name specified when the association was created.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The version of a Systems Manager document used when the association version was created.
     * </p>
     * 
     * @return The version of a Systems Manager document used when the association version was created.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * Parameters specified when the association version was created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Parameters specified when the association version was created.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The targets specified for the association when the association version was created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The targets specified for the association when the association version was created.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The cron or rate schedule specified for the association when the association version was created.
     * </p>
     * 
     * @return The cron or rate schedule specified for the association when the association version was created.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * The location in Amazon S3 specified for the association when the association version was created.
     * </p>
     * 
     * @return The location in Amazon S3 specified for the association when the association version was created.
     */
    public InstanceAssociationOutputLocation outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * The name specified for the association version when the association version was created.
     * </p>
     * 
     * @return The name specified for the association version when the association version was created.
     */
    public String associationName() {
        return associationName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(associationVersion());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssociationVersionInfo)) {
            return false;
        }
        AssociationVersionInfo other = (AssociationVersionInfo) obj;
        return Objects.equals(associationId(), other.associationId())
                && Objects.equals(associationVersion(), other.associationVersion())
                && Objects.equals(createdDate(), other.createdDate()) && Objects.equals(name(), other.name())
                && Objects.equals(documentVersion(), other.documentVersion()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(targets(), other.targets()) && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(associationName(), other.associationName());
    }

    @Override
    public String toString() {
        return ToString.builder("AssociationVersionInfo").add("AssociationId", associationId())
                .add("AssociationVersion", associationVersion()).add("CreatedDate", createdDate()).add("Name", name())
                .add("DocumentVersion", documentVersion()).add("Parameters", parameters()).add("Targets", targets())
                .add("ScheduleExpression", scheduleExpression()).add("OutputLocation", outputLocation())
                .add("AssociationName", associationName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociationId":
            return Optional.of(clazz.cast(associationId()));
        case "AssociationVersion":
            return Optional.of(clazz.cast(associationVersion()));
        case "CreatedDate":
            return Optional.of(clazz.cast(createdDate()));
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "DocumentVersion":
            return Optional.of(clazz.cast(documentVersion()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "Targets":
            return Optional.of(clazz.cast(targets()));
        case "ScheduleExpression":
            return Optional.of(clazz.cast(scheduleExpression()));
        case "OutputLocation":
            return Optional.of(clazz.cast(outputLocation()));
        case "AssociationName":
            return Optional.of(clazz.cast(associationName()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AssociationVersionInfoMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AssociationVersionInfo> {
        /**
         * <p>
         * The ID created by the system when the association was created.
         * </p>
         * 
         * @param associationId
         *        The ID created by the system when the association was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The association version.
         * </p>
         * 
         * @param associationVersion
         *        The association version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationVersion(String associationVersion);

        /**
         * <p>
         * The date the association version was created.
         * </p>
         * 
         * @param createdDate
         *        The date the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * The name specified when the association was created.
         * </p>
         * 
         * @param name
         *        The name specified when the association was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The version of a Systems Manager document used when the association version was created.
         * </p>
         * 
         * @param documentVersion
         *        The version of a Systems Manager document used when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * Parameters specified when the association version was created.
         * </p>
         * 
         * @param parameters
         *        Parameters specified when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The targets specified for the association when the association version was created.
         * </p>
         * 
         * @param targets
         *        The targets specified for the association when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets specified for the association when the association version was created.
         * </p>
         * 
         * @param targets
         *        The targets specified for the association when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets specified for the association when the association version was created.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The cron or rate schedule specified for the association when the association version was created.
         * </p>
         * 
         * @param scheduleExpression
         *        The cron or rate schedule specified for the association when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * The location in Amazon S3 specified for the association when the association version was created.
         * </p>
         * 
         * @param outputLocation
         *        The location in Amazon S3 specified for the association when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(InstanceAssociationOutputLocation outputLocation);

        /**
         * <p>
         * The location in Amazon S3 specified for the association when the association version was created.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceAssociationOutputLocation.Builder}
         * avoiding the need to create one manually via {@link InstanceAssociationOutputLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceAssociationOutputLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #outputLocation(InstanceAssociationOutputLocation)}.
         * 
         * @param outputLocation
         *        a consumer that will call methods on {@link InstanceAssociationOutputLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputLocation(InstanceAssociationOutputLocation)
         */
        default Builder outputLocation(Consumer<InstanceAssociationOutputLocation.Builder> outputLocation) {
            return outputLocation(InstanceAssociationOutputLocation.builder().apply(outputLocation).build());
        }

        /**
         * <p>
         * The name specified for the association version when the association version was created.
         * </p>
         * 
         * @param associationName
         *        The name specified for the association version when the association version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);
    }

    static final class BuilderImpl implements Builder {
        private String associationId;

        private String associationVersion;

        private Instant createdDate;

        private String name;

        private String documentVersion;

        private Map<String, List<String>> parameters;

        private List<Target> targets;

        private String scheduleExpression;

        private InstanceAssociationOutputLocation outputLocation;

        private String associationName;

        private BuilderImpl() {
        }

        private BuilderImpl(AssociationVersionInfo model) {
            associationId(model.associationId);
            associationVersion(model.associationVersion);
            createdDate(model.createdDate);
            name(model.name);
            documentVersion(model.documentVersion);
            parameters(model.parameters);
            targets(model.targets);
            scheduleExpression(model.scheduleExpression);
            outputLocation(model.outputLocation);
            associationName(model.associationName);
        }

        public final String getAssociationId() {
            return associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        public final String getAssociationVersion() {
            return associationVersion;
        }

        @Override
        public final Builder associationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
            return this;
        }

        public final void setAssociationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final InstanceAssociationOutputLocation.Builder getOutputLocation() {
            return outputLocation != null ? outputLocation.toBuilder() : null;
        }

        @Override
        public final Builder outputLocation(InstanceAssociationOutputLocation outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final void setOutputLocation(InstanceAssociationOutputLocation.BuilderImpl outputLocation) {
            this.outputLocation = outputLocation != null ? outputLocation.build() : null;
        }

        public final String getAssociationName() {
            return associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        @Override
        public AssociationVersionInfo build() {
            return new AssociationVersionInfo(this);
        }
    }
}
