/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.runtime.SdkInternalList;
import software.amazon.awssdk.runtime.StandardMemberCopier;
import software.amazon.awssdk.services.ssm.transform.CommandMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a command request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Command implements StructuredPojo, ToCopyableBuilder<Command.Builder, Command> {
    private final String commandId;

    private final String documentName;

    private final String comment;

    private final Date expiresAfter;

    private final Map<String, List<String>> parameters;

    private final List<String> instanceIds;

    private final List<Target> targets;

    private final Date requestedDateTime;

    private final String status;

    private final String statusDetails;

    private final String outputS3Region;

    private final String outputS3BucketName;

    private final String outputS3KeyPrefix;

    private final String maxConcurrency;

    private final String maxErrors;

    private final Integer targetCount;

    private final Integer completedCount;

    private final Integer errorCount;

    private final String serviceRole;

    private final NotificationConfig notificationConfig;

    private Command(BuilderImpl builder) {
        this.commandId = builder.commandId;
        this.documentName = builder.documentName;
        this.comment = builder.comment;
        this.expiresAfter = builder.expiresAfter;
        this.parameters = builder.parameters;
        this.instanceIds = builder.instanceIds;
        this.targets = builder.targets;
        this.requestedDateTime = builder.requestedDateTime;
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.outputS3Region = builder.outputS3Region;
        this.outputS3BucketName = builder.outputS3BucketName;
        this.outputS3KeyPrefix = builder.outputS3KeyPrefix;
        this.maxConcurrency = builder.maxConcurrency;
        this.maxErrors = builder.maxErrors;
        this.targetCount = builder.targetCount;
        this.completedCount = builder.completedCount;
        this.errorCount = builder.errorCount;
        this.serviceRole = builder.serviceRole;
        this.notificationConfig = builder.notificationConfig;
    }

    /**
     * <p>
     * A unique identifier for this command.
     * </p>
     * 
     * @return A unique identifier for this command.
     */
    public String commandId() {
        return commandId;
    }

    /**
     * <p>
     * The name of the document requested for execution.
     * </p>
     * 
     * @return The name of the document requested for execution.
     */
    public String documentName() {
        return documentName;
    }

    /**
     * <p>
     * User-specified information about the command, such as a brief description of what the command should do.
     * </p>
     * 
     * @return User-specified information about the command, such as a brief description of what the command should do.
     */
    public String comment() {
        return comment;
    }

    /**
     * <p>
     * If this time is reached and the command has not already started executing, it will not execute. Calculated based
     * on the ExpiresAfter user input provided as part of the SendCommand API.
     * </p>
     * 
     * @return If this time is reached and the command has not already started executing, it will not execute.
     *         Calculated based on the ExpiresAfter user input provided as part of the SendCommand API.
     */
    public Date expiresAfter() {
        return expiresAfter;
    }

    /**
     * <p>
     * The parameter values to be inserted in the document when executing the command.
     * </p>
     * 
     * @return The parameter values to be inserted in the document when executing the command.
     */
    public Map<String, List<String>> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The instance IDs against which this command was requested.
     * </p>
     * 
     * @return The instance IDs against which this command was requested.
     */
    public List<String> instanceIds() {
        return instanceIds;
    }

    /**
     * <p>
     * An array of search criteria that targets instances using a Key,Value combination that you specify. Targets is
     * required if you don't provide one or more instance IDs in the call.
     * </p>
     * 
     * @return An array of search criteria that targets instances using a Key,Value combination that you specify.
     *         Targets is required if you don't provide one or more instance IDs in the call.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The date and time the command was requested.
     * </p>
     * 
     * @return The date and time the command was requested.
     */
    public Date requestedDateTime() {
        return requestedDateTime;
    }

    /**
     * <p>
     * The status of the command.
     * </p>
     * 
     * @return The status of the command.
     * @see CommandStatus
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * A detailed status of the command execution. StatusDetails includes more information than Status because it
     * includes states resulting from error and concurrency control parameters. StatusDetails can show different results
     * than Status. For more information about these statuses, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-about-status.html">Run Command
     * Status</a>. StatusDetails can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pending: The command has not been sent to any instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * In Progress: The command has been sent to at least one instance but has not reached a final state on all
     * instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Success: The command successfully executed on all invocations. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delivery Timed Out: The value of MaxErrors or more command invocations shows a status of Delivery Timed Out. This
     * is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Execution Timed Out: The value of MaxErrors or more command invocations shows a status of Execution Timed Out.
     * This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The value of MaxErrors or more command invocations shows a status of Failed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Incomplete: The command was attempted on all instances and one or more invocations does not have a value of
     * Success but not enough invocations failed for the status to be Failed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Canceled: The command was terminated before it was completed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Rate Exceeded: The number of instances targeted by the command exceeded the account limit for pending
     * invocations. The system has canceled the command before executing it on any instance. This is a terminal state.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A detailed status of the command execution. StatusDetails includes more information than Status because
     *         it includes states resulting from error and concurrency control parameters. StatusDetails can show
     *         different results than Status. For more information about these statuses, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-about-status.html">Run Command
     *         Status</a>. StatusDetails can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pending: The command has not been sent to any instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In Progress: The command has been sent to at least one instance but has not reached a final state on all
     *         instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Success: The command successfully executed on all invocations. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delivery Timed Out: The value of MaxErrors or more command invocations shows a status of Delivery Timed
     *         Out. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Execution Timed Out: The value of MaxErrors or more command invocations shows a status of Execution Timed
     *         Out. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The value of MaxErrors or more command invocations shows a status of Failed. This is a terminal
     *         state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Incomplete: The command was attempted on all instances and one or more invocations does not have a value
     *         of Success but not enough invocations failed for the status to be Failed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Canceled: The command was terminated before it was completed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Rate Exceeded: The number of instances targeted by the command exceeded the account limit for pending
     *         invocations. The system has canceled the command before executing it on any instance. This is a terminal
     *         state.
     *         </p>
     *         </li>
     */
    public String statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
     * automatically determines the Amazon S3 bucket region.
     * </p>
     * 
     * @return (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
     *         automatically determines the Amazon S3 bucket region.
     */
    public String outputS3Region() {
        return outputS3Region;
    }

    /**
     * <p>
     * The S3 bucket where the responses to the command executions should be stored. This was requested when issuing the
     * command.
     * </p>
     * 
     * @return The S3 bucket where the responses to the command executions should be stored. This was requested when
     *         issuing the command.
     */
    public String outputS3BucketName() {
        return outputS3BucketName;
    }

    /**
     * <p>
     * The S3 directory path inside the bucket where the responses to the command executions should be stored. This was
     * requested when issuing the command.
     * </p>
     * 
     * @return The S3 directory path inside the bucket where the responses to the command executions should be stored.
     *         This was requested when issuing the command.
     */
    public String outputS3KeyPrefix() {
        return outputS3KeyPrefix;
    }

    /**
     * <p>
     * The maximum number of instances that are allowed to execute the command at the same time. You can specify a
     * number of instances, such as 10, or a percentage of instances, such as 10%. The default value is 50. For more
     * information about how to use MaxConcurrency, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command Using
     * Systems Manager Run Command</a>.
     * </p>
     * 
     * @return The maximum number of instances that are allowed to execute the command at the same time. You can specify
     *         a number of instances, such as 10, or a percentage of instances, such as 10%. The default value is 50.
     *         For more information about how to use MaxConcurrency, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command
     *         Using Systems Manager Run Command</a>.
     */
    public String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The maximum number of errors allowed before the system stops sending the command to additional targets. You can
     * specify a number of errors, such as 10, or a percentage or errors, such as 10%. The default value is 50. For more
     * information about how to use MaxErrors, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command Using
     * Systems Manager Run Command</a>.
     * </p>
     * 
     * @return The maximum number of errors allowed before the system stops sending the command to additional targets.
     *         You can specify a number of errors, such as 10, or a percentage or errors, such as 10%. The default value
     *         is 50. For more information about how to use MaxErrors, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command
     *         Using Systems Manager Run Command</a>.
     */
    public String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * The number of targets for the command.
     * </p>
     * 
     * @return The number of targets for the command.
     */
    public Integer targetCount() {
        return targetCount;
    }

    /**
     * <p>
     * The number of targets for which the command invocation reached a terminal state. Terminal states include the
     * following: Success, Failed, Execution Timed Out, Delivery Timed Out, Canceled, Terminated, or Undeliverable.
     * </p>
     * 
     * @return The number of targets for which the command invocation reached a terminal state. Terminal states include
     *         the following: Success, Failed, Execution Timed Out, Delivery Timed Out, Canceled, Terminated, or
     *         Undeliverable.
     */
    public Integer completedCount() {
        return completedCount;
    }

    /**
     * <p>
     * The number of targets for which the status is Failed or Execution Timed Out.
     * </p>
     * 
     * @return The number of targets for which the status is Failed or Execution Timed Out.
     */
    public Integer errorCount() {
        return errorCount;
    }

    /**
     * <p>
     * The IAM service role that Run Command uses to act on your behalf when sending notifications about command status
     * changes.
     * </p>
     * 
     * @return The IAM service role that Run Command uses to act on your behalf when sending notifications about command
     *         status changes.
     */
    public String serviceRole() {
        return serviceRole;
    }

    /**
     * <p>
     * Configurations for sending notifications about command status changes.
     * </p>
     * 
     * @return Configurations for sending notifications about command status changes.
     */
    public NotificationConfig notificationConfig() {
        return notificationConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((commandId() == null) ? 0 : commandId().hashCode());
        hashCode = 31 * hashCode + ((documentName() == null) ? 0 : documentName().hashCode());
        hashCode = 31 * hashCode + ((comment() == null) ? 0 : comment().hashCode());
        hashCode = 31 * hashCode + ((expiresAfter() == null) ? 0 : expiresAfter().hashCode());
        hashCode = 31 * hashCode + ((parameters() == null) ? 0 : parameters().hashCode());
        hashCode = 31 * hashCode + ((instanceIds() == null) ? 0 : instanceIds().hashCode());
        hashCode = 31 * hashCode + ((targets() == null) ? 0 : targets().hashCode());
        hashCode = 31 * hashCode + ((requestedDateTime() == null) ? 0 : requestedDateTime().hashCode());
        hashCode = 31 * hashCode + ((status() == null) ? 0 : status().hashCode());
        hashCode = 31 * hashCode + ((statusDetails() == null) ? 0 : statusDetails().hashCode());
        hashCode = 31 * hashCode + ((outputS3Region() == null) ? 0 : outputS3Region().hashCode());
        hashCode = 31 * hashCode + ((outputS3BucketName() == null) ? 0 : outputS3BucketName().hashCode());
        hashCode = 31 * hashCode + ((outputS3KeyPrefix() == null) ? 0 : outputS3KeyPrefix().hashCode());
        hashCode = 31 * hashCode + ((maxConcurrency() == null) ? 0 : maxConcurrency().hashCode());
        hashCode = 31 * hashCode + ((maxErrors() == null) ? 0 : maxErrors().hashCode());
        hashCode = 31 * hashCode + ((targetCount() == null) ? 0 : targetCount().hashCode());
        hashCode = 31 * hashCode + ((completedCount() == null) ? 0 : completedCount().hashCode());
        hashCode = 31 * hashCode + ((errorCount() == null) ? 0 : errorCount().hashCode());
        hashCode = 31 * hashCode + ((serviceRole() == null) ? 0 : serviceRole().hashCode());
        hashCode = 31 * hashCode + ((notificationConfig() == null) ? 0 : notificationConfig().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Command)) {
            return false;
        }
        Command other = (Command) obj;
        if (other.commandId() == null ^ this.commandId() == null) {
            return false;
        }
        if (other.commandId() != null && !other.commandId().equals(this.commandId())) {
            return false;
        }
        if (other.documentName() == null ^ this.documentName() == null) {
            return false;
        }
        if (other.documentName() != null && !other.documentName().equals(this.documentName())) {
            return false;
        }
        if (other.comment() == null ^ this.comment() == null) {
            return false;
        }
        if (other.comment() != null && !other.comment().equals(this.comment())) {
            return false;
        }
        if (other.expiresAfter() == null ^ this.expiresAfter() == null) {
            return false;
        }
        if (other.expiresAfter() != null && !other.expiresAfter().equals(this.expiresAfter())) {
            return false;
        }
        if (other.parameters() == null ^ this.parameters() == null) {
            return false;
        }
        if (other.parameters() != null && !other.parameters().equals(this.parameters())) {
            return false;
        }
        if (other.instanceIds() == null ^ this.instanceIds() == null) {
            return false;
        }
        if (other.instanceIds() != null && !other.instanceIds().equals(this.instanceIds())) {
            return false;
        }
        if (other.targets() == null ^ this.targets() == null) {
            return false;
        }
        if (other.targets() != null && !other.targets().equals(this.targets())) {
            return false;
        }
        if (other.requestedDateTime() == null ^ this.requestedDateTime() == null) {
            return false;
        }
        if (other.requestedDateTime() != null && !other.requestedDateTime().equals(this.requestedDateTime())) {
            return false;
        }
        if (other.status() == null ^ this.status() == null) {
            return false;
        }
        if (other.status() != null && !other.status().equals(this.status())) {
            return false;
        }
        if (other.statusDetails() == null ^ this.statusDetails() == null) {
            return false;
        }
        if (other.statusDetails() != null && !other.statusDetails().equals(this.statusDetails())) {
            return false;
        }
        if (other.outputS3Region() == null ^ this.outputS3Region() == null) {
            return false;
        }
        if (other.outputS3Region() != null && !other.outputS3Region().equals(this.outputS3Region())) {
            return false;
        }
        if (other.outputS3BucketName() == null ^ this.outputS3BucketName() == null) {
            return false;
        }
        if (other.outputS3BucketName() != null && !other.outputS3BucketName().equals(this.outputS3BucketName())) {
            return false;
        }
        if (other.outputS3KeyPrefix() == null ^ this.outputS3KeyPrefix() == null) {
            return false;
        }
        if (other.outputS3KeyPrefix() != null && !other.outputS3KeyPrefix().equals(this.outputS3KeyPrefix())) {
            return false;
        }
        if (other.maxConcurrency() == null ^ this.maxConcurrency() == null) {
            return false;
        }
        if (other.maxConcurrency() != null && !other.maxConcurrency().equals(this.maxConcurrency())) {
            return false;
        }
        if (other.maxErrors() == null ^ this.maxErrors() == null) {
            return false;
        }
        if (other.maxErrors() != null && !other.maxErrors().equals(this.maxErrors())) {
            return false;
        }
        if (other.targetCount() == null ^ this.targetCount() == null) {
            return false;
        }
        if (other.targetCount() != null && !other.targetCount().equals(this.targetCount())) {
            return false;
        }
        if (other.completedCount() == null ^ this.completedCount() == null) {
            return false;
        }
        if (other.completedCount() != null && !other.completedCount().equals(this.completedCount())) {
            return false;
        }
        if (other.errorCount() == null ^ this.errorCount() == null) {
            return false;
        }
        if (other.errorCount() != null && !other.errorCount().equals(this.errorCount())) {
            return false;
        }
        if (other.serviceRole() == null ^ this.serviceRole() == null) {
            return false;
        }
        if (other.serviceRole() != null && !other.serviceRole().equals(this.serviceRole())) {
            return false;
        }
        if (other.notificationConfig() == null ^ this.notificationConfig() == null) {
            return false;
        }
        if (other.notificationConfig() != null && !other.notificationConfig().equals(this.notificationConfig())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (commandId() != null) {
            sb.append("CommandId: ").append(commandId()).append(",");
        }
        if (documentName() != null) {
            sb.append("DocumentName: ").append(documentName()).append(",");
        }
        if (comment() != null) {
            sb.append("Comment: ").append(comment()).append(",");
        }
        if (expiresAfter() != null) {
            sb.append("ExpiresAfter: ").append(expiresAfter()).append(",");
        }
        if (parameters() != null) {
            sb.append("Parameters: ").append(parameters()).append(",");
        }
        if (instanceIds() != null) {
            sb.append("InstanceIds: ").append(instanceIds()).append(",");
        }
        if (targets() != null) {
            sb.append("Targets: ").append(targets()).append(",");
        }
        if (requestedDateTime() != null) {
            sb.append("RequestedDateTime: ").append(requestedDateTime()).append(",");
        }
        if (status() != null) {
            sb.append("Status: ").append(status()).append(",");
        }
        if (statusDetails() != null) {
            sb.append("StatusDetails: ").append(statusDetails()).append(",");
        }
        if (outputS3Region() != null) {
            sb.append("OutputS3Region: ").append(outputS3Region()).append(",");
        }
        if (outputS3BucketName() != null) {
            sb.append("OutputS3BucketName: ").append(outputS3BucketName()).append(",");
        }
        if (outputS3KeyPrefix() != null) {
            sb.append("OutputS3KeyPrefix: ").append(outputS3KeyPrefix()).append(",");
        }
        if (maxConcurrency() != null) {
            sb.append("MaxConcurrency: ").append(maxConcurrency()).append(",");
        }
        if (maxErrors() != null) {
            sb.append("MaxErrors: ").append(maxErrors()).append(",");
        }
        if (targetCount() != null) {
            sb.append("TargetCount: ").append(targetCount()).append(",");
        }
        if (completedCount() != null) {
            sb.append("CompletedCount: ").append(completedCount()).append(",");
        }
        if (errorCount() != null) {
            sb.append("ErrorCount: ").append(errorCount()).append(",");
        }
        if (serviceRole() != null) {
            sb.append("ServiceRole: ").append(serviceRole()).append(",");
        }
        if (notificationConfig() != null) {
            sb.append("NotificationConfig: ").append(notificationConfig()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        CommandMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Command> {
        /**
         * <p>
         * A unique identifier for this command.
         * </p>
         * 
         * @param commandId
         *        A unique identifier for this command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commandId(String commandId);

        /**
         * <p>
         * The name of the document requested for execution.
         * </p>
         * 
         * @param documentName
         *        The name of the document requested for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * User-specified information about the command, such as a brief description of what the command should do.
         * </p>
         * 
         * @param comment
         *        User-specified information about the command, such as a brief description of what the command should
         *        do.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * If this time is reached and the command has not already started executing, it will not execute. Calculated
         * based on the ExpiresAfter user input provided as part of the SendCommand API.
         * </p>
         * 
         * @param expiresAfter
         *        If this time is reached and the command has not already started executing, it will not execute.
         *        Calculated based on the ExpiresAfter user input provided as part of the SendCommand API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiresAfter(Date expiresAfter);

        /**
         * <p>
         * The parameter values to be inserted in the document when executing the command.
         * </p>
         * 
         * @param parameters
         *        The parameter values to be inserted in the document when executing the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        /**
         * <p>
         * The instance IDs against which this command was requested.
         * </p>
         * 
         * @param instanceIds
         *        The instance IDs against which this command was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceIds(Collection<String> instanceIds);

        /**
         * <p>
         * The instance IDs against which this command was requested.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setInstanceIds(java.util.Collection)} or {@link #withInstanceIds(java.util.Collection)} if you want
         * to override the existing values.
         * </p>
         * 
         * @param instanceIds
         *        The instance IDs against which this command was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceIds(String... instanceIds);

        /**
         * <p>
         * An array of search criteria that targets instances using a Key,Value combination that you specify. Targets is
         * required if you don't provide one or more instance IDs in the call.
         * </p>
         * 
         * @param targets
         *        An array of search criteria that targets instances using a Key,Value combination that you specify.
         *        Targets is required if you don't provide one or more instance IDs in the call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * An array of search criteria that targets instances using a Key,Value combination that you specify. Targets is
         * required if you don't provide one or more instance IDs in the call.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setTargets(java.util.Collection)} or {@link #withTargets(java.util.Collection)} if you want to
         * override the existing values.
         * </p>
         * 
         * @param targets
         *        An array of search criteria that targets instances using a Key,Value combination that you specify.
         *        Targets is required if you don't provide one or more instance IDs in the call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The date and time the command was requested.
         * </p>
         * 
         * @param requestedDateTime
         *        The date and time the command was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedDateTime(Date requestedDateTime);

        /**
         * <p>
         * The status of the command.
         * </p>
         * 
         * @param status
         *        The status of the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the command.
         * </p>
         * 
         * @param status
         *        The status of the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandStatus
         */
        Builder status(CommandStatus status);

        /**
         * <p>
         * A detailed status of the command execution. StatusDetails includes more information than Status because it
         * includes states resulting from error and concurrency control parameters. StatusDetails can show different
         * results than Status. For more information about these statuses, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-about-status.html">Run Command
         * Status</a>. StatusDetails can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pending: The command has not been sent to any instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * In Progress: The command has been sent to at least one instance but has not reached a final state on all
         * instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Success: The command successfully executed on all invocations. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Delivery Timed Out: The value of MaxErrors or more command invocations shows a status of Delivery Timed Out.
         * This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Execution Timed Out: The value of MaxErrors or more command invocations shows a status of Execution Timed
         * Out. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The value of MaxErrors or more command invocations shows a status of Failed. This is a terminal
         * state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Incomplete: The command was attempted on all instances and one or more invocations does not have a value of
         * Success but not enough invocations failed for the status to be Failed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Canceled: The command was terminated before it was completed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Rate Exceeded: The number of instances targeted by the command exceeded the account limit for pending
         * invocations. The system has canceled the command before executing it on any instance. This is a terminal
         * state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusDetails
         *        A detailed status of the command execution. StatusDetails includes more information than Status
         *        because it includes states resulting from error and concurrency control parameters. StatusDetails can
         *        show different results than Status. For more information about these statuses, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-about-status.html">Run
         *        Command Status</a>. StatusDetails can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pending: The command has not been sent to any instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In Progress: The command has been sent to at least one instance but has not reached a final state on
         *        all instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Success: The command successfully executed on all invocations. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delivery Timed Out: The value of MaxErrors or more command invocations shows a status of Delivery
         *        Timed Out. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Execution Timed Out: The value of MaxErrors or more command invocations shows a status of Execution
         *        Timed Out. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The value of MaxErrors or more command invocations shows a status of Failed. This is a
         *        terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Incomplete: The command was attempted on all instances and one or more invocations does not have a
         *        value of Success but not enough invocations failed for the status to be Failed. This is a terminal
         *        state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Canceled: The command was terminated before it was completed. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Rate Exceeded: The number of instances targeted by the command exceeded the account limit for pending
         *        invocations. The system has canceled the command before executing it on any instance. This is a
         *        terminal state.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(String statusDetails);

        /**
         * <p>
         * (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
         * automatically determines the Amazon S3 bucket region.
         * </p>
         * 
         * @param outputS3Region
         *        (Deprecated) You can no longer specify this parameter. The system ignores it. Instead, Systems Manager
         *        automatically determines the Amazon S3 bucket region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3Region(String outputS3Region);

        /**
         * <p>
         * The S3 bucket where the responses to the command executions should be stored. This was requested when issuing
         * the command.
         * </p>
         * 
         * @param outputS3BucketName
         *        The S3 bucket where the responses to the command executions should be stored. This was requested when
         *        issuing the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3BucketName(String outputS3BucketName);

        /**
         * <p>
         * The S3 directory path inside the bucket where the responses to the command executions should be stored. This
         * was requested when issuing the command.
         * </p>
         * 
         * @param outputS3KeyPrefix
         *        The S3 directory path inside the bucket where the responses to the command executions should be
         *        stored. This was requested when issuing the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputS3KeyPrefix(String outputS3KeyPrefix);

        /**
         * <p>
         * The maximum number of instances that are allowed to execute the command at the same time. You can specify a
         * number of instances, such as 10, or a percentage of instances, such as 10%. The default value is 50. For more
         * information about how to use MaxConcurrency, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command Using
         * Systems Manager Run Command</a>.
         * </p>
         * 
         * @param maxConcurrency
         *        The maximum number of instances that are allowed to execute the command at the same time. You can
         *        specify a number of instances, such as 10, or a percentage of instances, such as 10%. The default
         *        value is 50. For more information about how to use MaxConcurrency, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a
         *        Command Using Systems Manager Run Command</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The maximum number of errors allowed before the system stops sending the command to additional targets. You
         * can specify a number of errors, such as 10, or a percentage or errors, such as 10%. The default value is 50.
         * For more information about how to use MaxErrors, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a Command Using
         * Systems Manager Run Command</a>.
         * </p>
         * 
         * @param maxErrors
         *        The maximum number of errors allowed before the system stops sending the command to additional
         *        targets. You can specify a number of errors, such as 10, or a percentage or errors, such as 10%. The
         *        default value is 50. For more information about how to use MaxErrors, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/run-command.html">Executing a
         *        Command Using Systems Manager Run Command</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * The number of targets for the command.
         * </p>
         * 
         * @param targetCount
         *        The number of targets for the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetCount(Integer targetCount);

        /**
         * <p>
         * The number of targets for which the command invocation reached a terminal state. Terminal states include the
         * following: Success, Failed, Execution Timed Out, Delivery Timed Out, Canceled, Terminated, or Undeliverable.
         * </p>
         * 
         * @param completedCount
         *        The number of targets for which the command invocation reached a terminal state. Terminal states
         *        include the following: Success, Failed, Execution Timed Out, Delivery Timed Out, Canceled, Terminated,
         *        or Undeliverable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedCount(Integer completedCount);

        /**
         * <p>
         * The number of targets for which the status is Failed or Execution Timed Out.
         * </p>
         * 
         * @param errorCount
         *        The number of targets for which the status is Failed or Execution Timed Out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorCount(Integer errorCount);

        /**
         * <p>
         * The IAM service role that Run Command uses to act on your behalf when sending notifications about command
         * status changes.
         * </p>
         * 
         * @param serviceRole
         *        The IAM service role that Run Command uses to act on your behalf when sending notifications about
         *        command status changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * Configurations for sending notifications about command status changes.
         * </p>
         * 
         * @param notificationConfig
         *        Configurations for sending notifications about command status changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationConfig(NotificationConfig notificationConfig);
    }

    private static final class BuilderImpl implements Builder {
        private String commandId;

        private String documentName;

        private String comment;

        private Date expiresAfter;

        private Map<String, List<String>> parameters;

        private List<String> instanceIds = new SdkInternalList<>();

        private List<Target> targets = new SdkInternalList<>();

        private Date requestedDateTime;

        private String status;

        private String statusDetails;

        private String outputS3Region;

        private String outputS3BucketName;

        private String outputS3KeyPrefix;

        private String maxConcurrency;

        private String maxErrors;

        private Integer targetCount;

        private Integer completedCount;

        private Integer errorCount;

        private String serviceRole;

        private NotificationConfig notificationConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Command model) {
            setCommandId(model.commandId);
            setDocumentName(model.documentName);
            setComment(model.comment);
            setExpiresAfter(model.expiresAfter);
            setParameters(model.parameters);
            setInstanceIds(model.instanceIds);
            setTargets(model.targets);
            setRequestedDateTime(model.requestedDateTime);
            setStatus(model.status);
            setStatusDetails(model.statusDetails);
            setOutputS3Region(model.outputS3Region);
            setOutputS3BucketName(model.outputS3BucketName);
            setOutputS3KeyPrefix(model.outputS3KeyPrefix);
            setMaxConcurrency(model.maxConcurrency);
            setMaxErrors(model.maxErrors);
            setTargetCount(model.targetCount);
            setCompletedCount(model.completedCount);
            setErrorCount(model.errorCount);
            setServiceRole(model.serviceRole);
            setNotificationConfig(model.notificationConfig);
        }

        public final String getCommandId() {
            return commandId;
        }

        @Override
        public final Builder commandId(String commandId) {
            this.commandId = commandId;
            return this;
        }

        public final void setCommandId(String commandId) {
            this.commandId = commandId;
        }

        public final String getDocumentName() {
            return documentName;
        }

        @Override
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        public final String getComment() {
            return comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        public final Date getExpiresAfter() {
            return expiresAfter;
        }

        @Override
        public final Builder expiresAfter(Date expiresAfter) {
            this.expiresAfter = StandardMemberCopier.copy(expiresAfter);
            return this;
        }

        public final void setExpiresAfter(Date expiresAfter) {
            this.expiresAfter = StandardMemberCopier.copy(expiresAfter);
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
        }

        public final Collection<String> getInstanceIds() {
            return instanceIds;
        }

        @Override
        public final Builder instanceIds(Collection<String> instanceIds) {
            this.instanceIds = InstanceIdListCopier.copy(instanceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceIds(String... instanceIds) {
            if (this.instanceIds == null) {
                this.instanceIds = new SdkInternalList<>(instanceIds.length);
            }
            for (String e : instanceIds) {
                this.instanceIds.add(e);
            }
            return this;
        }

        public final void setInstanceIds(Collection<String> instanceIds) {
            this.instanceIds = InstanceIdListCopier.copy(instanceIds);
        }

        @SafeVarargs
        public final void setInstanceIds(String... instanceIds) {
            if (this.instanceIds == null) {
                this.instanceIds = new SdkInternalList<>(instanceIds.length);
            }
            for (String e : instanceIds) {
                this.instanceIds.add(e);
            }
        }

        public final Collection<Target> getTargets() {
            return targets;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            if (this.targets == null) {
                this.targets = new SdkInternalList<>(targets.length);
            }
            for (Target e : targets) {
                this.targets.add(e);
            }
            return this;
        }

        public final void setTargets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
        }

        @SafeVarargs
        public final void setTargets(Target... targets) {
            if (this.targets == null) {
                this.targets = new SdkInternalList<>(targets.length);
            }
            for (Target e : targets) {
                this.targets.add(e);
            }
        }

        public final Date getRequestedDateTime() {
            return requestedDateTime;
        }

        @Override
        public final Builder requestedDateTime(Date requestedDateTime) {
            this.requestedDateTime = StandardMemberCopier.copy(requestedDateTime);
            return this;
        }

        public final void setRequestedDateTime(Date requestedDateTime) {
            this.requestedDateTime = StandardMemberCopier.copy(requestedDateTime);
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CommandStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final void setStatus(CommandStatus status) {
            this.status(status.toString());
        }

        public final String getStatusDetails() {
            return statusDetails;
        }

        @Override
        public final Builder statusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final void setStatusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
        }

        public final String getOutputS3Region() {
            return outputS3Region;
        }

        @Override
        public final Builder outputS3Region(String outputS3Region) {
            this.outputS3Region = outputS3Region;
            return this;
        }

        public final void setOutputS3Region(String outputS3Region) {
            this.outputS3Region = outputS3Region;
        }

        public final String getOutputS3BucketName() {
            return outputS3BucketName;
        }

        @Override
        public final Builder outputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
            return this;
        }

        public final void setOutputS3BucketName(String outputS3BucketName) {
            this.outputS3BucketName = outputS3BucketName;
        }

        public final String getOutputS3KeyPrefix() {
            return outputS3KeyPrefix;
        }

        @Override
        public final Builder outputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
            return this;
        }

        public final void setOutputS3KeyPrefix(String outputS3KeyPrefix) {
            this.outputS3KeyPrefix = outputS3KeyPrefix;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        @Override
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        public final Integer getTargetCount() {
            return targetCount;
        }

        @Override
        public final Builder targetCount(Integer targetCount) {
            this.targetCount = targetCount;
            return this;
        }

        public final void setTargetCount(Integer targetCount) {
            this.targetCount = targetCount;
        }

        public final Integer getCompletedCount() {
            return completedCount;
        }

        @Override
        public final Builder completedCount(Integer completedCount) {
            this.completedCount = completedCount;
            return this;
        }

        public final void setCompletedCount(Integer completedCount) {
            this.completedCount = completedCount;
        }

        public final Integer getErrorCount() {
            return errorCount;
        }

        @Override
        public final Builder errorCount(Integer errorCount) {
            this.errorCount = errorCount;
            return this;
        }

        public final void setErrorCount(Integer errorCount) {
            this.errorCount = errorCount;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        public final NotificationConfig getNotificationConfig() {
            return notificationConfig;
        }

        @Override
        public final Builder notificationConfig(NotificationConfig notificationConfig) {
            this.notificationConfig = notificationConfig;
            return this;
        }

        public final void setNotificationConfig(NotificationConfig notificationConfig) {
            this.notificationConfig = notificationConfig;
        }

        @Override
        public Command build() {
            return new Command(this);
        }
    }
}
