/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Amazon Simple Notification Service (Amazon SNS) notification settings associated with a specific job. The
 * <code>Notification</code> object is returned as a part of the response syntax of the <code>DescribeJob</code> action
 * in the <code>JobMetadata</code> data type.
 * </p>
 * <p>
 * When the notification settings are defined during job creation, you can choose to notify based on a specific set of
 * job states using the <code>JobStatesToNotify</code> array of strings, or you can specify that you want to have Amazon
 * SNS notifications sent out for all job states with <code>NotifyAll</code> set to true.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Notification implements SdkPojo, Serializable, ToCopyableBuilder<Notification.Builder, Notification> {
    private static final SdkField<String> SNS_TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnsTopicARN").getter(getter(Notification::snsTopicARN)).setter(setter(Builder::snsTopicARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsTopicARN").build()).build();

    private static final SdkField<List<String>> JOB_STATES_TO_NOTIFY_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("JobStatesToNotify")
            .getter(getter(Notification::jobStatesToNotifyAsStrings))
            .setter(setter(Builder::jobStatesToNotifyWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobStatesToNotify").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DEVICE_PICKUP_SNS_TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DevicePickupSnsTopicARN").getter(getter(Notification::devicePickupSnsTopicARN))
            .setter(setter(Builder::devicePickupSnsTopicARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DevicePickupSnsTopicARN").build())
            .build();

    private static final SdkField<Boolean> NOTIFY_ALL_JOB_STATES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("NotifyAllJobStates").getter(getter(Notification::notifyAllJobStates))
            .setter(setter(Builder::notifyAllJobStates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotifyAll").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SNS_TOPIC_ARN_FIELD,
            JOB_STATES_TO_NOTIFY_FIELD, DEVICE_PICKUP_SNS_TOPIC_ARN_FIELD, NOTIFY_ALL_JOB_STATES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String snsTopicARN;

    private final List<String> jobStatesToNotify;

    private final String devicePickupSnsTopicARN;

    private final Boolean notifyAllJobStates;

    private Notification(BuilderImpl builder) {
        this.snsTopicARN = builder.snsTopicARN;
        this.jobStatesToNotify = builder.jobStatesToNotify;
        this.devicePickupSnsTopicARN = builder.devicePickupSnsTopicARN;
        this.notifyAllJobStates = builder.notifyAllJobStates;
    }

    /**
     * <p>
     * The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon Resource Names
     * (ARNs) for topics by using the <a
     * href="https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API action.
     * </p>
     * <p>
     * You can subscribe email addresses to an Amazon SNS topic through the Amazon Web Services Management Console, or
     * by using the <a href="https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> Amazon Simple
     * Notification Service (Amazon SNS) API action.
     * </p>
     * 
     * @return The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon
     *         Resource Names (ARNs) for topics by using the <a
     *         href="https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API
     *         action.</p>
     *         <p>
     *         You can subscribe email addresses to an Amazon SNS topic through the Amazon Web Services Management
     *         Console, or by using the <a
     *         href="https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> Amazon Simple
     *         Notification Service (Amazon SNS) API action.
     */
    public final String snsTopicARN() {
        return snsTopicARN;
    }

    /**
     * <p>
     * The list of job states that will trigger a notification for this job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasJobStatesToNotify} method.
     * </p>
     * 
     * @return The list of job states that will trigger a notification for this job.
     */
    public final List<JobState> jobStatesToNotify() {
        return JobStateListCopier.copyStringToEnum(jobStatesToNotify);
    }

    /**
     * For responses, this returns true if the service returned a value for the JobStatesToNotify property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasJobStatesToNotify() {
        return jobStatesToNotify != null && !(jobStatesToNotify instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of job states that will trigger a notification for this job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasJobStatesToNotify} method.
     * </p>
     * 
     * @return The list of job states that will trigger a notification for this job.
     */
    public final List<String> jobStatesToNotifyAsStrings() {
        return jobStatesToNotify;
    }

    /**
     * <p>
     * Used to send SNS notifications for the person picking up the device (identified during job creation).
     * </p>
     * 
     * @return Used to send SNS notifications for the person picking up the device (identified during job creation).
     */
    public final String devicePickupSnsTopicARN() {
        return devicePickupSnsTopicARN;
    }

    /**
     * <p>
     * Any change in job state will trigger a notification for this job.
     * </p>
     * 
     * @return Any change in job state will trigger a notification for this job.
     */
    public final Boolean notifyAllJobStates() {
        return notifyAllJobStates;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(snsTopicARN());
        hashCode = 31 * hashCode + Objects.hashCode(hasJobStatesToNotify() ? jobStatesToNotifyAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(devicePickupSnsTopicARN());
        hashCode = 31 * hashCode + Objects.hashCode(notifyAllJobStates());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Notification)) {
            return false;
        }
        Notification other = (Notification) obj;
        return Objects.equals(snsTopicARN(), other.snsTopicARN()) && hasJobStatesToNotify() == other.hasJobStatesToNotify()
                && Objects.equals(jobStatesToNotifyAsStrings(), other.jobStatesToNotifyAsStrings())
                && Objects.equals(devicePickupSnsTopicARN(), other.devicePickupSnsTopicARN())
                && Objects.equals(notifyAllJobStates(), other.notifyAllJobStates());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Notification").add("SnsTopicARN", snsTopicARN())
                .add("JobStatesToNotify", hasJobStatesToNotify() ? jobStatesToNotifyAsStrings() : null)
                .add("DevicePickupSnsTopicARN", devicePickupSnsTopicARN()).add("NotifyAllJobStates", notifyAllJobStates())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SnsTopicARN":
            return Optional.ofNullable(clazz.cast(snsTopicARN()));
        case "JobStatesToNotify":
            return Optional.ofNullable(clazz.cast(jobStatesToNotifyAsStrings()));
        case "DevicePickupSnsTopicARN":
            return Optional.ofNullable(clazz.cast(devicePickupSnsTopicARN()));
        case "NotifyAllJobStates":
            return Optional.ofNullable(clazz.cast(notifyAllJobStates()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SnsTopicARN", SNS_TOPIC_ARN_FIELD);
        map.put("JobStatesToNotify", JOB_STATES_TO_NOTIFY_FIELD);
        map.put("DevicePickupSnsTopicARN", DEVICE_PICKUP_SNS_TOPIC_ARN_FIELD);
        map.put("NotifyAll", NOTIFY_ALL_JOB_STATES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Notification, T> g) {
        return obj -> g.apply((Notification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Notification> {
        /**
         * <p>
         * The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon Resource
         * Names (ARNs) for topics by using the <a
         * href="https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API action.
         * </p>
         * <p>
         * You can subscribe email addresses to an Amazon SNS topic through the Amazon Web Services Management Console,
         * or by using the <a href="https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> Amazon
         * Simple Notification Service (Amazon SNS) API action.
         * </p>
         * 
         * @param snsTopicARN
         *        The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon
         *        Resource Names (ARNs) for topics by using the <a
         *        href="https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API
         *        action.</p>
         *        <p>
         *        You can subscribe email addresses to an Amazon SNS topic through the Amazon Web Services Management
         *        Console, or by using the <a
         *        href="https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> Amazon Simple
         *        Notification Service (Amazon SNS) API action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicARN(String snsTopicARN);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatesToNotifyWithStrings(Collection<String> jobStatesToNotify);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatesToNotifyWithStrings(String... jobStatesToNotify);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatesToNotify(Collection<JobState> jobStatesToNotify);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatesToNotify(JobState... jobStatesToNotify);

        /**
         * <p>
         * Used to send SNS notifications for the person picking up the device (identified during job creation).
         * </p>
         * 
         * @param devicePickupSnsTopicARN
         *        Used to send SNS notifications for the person picking up the device (identified during job creation).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder devicePickupSnsTopicARN(String devicePickupSnsTopicARN);

        /**
         * <p>
         * Any change in job state will trigger a notification for this job.
         * </p>
         * 
         * @param notifyAllJobStates
         *        Any change in job state will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifyAllJobStates(Boolean notifyAllJobStates);
    }

    static final class BuilderImpl implements Builder {
        private String snsTopicARN;

        private List<String> jobStatesToNotify = DefaultSdkAutoConstructList.getInstance();

        private String devicePickupSnsTopicARN;

        private Boolean notifyAllJobStates;

        private BuilderImpl() {
        }

        private BuilderImpl(Notification model) {
            snsTopicARN(model.snsTopicARN);
            jobStatesToNotifyWithStrings(model.jobStatesToNotify);
            devicePickupSnsTopicARN(model.devicePickupSnsTopicARN);
            notifyAllJobStates(model.notifyAllJobStates);
        }

        public final String getSnsTopicARN() {
            return snsTopicARN;
        }

        public final void setSnsTopicARN(String snsTopicARN) {
            this.snsTopicARN = snsTopicARN;
        }

        @Override
        public final Builder snsTopicARN(String snsTopicARN) {
            this.snsTopicARN = snsTopicARN;
            return this;
        }

        public final Collection<String> getJobStatesToNotify() {
            if (jobStatesToNotify instanceof SdkAutoConstructList) {
                return null;
            }
            return jobStatesToNotify;
        }

        public final void setJobStatesToNotify(Collection<String> jobStatesToNotify) {
            this.jobStatesToNotify = JobStateListCopier.copy(jobStatesToNotify);
        }

        @Override
        public final Builder jobStatesToNotifyWithStrings(Collection<String> jobStatesToNotify) {
            this.jobStatesToNotify = JobStateListCopier.copy(jobStatesToNotify);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStatesToNotifyWithStrings(String... jobStatesToNotify) {
            jobStatesToNotifyWithStrings(Arrays.asList(jobStatesToNotify));
            return this;
        }

        @Override
        public final Builder jobStatesToNotify(Collection<JobState> jobStatesToNotify) {
            this.jobStatesToNotify = JobStateListCopier.copyEnumToString(jobStatesToNotify);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStatesToNotify(JobState... jobStatesToNotify) {
            jobStatesToNotify(Arrays.asList(jobStatesToNotify));
            return this;
        }

        public final String getDevicePickupSnsTopicARN() {
            return devicePickupSnsTopicARN;
        }

        public final void setDevicePickupSnsTopicARN(String devicePickupSnsTopicARN) {
            this.devicePickupSnsTopicARN = devicePickupSnsTopicARN;
        }

        @Override
        public final Builder devicePickupSnsTopicARN(String devicePickupSnsTopicARN) {
            this.devicePickupSnsTopicARN = devicePickupSnsTopicARN;
            return this;
        }

        public final Boolean getNotifyAllJobStates() {
            return notifyAllJobStates;
        }

        public final void setNotifyAllJobStates(Boolean notifyAllJobStates) {
            this.notifyAllJobStates = notifyAllJobStates;
        }

        @Override
        public final Builder notifyAllJobStates(Boolean notifyAllJobStates) {
            this.notifyAllJobStates = notifyAllJobStates;
            return this;
        }

        @Override
        public Notification build() {
            return new Notification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
