/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.simspaceweaver;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.simspaceweaver.internal.SimSpaceWeaverServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.simspaceweaver.model.AccessDeniedException;
import software.amazon.awssdk.services.simspaceweaver.model.ConflictException;
import software.amazon.awssdk.services.simspaceweaver.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.simspaceweaver.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.simspaceweaver.model.DeleteAppRequest;
import software.amazon.awssdk.services.simspaceweaver.model.DeleteAppResponse;
import software.amazon.awssdk.services.simspaceweaver.model.DeleteSimulationRequest;
import software.amazon.awssdk.services.simspaceweaver.model.DeleteSimulationResponse;
import software.amazon.awssdk.services.simspaceweaver.model.DescribeAppRequest;
import software.amazon.awssdk.services.simspaceweaver.model.DescribeAppResponse;
import software.amazon.awssdk.services.simspaceweaver.model.DescribeSimulationRequest;
import software.amazon.awssdk.services.simspaceweaver.model.DescribeSimulationResponse;
import software.amazon.awssdk.services.simspaceweaver.model.InternalServerException;
import software.amazon.awssdk.services.simspaceweaver.model.ListAppsRequest;
import software.amazon.awssdk.services.simspaceweaver.model.ListAppsResponse;
import software.amazon.awssdk.services.simspaceweaver.model.ListSimulationsRequest;
import software.amazon.awssdk.services.simspaceweaver.model.ListSimulationsResponse;
import software.amazon.awssdk.services.simspaceweaver.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.simspaceweaver.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.simspaceweaver.model.ResourceNotFoundException;
import software.amazon.awssdk.services.simspaceweaver.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.simspaceweaver.model.SimSpaceWeaverException;
import software.amazon.awssdk.services.simspaceweaver.model.StartAppRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StartAppResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StartClockRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StartClockResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StartSimulationRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StartSimulationResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StopAppRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StopAppResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StopClockRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StopClockResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StopSimulationRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StopSimulationResponse;
import software.amazon.awssdk.services.simspaceweaver.model.TagResourceRequest;
import software.amazon.awssdk.services.simspaceweaver.model.TagResourceResponse;
import software.amazon.awssdk.services.simspaceweaver.model.TooManyTagsException;
import software.amazon.awssdk.services.simspaceweaver.model.UntagResourceRequest;
import software.amazon.awssdk.services.simspaceweaver.model.UntagResourceResponse;
import software.amazon.awssdk.services.simspaceweaver.model.ValidationException;
import software.amazon.awssdk.services.simspaceweaver.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.DeleteAppRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.DeleteSimulationRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.DescribeAppRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.DescribeSimulationRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.ListAppsRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.ListSimulationsRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StartAppRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StartClockRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StartSimulationRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StopAppRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StopClockRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StopSimulationRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SimSpaceWeaverAsyncClient}.
 *
 * @see SimSpaceWeaverAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSimSpaceWeaverAsyncClient implements SimSpaceWeaverAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSimSpaceWeaverAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSimSpaceWeaverAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a snapshot of the specified simulation. A snapshot is a file that contains simulation state data at a
     * specific time. The state data saved in a snapshot includes entity data from the State Fabric, the simulation
     * configuration specified in the schema, and the clock tick number. You can use the snapshot to initialize a new
     * simulation. For more information about snapshots, see <a
     * href="https://docs.aws.amazon.com/simspaceweaver/latest/userguide/working-with_snapshots.html">Snapshots</a> in
     * the <i>SimSpace Weaver User Guide</i>.
     * </p>
     * <p>
     * You specify a <code>Destination</code> when you create a snapshot. The <code>Destination</code> is the name of an
     * Amazon S3 bucket and an optional <code>ObjectKeyPrefix</code>. The <code>ObjectKeyPrefix</code> is usually the
     * name of a folder in the bucket. SimSpace Weaver creates a <code>snapshot</code> folder inside the
     * <code>Destination</code> and places the snapshot file there.
     * </p>
     * <p>
     * The snapshot file is an Amazon S3 object. It has an object key with the form:
     * <code> <i>object-key-prefix</i>/snapshot/<i>simulation-name</i>-<i>YYMMdd</i>-<i>HHmm</i>-<i>ss</i>.zip</code>,
     * where:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code> <i>YY</i> </code> is the 2-digit year
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>MM</i> </code> is the 2-digit month
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>dd</i> </code> is the 2-digit day of the month
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>HH</i> </code> is the 2-digit hour (24-hour clock)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>mm</i> </code> is the 2-digit minutes
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>ss</i> </code> is the 2-digit seconds
     * </p>
     * </li>
     * </ul>
     *
     * @param createSnapshotRequest
     * @return A Java Future containing the result of the CreateSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.CreateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/CreateSnapshot" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSnapshotResponse> createSnapshot(CreateSnapshotRequest createSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                            .withOperationName("CreateSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSnapshotRequest));
            CompletableFuture<CreateSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the instance of the given custom app.
     * </p>
     *
     * @param deleteAppRequest
     * @return A Java Future containing the result of the DeleteApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.DeleteApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/DeleteApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppResponse> deleteApp(DeleteAppRequest deleteAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppRequest, DeleteAppResponse>().withOperationName("DeleteApp")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAppRequest));
            CompletableFuture<DeleteAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes all SimSpace Weaver resources assigned to the given simulation.
     * </p>
     * <note>
     * <p>
     * Your simulation uses resources in other Amazon Web Services. This API operation doesn't delete resources in other
     * Amazon Web Services.
     * </p>
     * </note>
     *
     * @param deleteSimulationRequest
     * @return A Java Future containing the result of the DeleteSimulation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.DeleteSimulation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/DeleteSimulation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSimulationResponse> deleteSimulation(DeleteSimulationRequest deleteSimulationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSimulationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSimulationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSimulation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSimulationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSimulationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSimulationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSimulationRequest, DeleteSimulationResponse>()
                            .withOperationName("DeleteSimulation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSimulationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSimulationRequest));
            CompletableFuture<DeleteSimulationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the state of the given custom app.
     * </p>
     *
     * @param describeAppRequest
     * @return A Java Future containing the result of the DescribeApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.DescribeApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/DescribeApp" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppResponse> describeApp(DescribeAppRequest describeAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppRequest, DescribeAppResponse>()
                            .withOperationName("DescribeApp").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAppRequest));
            CompletableFuture<DescribeAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the current state of the given simulation.
     * </p>
     *
     * @param describeSimulationRequest
     * @return A Java Future containing the result of the DescribeSimulation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.DescribeSimulation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/DescribeSimulation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSimulationResponse> describeSimulation(DescribeSimulationRequest describeSimulationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSimulationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSimulationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSimulation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSimulationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSimulationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSimulationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSimulationRequest, DescribeSimulationResponse>()
                            .withOperationName("DescribeSimulation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSimulationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSimulationRequest));
            CompletableFuture<DescribeSimulationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all custom apps or service apps for the given simulation and domain.
     * </p>
     *
     * @param listAppsRequest
     * @return A Java Future containing the result of the ListApps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.ListApps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/ListApps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppsResponse> listApps(ListAppsRequest listAppsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAppsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppsRequest, ListAppsResponse>().withOperationName("ListApps")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAppsRequest));
            CompletableFuture<ListAppsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the SimSpace Weaver simulations in the Amazon Web Services account used to make the API call.
     * </p>
     *
     * @param listSimulationsRequest
     * @return A Java Future containing the result of the ListSimulations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.ListSimulations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/ListSimulations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSimulationsResponse> listSimulations(ListSimulationsRequest listSimulationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSimulationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSimulationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSimulations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSimulationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSimulationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSimulationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSimulationsRequest, ListSimulationsResponse>()
                            .withOperationName("ListSimulations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSimulationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSimulationsRequest));
            CompletableFuture<ListSimulationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all tags on a SimSpace Weaver resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a custom app with the configuration specified in the simulation schema.
     * </p>
     *
     * @param startAppRequest
     * @return A Java Future containing the result of the StartApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.StartApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StartApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartAppResponse> startApp(StartAppRequest startAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartAppRequest, StartAppResponse>().withOperationName("StartApp")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(startAppRequest));
            CompletableFuture<StartAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the simulation clock.
     * </p>
     *
     * @param startClockRequest
     * @return A Java Future containing the result of the StartClock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.StartClock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StartClock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartClockResponse> startClock(StartClockRequest startClockRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startClockRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startClockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartClock");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartClockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartClockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartClockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartClockRequest, StartClockResponse>().withOperationName("StartClock")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartClockRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startClockRequest));
            CompletableFuture<StartClockResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a simulation with the given name. You must choose to start your simulation from a schema or from a
     * snapshot. For more information about the schema, see the <a
     * href="https://docs.aws.amazon.com/simspaceweaver/latest/userguide/schema-reference.html">schema reference</a> in
     * the <i>SimSpace Weaver User Guide</i>. For more information about snapshots, see <a
     * href="https://docs.aws.amazon.com/simspaceweaver/latest/userguide/working-with_snapshots.html">Snapshots</a> in
     * the <i>SimSpace Weaver User Guide</i>.
     * </p>
     *
     * @param startSimulationRequest
     * @return A Java Future containing the result of the StartSimulation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.StartSimulation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StartSimulation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSimulationResponse> startSimulation(StartSimulationRequest startSimulationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSimulationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSimulationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSimulation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSimulationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSimulationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSimulationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSimulationRequest, StartSimulationResponse>()
                            .withOperationName("StartSimulation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartSimulationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startSimulationRequest));
            CompletableFuture<StartSimulationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the given custom app and shuts down all of its allocated compute resources.
     * </p>
     *
     * @param stopAppRequest
     * @return A Java Future containing the result of the StopApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.StopApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StopApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopAppResponse> stopApp(StopAppRequest stopAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopAppRequest, StopAppResponse>().withOperationName("StopApp")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new StopAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopAppRequest));
            CompletableFuture<StopAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the simulation clock.
     * </p>
     *
     * @param stopClockRequest
     * @return A Java Future containing the result of the StopClock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.StopClock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StopClock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopClockResponse> stopClock(StopClockRequest stopClockRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopClockRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopClockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopClock");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopClockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopClockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopClockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopClockRequest, StopClockResponse>().withOperationName("StopClock")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopClockRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopClockRequest));
            CompletableFuture<StopClockResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the given simulation.
     * </p>
     * <important>
     * <p>
     * You can't restart a simulation after you stop it. If you want to restart a simulation, then you must stop it,
     * delete it, and start a new instance of it.
     * </p>
     * </important>
     *
     * @param stopSimulationRequest
     * @return A Java Future containing the result of the StopSimulation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.StopSimulation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StopSimulation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StopSimulationResponse> stopSimulation(StopSimulationRequest stopSimulationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopSimulationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopSimulationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopSimulation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopSimulationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopSimulationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopSimulationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopSimulationRequest, StopSimulationResponse>()
                            .withOperationName("StopSimulation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopSimulationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopSimulationRequest));
            CompletableFuture<StopSimulationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to a SimSpace Weaver resource. For more information about tags, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services resources</a>
     * in the <i>Amazon Web Services General Reference</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyTagsException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a SimSpace Weaver resource. For more information about tags, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services resources</a>
     * in the <i>Amazon Web Services General Reference</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SimSpaceWeaverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SimSpaceWeaverAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SimSpaceWeaverServiceClientConfiguration serviceClientConfiguration() {
        return new SimSpaceWeaverServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SimSpaceWeaverException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SimSpaceWeaverServiceClientConfigurationBuilder serviceConfigBuilder = new SimSpaceWeaverServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
