/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.swf.model;

import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.swf.transform.DecisionMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a decision made by the decider. A decision can be one of these types:
 * </p>
 * <ul>
 * <li><b>CancelTimer</b>: cancels a previously started timer and records a <code>TimerCanceled</code> event in the
 * history.</li>
 * <li><b>CancelWorkflowExecution</b>: closes the workflow execution and records a
 * <code>WorkflowExecutionCanceled</code> event in the history.</li>
 * <li><b>CompleteWorkflowExecution</b>: closes the workflow execution and records a
 * <code>WorkflowExecutionCompleted</code> event in the history .</li>
 * <li><b>ContinueAsNewWorkflowExecution</b>: closes the workflow execution and starts a new workflow execution of the
 * same type using the same workflow ID and a unique run ID. A <code>WorkflowExecutionContinuedAsNew</code> event is
 * recorded in the history.</li>
 * <li><b>FailWorkflowExecution</b>: closes the workflow execution and records a <code>WorkflowExecutionFailed</code>
 * event in the history.</li>
 * <li><b>RecordMarker</b>: records a <code>MarkerRecorded</code> event in the history. Markers can be used for adding
 * custom information in the history for instance to let deciders know that they do not need to look at the history
 * beyond the marker event.</li>
 * <li><b>RequestCancelActivityTask</b>: attempts to cancel a previously scheduled activity task. If the activity task
 * was scheduled but has not been assigned to a worker, then it will be canceled. If the activity task was already
 * assigned to a worker, then the worker will be informed that cancellation has been requested in the response to
 * <a>RecordActivityTaskHeartbeat</a>.</li>
 * <li><b>RequestCancelExternalWorkflowExecution</b>: requests that a request be made to cancel the specified external
 * workflow execution and records a <code>RequestCancelExternalWorkflowExecutionInitiated</code> event in the history.</li>
 * <li><b>ScheduleActivityTask</b>: schedules an activity task.</li>
 * <li><b>ScheduleLambdaFunction</b>: schedules a AWS Lambda function.</li>
 * <li><b>SignalExternalWorkflowExecution</b>: requests a signal to be delivered to the specified external workflow
 * execution and records a <code>SignalExternalWorkflowExecutionInitiated</code> event in the history.</li>
 * <li><b>StartChildWorkflowExecution</b>: requests that a child workflow execution be started and records a
 * <code>StartChildWorkflowExecutionInitiated</code> event in the history. The child workflow execution is a separate
 * workflow execution with its own history.</li>
 * <li><b>StartTimer</b>: starts a timer for this workflow execution and records a <code>TimerStarted</code> event in
 * the history. This timer will fire after the specified delay and record a <code>TimerFired</code> event.</li>
 * </ul>
 * <p>
 * <b>Access Control</b>
 * </p>
 * <p>
 * If you grant permission to use <code>RespondDecisionTaskCompleted</code>, you can use IAM policies to express
 * permissions for the list of decisions returned by this action as if they were members of the API. Treating decisions
 * as a pseudo API maintains a uniform conceptual model and helps keep policies readable. For details and example IAM
 * policies, see <a href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to
 * Manage Access to Amazon SWF Workflows</a>.
 * </p>
 * <p>
 * <b>Decision Failure</b>
 * </p>
 * <p>
 * Decisions can fail for several reasons
 * </p>
 * <ul>
 * <li>The ordering of decisions should follow a logical flow. Some decisions might not make sense in the current
 * context of the workflow execution and will therefore fail.</li>
 * <li>A limit on your account was reached.</li>
 * <li>The decision lacks sufficient permissions.</li>
 * </ul>
 * <p>
 * One of the following events might be added to the history to indicate an error. The event attribute's <b>cause</b>
 * parameter indicates the cause. If <b>cause</b> is set to OPERATION_NOT_PERMITTED, the decision failed because it
 * lacked sufficient permissions. For details and example IAM policies, see <a
 * href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
 * Amazon SWF Workflows</a>.
 * </p>
 * <ul>
 * <li><b>ScheduleActivityTaskFailed</b>: a ScheduleActivityTask decision failed. This could happen if the activity type
 * specified in the decision is not registered, is in a deprecated state, or the decision is not properly configured.</li>
 * <li><b>ScheduleLambdaFunctionFailed</b>: a ScheduleLambdaFunctionFailed decision failed. This could happen if the AWS
 * Lambda function specified in the decision does not exist, or the AWS Lambda service's limits are exceeded.</li>
 * <li><b>RequestCancelActivityTaskFailed</b>: a RequestCancelActivityTask decision failed. This could happen if there
 * is no open activity task with the specified activityId.</li>
 * <li><b>StartTimerFailed</b>: a StartTimer decision failed. This could happen if there is another open timer with the
 * same timerId.</li>
 * <li><b>CancelTimerFailed</b>: a CancelTimer decision failed. This could happen if there is no open timer with the
 * specified timerId.</li>
 * <li><b>StartChildWorkflowExecutionFailed</b>: a StartChildWorkflowExecution decision failed. This could happen if the
 * workflow type specified is not registered, is deprecated, or the decision is not properly configured.</li>
 * <li><b>SignalExternalWorkflowExecutionFailed</b>: a SignalExternalWorkflowExecution decision failed. This could
 * happen if the <code>workflowID</code> specified in the decision was incorrect.</li>
 * <li><b>RequestCancelExternalWorkflowExecutionFailed</b>: a RequestCancelExternalWorkflowExecution decision failed.
 * This could happen if the <code>workflowID</code> specified in the decision was incorrect.</li>
 * <li><b>CancelWorkflowExecutionFailed</b>: a CancelWorkflowExecution decision failed. This could happen if there is an
 * unhandled decision task pending in the workflow execution.</li>
 * <li><b>CompleteWorkflowExecutionFailed</b>: a CompleteWorkflowExecution decision failed. This could happen if there
 * is an unhandled decision task pending in the workflow execution.</li>
 * <li><b>ContinueAsNewWorkflowExecutionFailed</b>: a ContinueAsNewWorkflowExecution decision failed. This could happen
 * if there is an unhandled decision task pending in the workflow execution or the ContinueAsNewWorkflowExecution
 * decision was not configured correctly.</li>
 * <li><b>FailWorkflowExecutionFailed</b>: a FailWorkflowExecution decision failed. This could happen if there is an
 * unhandled decision task pending in the workflow execution.</li>
 * </ul>
 * <p>
 * The preceding error events might occur due to an error in the decider logic, which might put the workflow execution
 * in an unstable state The cause field in the event structure for the error event indicates the cause of the error.
 * </p>
 * <note>A workflow execution may be closed by the decider by returning one of the following decisions when completing a
 * decision task: <code>CompleteWorkflowExecution</code>, <code>FailWorkflowExecution</code>,
 * <code>CancelWorkflowExecution</code> and <code>ContinueAsNewWorkflowExecution</code>. An UnhandledDecision fault will
 * be returned if a workflow closing decision is specified and a signal or activity event had been added to the history
 * while the decision task was being performed by the decider. Unlike the above situations which are logic issues, this
 * fault is always possible because of race conditions in a distributed system. The right action here is to call
 * <a>RespondDecisionTaskCompleted</a> without any decisions. This would result in another decision task with these new
 * events included in the history. The decider should handle the new events and may decide to close the workflow
 * execution.</note>
 * <p>
 * <b>How to code a decision</b>
 * </p>
 * <p>
 * You code a decision by first setting the decision type field to one of the above decision values, and then set the
 * corresponding attributes field shown below:
 * </p>
 * <ul>
 * <li><a>ScheduleActivityTaskDecisionAttributes</a></li>
 * <li><a>ScheduleLambdaFunctionDecisionAttributes</a></li>
 * <li><a>RequestCancelActivityTaskDecisionAttributes</a></li>
 * <li><a>CompleteWorkflowExecutionDecisionAttributes</a></li>
 * <li><a>FailWorkflowExecutionDecisionAttributes</a></li>
 * <li><a>CancelWorkflowExecutionDecisionAttributes</a></li>
 * <li><a>ContinueAsNewWorkflowExecutionDecisionAttributes</a></li>
 * <li><a>RecordMarkerDecisionAttributes</a></li>
 * <li><a>StartTimerDecisionAttributes</a></li>
 * <li><a>CancelTimerDecisionAttributes</a></li>
 * <li><a>SignalExternalWorkflowExecutionDecisionAttributes</a></li>
 * <li><a>RequestCancelExternalWorkflowExecutionDecisionAttributes</a></li>
 * <li><a>StartChildWorkflowExecutionDecisionAttributes</a></li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public class Decision implements StructuredPojo, ToCopyableBuilder<Decision.Builder, Decision> {
    private final String decisionType;

    private final ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes;

    private final RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes;

    private final CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes;

    private final FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes;

    private final CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes;

    private final ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes;

    private final RecordMarkerDecisionAttributes recordMarkerDecisionAttributes;

    private final StartTimerDecisionAttributes startTimerDecisionAttributes;

    private final CancelTimerDecisionAttributes cancelTimerDecisionAttributes;

    private final SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes;

    private final RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes;

    private final StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes;

    private final ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes;

    private Decision(BuilderImpl builder) {
        this.decisionType = builder.decisionType;
        this.scheduleActivityTaskDecisionAttributes = builder.scheduleActivityTaskDecisionAttributes;
        this.requestCancelActivityTaskDecisionAttributes = builder.requestCancelActivityTaskDecisionAttributes;
        this.completeWorkflowExecutionDecisionAttributes = builder.completeWorkflowExecutionDecisionAttributes;
        this.failWorkflowExecutionDecisionAttributes = builder.failWorkflowExecutionDecisionAttributes;
        this.cancelWorkflowExecutionDecisionAttributes = builder.cancelWorkflowExecutionDecisionAttributes;
        this.continueAsNewWorkflowExecutionDecisionAttributes = builder.continueAsNewWorkflowExecutionDecisionAttributes;
        this.recordMarkerDecisionAttributes = builder.recordMarkerDecisionAttributes;
        this.startTimerDecisionAttributes = builder.startTimerDecisionAttributes;
        this.cancelTimerDecisionAttributes = builder.cancelTimerDecisionAttributes;
        this.signalExternalWorkflowExecutionDecisionAttributes = builder.signalExternalWorkflowExecutionDecisionAttributes;
        this.requestCancelExternalWorkflowExecutionDecisionAttributes = builder.requestCancelExternalWorkflowExecutionDecisionAttributes;
        this.startChildWorkflowExecutionDecisionAttributes = builder.startChildWorkflowExecutionDecisionAttributes;
        this.scheduleLambdaFunctionDecisionAttributes = builder.scheduleLambdaFunctionDecisionAttributes;
    }

    /**
     * <p>
     * Specifies the type of the decision.
     * </p>
     * 
     * @return Specifies the type of the decision.
     * @see DecisionType
     */
    public String decisionType() {
        return decisionType;
    }

    /**
     * <p>
     * Provides details of the <code>ScheduleActivityTask</code> decision. It is not set for other decision types.
     * </p>
     * 
     * @return Provides details of the <code>ScheduleActivityTask</code> decision. It is not set for other decision
     *         types.
     */
    public ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes() {
        return scheduleActivityTaskDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>RequestCancelActivityTask</code> decision. It is not set for other decision types.
     * </p>
     * 
     * @return Provides details of the <code>RequestCancelActivityTask</code> decision. It is not set for other decision
     *         types.
     */
    public RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes() {
        return requestCancelActivityTaskDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>CompleteWorkflowExecution</code> decision. It is not set for other decision types.
     * </p>
     * 
     * @return Provides details of the <code>CompleteWorkflowExecution</code> decision. It is not set for other decision
     *         types.
     */
    public CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes() {
        return completeWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>FailWorkflowExecution</code> decision. It is not set for other decision types.
     * </p>
     * 
     * @return Provides details of the <code>FailWorkflowExecution</code> decision. It is not set for other decision
     *         types.
     */
    public FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes() {
        return failWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>CancelWorkflowExecution</code> decision. It is not set for other decision types.
     * </p>
     * 
     * @return Provides details of the <code>CancelWorkflowExecution</code> decision. It is not set for other decision
     *         types.
     */
    public CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes() {
        return cancelWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>ContinueAsNewWorkflowExecution</code> decision. It is not set for other decision
     * types.
     * </p>
     * 
     * @return Provides details of the <code>ContinueAsNewWorkflowExecution</code> decision. It is not set for other
     *         decision types.
     */
    public ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes() {
        return continueAsNewWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>RecordMarker</code> decision. It is not set for other decision types.
     * </p>
     * 
     * @return Provides details of the <code>RecordMarker</code> decision. It is not set for other decision types.
     */
    public RecordMarkerDecisionAttributes recordMarkerDecisionAttributes() {
        return recordMarkerDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>StartTimer</code> decision. It is not set for other decision types.
     * </p>
     * 
     * @return Provides details of the <code>StartTimer</code> decision. It is not set for other decision types.
     */
    public StartTimerDecisionAttributes startTimerDecisionAttributes() {
        return startTimerDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>CancelTimer</code> decision. It is not set for other decision types.
     * </p>
     * 
     * @return Provides details of the <code>CancelTimer</code> decision. It is not set for other decision types.
     */
    public CancelTimerDecisionAttributes cancelTimerDecisionAttributes() {
        return cancelTimerDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>SignalExternalWorkflowExecution</code> decision. It is not set for other decision
     * types.
     * </p>
     * 
     * @return Provides details of the <code>SignalExternalWorkflowExecution</code> decision. It is not set for other
     *         decision types.
     */
    public SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes() {
        return signalExternalWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It is not set for other
     * decision types.
     * </p>
     * 
     * @return Provides details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It is not set for
     *         other decision types.
     */
    public RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes() {
        return requestCancelExternalWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides details of the <code>StartChildWorkflowExecution</code> decision. It is not set for other decision
     * types.
     * </p>
     * 
     * @return Provides details of the <code>StartChildWorkflowExecution</code> decision. It is not set for other
     *         decision types.
     */
    public StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes() {
        return startChildWorkflowExecutionDecisionAttributes;
    }

    /**
     *
     * @return
     */
    public ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes() {
        return scheduleLambdaFunctionDecisionAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((decisionType() == null) ? 0 : decisionType().hashCode());
        hashCode = 31 * hashCode
                + ((scheduleActivityTaskDecisionAttributes() == null) ? 0 : scheduleActivityTaskDecisionAttributes().hashCode());
        hashCode = 31
                * hashCode
                + ((requestCancelActivityTaskDecisionAttributes() == null) ? 0 : requestCancelActivityTaskDecisionAttributes()
                        .hashCode());
        hashCode = 31
                * hashCode
                + ((completeWorkflowExecutionDecisionAttributes() == null) ? 0 : completeWorkflowExecutionDecisionAttributes()
                        .hashCode());
        hashCode = 31
                * hashCode
                + ((failWorkflowExecutionDecisionAttributes() == null) ? 0 : failWorkflowExecutionDecisionAttributes().hashCode());
        hashCode = 31
                * hashCode
                + ((cancelWorkflowExecutionDecisionAttributes() == null) ? 0 : cancelWorkflowExecutionDecisionAttributes()
                        .hashCode());
        hashCode = 31
                * hashCode
                + ((continueAsNewWorkflowExecutionDecisionAttributes() == null) ? 0
                        : continueAsNewWorkflowExecutionDecisionAttributes().hashCode());
        hashCode = 31 * hashCode + ((recordMarkerDecisionAttributes() == null) ? 0 : recordMarkerDecisionAttributes().hashCode());
        hashCode = 31 * hashCode + ((startTimerDecisionAttributes() == null) ? 0 : startTimerDecisionAttributes().hashCode());
        hashCode = 31 * hashCode + ((cancelTimerDecisionAttributes() == null) ? 0 : cancelTimerDecisionAttributes().hashCode());
        hashCode = 31
                * hashCode
                + ((signalExternalWorkflowExecutionDecisionAttributes() == null) ? 0
                        : signalExternalWorkflowExecutionDecisionAttributes().hashCode());
        hashCode = 31
                * hashCode
                + ((requestCancelExternalWorkflowExecutionDecisionAttributes() == null) ? 0
                        : requestCancelExternalWorkflowExecutionDecisionAttributes().hashCode());
        hashCode = 31
                * hashCode
                + ((startChildWorkflowExecutionDecisionAttributes() == null) ? 0
                        : startChildWorkflowExecutionDecisionAttributes().hashCode());
        hashCode = 31
                * hashCode
                + ((scheduleLambdaFunctionDecisionAttributes() == null) ? 0 : scheduleLambdaFunctionDecisionAttributes()
                        .hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Decision)) {
            return false;
        }
        Decision other = (Decision) obj;
        if (other.decisionType() == null ^ this.decisionType() == null) {
            return false;
        }
        if (other.decisionType() != null && !other.decisionType().equals(this.decisionType())) {
            return false;
        }
        if (other.scheduleActivityTaskDecisionAttributes() == null ^ this.scheduleActivityTaskDecisionAttributes() == null) {
            return false;
        }
        if (other.scheduleActivityTaskDecisionAttributes() != null
                && !other.scheduleActivityTaskDecisionAttributes().equals(this.scheduleActivityTaskDecisionAttributes())) {
            return false;
        }
        if (other.requestCancelActivityTaskDecisionAttributes() == null
                ^ this.requestCancelActivityTaskDecisionAttributes() == null) {
            return false;
        }
        if (other.requestCancelActivityTaskDecisionAttributes() != null
                && !other.requestCancelActivityTaskDecisionAttributes()
                        .equals(this.requestCancelActivityTaskDecisionAttributes())) {
            return false;
        }
        if (other.completeWorkflowExecutionDecisionAttributes() == null
                ^ this.completeWorkflowExecutionDecisionAttributes() == null) {
            return false;
        }
        if (other.completeWorkflowExecutionDecisionAttributes() != null
                && !other.completeWorkflowExecutionDecisionAttributes()
                        .equals(this.completeWorkflowExecutionDecisionAttributes())) {
            return false;
        }
        if (other.failWorkflowExecutionDecisionAttributes() == null ^ this.failWorkflowExecutionDecisionAttributes() == null) {
            return false;
        }
        if (other.failWorkflowExecutionDecisionAttributes() != null
                && !other.failWorkflowExecutionDecisionAttributes().equals(this.failWorkflowExecutionDecisionAttributes())) {
            return false;
        }
        if (other.cancelWorkflowExecutionDecisionAttributes() == null ^ this.cancelWorkflowExecutionDecisionAttributes() == null) {
            return false;
        }
        if (other.cancelWorkflowExecutionDecisionAttributes() != null
                && !other.cancelWorkflowExecutionDecisionAttributes().equals(this.cancelWorkflowExecutionDecisionAttributes())) {
            return false;
        }
        if (other.continueAsNewWorkflowExecutionDecisionAttributes() == null
                ^ this.continueAsNewWorkflowExecutionDecisionAttributes() == null) {
            return false;
        }
        if (other.continueAsNewWorkflowExecutionDecisionAttributes() != null
                && !other.continueAsNewWorkflowExecutionDecisionAttributes().equals(
                        this.continueAsNewWorkflowExecutionDecisionAttributes())) {
            return false;
        }
        if (other.recordMarkerDecisionAttributes() == null ^ this.recordMarkerDecisionAttributes() == null) {
            return false;
        }
        if (other.recordMarkerDecisionAttributes() != null
                && !other.recordMarkerDecisionAttributes().equals(this.recordMarkerDecisionAttributes())) {
            return false;
        }
        if (other.startTimerDecisionAttributes() == null ^ this.startTimerDecisionAttributes() == null) {
            return false;
        }
        if (other.startTimerDecisionAttributes() != null
                && !other.startTimerDecisionAttributes().equals(this.startTimerDecisionAttributes())) {
            return false;
        }
        if (other.cancelTimerDecisionAttributes() == null ^ this.cancelTimerDecisionAttributes() == null) {
            return false;
        }
        if (other.cancelTimerDecisionAttributes() != null
                && !other.cancelTimerDecisionAttributes().equals(this.cancelTimerDecisionAttributes())) {
            return false;
        }
        if (other.signalExternalWorkflowExecutionDecisionAttributes() == null
                ^ this.signalExternalWorkflowExecutionDecisionAttributes() == null) {
            return false;
        }
        if (other.signalExternalWorkflowExecutionDecisionAttributes() != null
                && !other.signalExternalWorkflowExecutionDecisionAttributes().equals(
                        this.signalExternalWorkflowExecutionDecisionAttributes())) {
            return false;
        }
        if (other.requestCancelExternalWorkflowExecutionDecisionAttributes() == null
                ^ this.requestCancelExternalWorkflowExecutionDecisionAttributes() == null) {
            return false;
        }
        if (other.requestCancelExternalWorkflowExecutionDecisionAttributes() != null
                && !other.requestCancelExternalWorkflowExecutionDecisionAttributes().equals(
                        this.requestCancelExternalWorkflowExecutionDecisionAttributes())) {
            return false;
        }
        if (other.startChildWorkflowExecutionDecisionAttributes() == null
                ^ this.startChildWorkflowExecutionDecisionAttributes() == null) {
            return false;
        }
        if (other.startChildWorkflowExecutionDecisionAttributes() != null
                && !other.startChildWorkflowExecutionDecisionAttributes().equals(
                        this.startChildWorkflowExecutionDecisionAttributes())) {
            return false;
        }
        if (other.scheduleLambdaFunctionDecisionAttributes() == null ^ this.scheduleLambdaFunctionDecisionAttributes() == null) {
            return false;
        }
        if (other.scheduleLambdaFunctionDecisionAttributes() != null
                && !other.scheduleLambdaFunctionDecisionAttributes().equals(this.scheduleLambdaFunctionDecisionAttributes())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (decisionType() != null) {
            sb.append("DecisionType: ").append(decisionType()).append(",");
        }
        if (scheduleActivityTaskDecisionAttributes() != null) {
            sb.append("ScheduleActivityTaskDecisionAttributes: ").append(scheduleActivityTaskDecisionAttributes()).append(",");
        }
        if (requestCancelActivityTaskDecisionAttributes() != null) {
            sb.append("RequestCancelActivityTaskDecisionAttributes: ").append(requestCancelActivityTaskDecisionAttributes())
                    .append(",");
        }
        if (completeWorkflowExecutionDecisionAttributes() != null) {
            sb.append("CompleteWorkflowExecutionDecisionAttributes: ").append(completeWorkflowExecutionDecisionAttributes())
                    .append(",");
        }
        if (failWorkflowExecutionDecisionAttributes() != null) {
            sb.append("FailWorkflowExecutionDecisionAttributes: ").append(failWorkflowExecutionDecisionAttributes()).append(",");
        }
        if (cancelWorkflowExecutionDecisionAttributes() != null) {
            sb.append("CancelWorkflowExecutionDecisionAttributes: ").append(cancelWorkflowExecutionDecisionAttributes())
                    .append(",");
        }
        if (continueAsNewWorkflowExecutionDecisionAttributes() != null) {
            sb.append("ContinueAsNewWorkflowExecutionDecisionAttributes: ")
                    .append(continueAsNewWorkflowExecutionDecisionAttributes()).append(",");
        }
        if (recordMarkerDecisionAttributes() != null) {
            sb.append("RecordMarkerDecisionAttributes: ").append(recordMarkerDecisionAttributes()).append(",");
        }
        if (startTimerDecisionAttributes() != null) {
            sb.append("StartTimerDecisionAttributes: ").append(startTimerDecisionAttributes()).append(",");
        }
        if (cancelTimerDecisionAttributes() != null) {
            sb.append("CancelTimerDecisionAttributes: ").append(cancelTimerDecisionAttributes()).append(",");
        }
        if (signalExternalWorkflowExecutionDecisionAttributes() != null) {
            sb.append("SignalExternalWorkflowExecutionDecisionAttributes: ")
                    .append(signalExternalWorkflowExecutionDecisionAttributes()).append(",");
        }
        if (requestCancelExternalWorkflowExecutionDecisionAttributes() != null) {
            sb.append("RequestCancelExternalWorkflowExecutionDecisionAttributes: ")
                    .append(requestCancelExternalWorkflowExecutionDecisionAttributes()).append(",");
        }
        if (startChildWorkflowExecutionDecisionAttributes() != null) {
            sb.append("StartChildWorkflowExecutionDecisionAttributes: ").append(startChildWorkflowExecutionDecisionAttributes())
                    .append(",");
        }
        if (scheduleLambdaFunctionDecisionAttributes() != null) {
            sb.append("ScheduleLambdaFunctionDecisionAttributes: ").append(scheduleLambdaFunctionDecisionAttributes())
                    .append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DecisionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Decision> {
        /**
         * <p>
         * Specifies the type of the decision.
         * </p>
         * 
         * @param decisionType
         *        Specifies the type of the decision.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DecisionType
         */
        Builder decisionType(String decisionType);

        /**
         * <p>
         * Specifies the type of the decision.
         * </p>
         * 
         * @param decisionType
         *        Specifies the type of the decision.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DecisionType
         */
        Builder decisionType(DecisionType decisionType);

        /**
         * <p>
         * Provides details of the <code>ScheduleActivityTask</code> decision. It is not set for other decision types.
         * </p>
         * 
         * @param scheduleActivityTaskDecisionAttributes
         *        Provides details of the <code>ScheduleActivityTask</code> decision. It is not set for other decision
         *        types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleActivityTaskDecisionAttributes(
                ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>RequestCancelActivityTask</code> decision. It is not set for other decision
         * types.
         * </p>
         * 
         * @param requestCancelActivityTaskDecisionAttributes
         *        Provides details of the <code>RequestCancelActivityTask</code> decision. It is not set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestCancelActivityTaskDecisionAttributes(
                RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>CompleteWorkflowExecution</code> decision. It is not set for other decision
         * types.
         * </p>
         * 
         * @param completeWorkflowExecutionDecisionAttributes
         *        Provides details of the <code>CompleteWorkflowExecution</code> decision. It is not set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completeWorkflowExecutionDecisionAttributes(
                CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>FailWorkflowExecution</code> decision. It is not set for other decision types.
         * </p>
         * 
         * @param failWorkflowExecutionDecisionAttributes
         *        Provides details of the <code>FailWorkflowExecution</code> decision. It is not set for other decision
         *        types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failWorkflowExecutionDecisionAttributes(
                FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>CancelWorkflowExecution</code> decision. It is not set for other decision
         * types.
         * </p>
         * 
         * @param cancelWorkflowExecutionDecisionAttributes
         *        Provides details of the <code>CancelWorkflowExecution</code> decision. It is not set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cancelWorkflowExecutionDecisionAttributes(
                CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>ContinueAsNewWorkflowExecution</code> decision. It is not set for other
         * decision types.
         * </p>
         * 
         * @param continueAsNewWorkflowExecutionDecisionAttributes
         *        Provides details of the <code>ContinueAsNewWorkflowExecution</code> decision. It is not set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder continueAsNewWorkflowExecutionDecisionAttributes(
                ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>RecordMarker</code> decision. It is not set for other decision types.
         * </p>
         * 
         * @param recordMarkerDecisionAttributes
         *        Provides details of the <code>RecordMarker</code> decision. It is not set for other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordMarkerDecisionAttributes(RecordMarkerDecisionAttributes recordMarkerDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>StartTimer</code> decision. It is not set for other decision types.
         * </p>
         * 
         * @param startTimerDecisionAttributes
         *        Provides details of the <code>StartTimer</code> decision. It is not set for other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTimerDecisionAttributes(StartTimerDecisionAttributes startTimerDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>CancelTimer</code> decision. It is not set for other decision types.
         * </p>
         * 
         * @param cancelTimerDecisionAttributes
         *        Provides details of the <code>CancelTimer</code> decision. It is not set for other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cancelTimerDecisionAttributes(CancelTimerDecisionAttributes cancelTimerDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>SignalExternalWorkflowExecution</code> decision. It is not set for other
         * decision types.
         * </p>
         * 
         * @param signalExternalWorkflowExecutionDecisionAttributes
         *        Provides details of the <code>SignalExternalWorkflowExecution</code> decision. It is not set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signalExternalWorkflowExecutionDecisionAttributes(
                SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It is not set for other
         * decision types.
         * </p>
         * 
         * @param requestCancelExternalWorkflowExecutionDecisionAttributes
         *        Provides details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It is not set
         *        for other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestCancelExternalWorkflowExecutionDecisionAttributes(
                RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides details of the <code>StartChildWorkflowExecution</code> decision. It is not set for other decision
         * types.
         * </p>
         * 
         * @param startChildWorkflowExecutionDecisionAttributes
         *        Provides details of the <code>StartChildWorkflowExecution</code> decision. It is not set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startChildWorkflowExecutionDecisionAttributes(
                StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes);

        /**
         *
         * @param scheduleLambdaFunctionDecisionAttributes
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleLambdaFunctionDecisionAttributes(
                ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes);
    }

    private static final class BuilderImpl implements Builder {
        private String decisionType;

        private ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes;

        private RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes;

        private CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes;

        private FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes;

        private CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes;

        private ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes;

        private RecordMarkerDecisionAttributes recordMarkerDecisionAttributes;

        private StartTimerDecisionAttributes startTimerDecisionAttributes;

        private CancelTimerDecisionAttributes cancelTimerDecisionAttributes;

        private SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes;

        private RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes;

        private StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes;

        private ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes;

        private BuilderImpl() {
        }

        private BuilderImpl(Decision model) {
            setDecisionType(model.decisionType);
            setScheduleActivityTaskDecisionAttributes(model.scheduleActivityTaskDecisionAttributes);
            setRequestCancelActivityTaskDecisionAttributes(model.requestCancelActivityTaskDecisionAttributes);
            setCompleteWorkflowExecutionDecisionAttributes(model.completeWorkflowExecutionDecisionAttributes);
            setFailWorkflowExecutionDecisionAttributes(model.failWorkflowExecutionDecisionAttributes);
            setCancelWorkflowExecutionDecisionAttributes(model.cancelWorkflowExecutionDecisionAttributes);
            setContinueAsNewWorkflowExecutionDecisionAttributes(model.continueAsNewWorkflowExecutionDecisionAttributes);
            setRecordMarkerDecisionAttributes(model.recordMarkerDecisionAttributes);
            setStartTimerDecisionAttributes(model.startTimerDecisionAttributes);
            setCancelTimerDecisionAttributes(model.cancelTimerDecisionAttributes);
            setSignalExternalWorkflowExecutionDecisionAttributes(model.signalExternalWorkflowExecutionDecisionAttributes);
            setRequestCancelExternalWorkflowExecutionDecisionAttributes(model.requestCancelExternalWorkflowExecutionDecisionAttributes);
            setStartChildWorkflowExecutionDecisionAttributes(model.startChildWorkflowExecutionDecisionAttributes);
            setScheduleLambdaFunctionDecisionAttributes(model.scheduleLambdaFunctionDecisionAttributes);
        }

        public final String getDecisionType() {
            return decisionType;
        }

        @Override
        public final Builder decisionType(String decisionType) {
            this.decisionType = decisionType;
            return this;
        }

        @Override
        public final Builder decisionType(DecisionType decisionType) {
            this.decisionType(decisionType.toString());
            return this;
        }

        public final void setDecisionType(String decisionType) {
            this.decisionType = decisionType;
        }

        public final void setDecisionType(DecisionType decisionType) {
            this.decisionType(decisionType.toString());
        }

        public final ScheduleActivityTaskDecisionAttributes getScheduleActivityTaskDecisionAttributes() {
            return scheduleActivityTaskDecisionAttributes;
        }

        @Override
        public final Builder scheduleActivityTaskDecisionAttributes(
                ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes) {
            this.scheduleActivityTaskDecisionAttributes = scheduleActivityTaskDecisionAttributes;
            return this;
        }

        public final void setScheduleActivityTaskDecisionAttributes(
                ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes) {
            this.scheduleActivityTaskDecisionAttributes = scheduleActivityTaskDecisionAttributes;
        }

        public final RequestCancelActivityTaskDecisionAttributes getRequestCancelActivityTaskDecisionAttributes() {
            return requestCancelActivityTaskDecisionAttributes;
        }

        @Override
        public final Builder requestCancelActivityTaskDecisionAttributes(
                RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes) {
            this.requestCancelActivityTaskDecisionAttributes = requestCancelActivityTaskDecisionAttributes;
            return this;
        }

        public final void setRequestCancelActivityTaskDecisionAttributes(
                RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes) {
            this.requestCancelActivityTaskDecisionAttributes = requestCancelActivityTaskDecisionAttributes;
        }

        public final CompleteWorkflowExecutionDecisionAttributes getCompleteWorkflowExecutionDecisionAttributes() {
            return completeWorkflowExecutionDecisionAttributes;
        }

        @Override
        public final Builder completeWorkflowExecutionDecisionAttributes(
                CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes) {
            this.completeWorkflowExecutionDecisionAttributes = completeWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setCompleteWorkflowExecutionDecisionAttributes(
                CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes) {
            this.completeWorkflowExecutionDecisionAttributes = completeWorkflowExecutionDecisionAttributes;
        }

        public final FailWorkflowExecutionDecisionAttributes getFailWorkflowExecutionDecisionAttributes() {
            return failWorkflowExecutionDecisionAttributes;
        }

        @Override
        public final Builder failWorkflowExecutionDecisionAttributes(
                FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes) {
            this.failWorkflowExecutionDecisionAttributes = failWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setFailWorkflowExecutionDecisionAttributes(
                FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes) {
            this.failWorkflowExecutionDecisionAttributes = failWorkflowExecutionDecisionAttributes;
        }

        public final CancelWorkflowExecutionDecisionAttributes getCancelWorkflowExecutionDecisionAttributes() {
            return cancelWorkflowExecutionDecisionAttributes;
        }

        @Override
        public final Builder cancelWorkflowExecutionDecisionAttributes(
                CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes) {
            this.cancelWorkflowExecutionDecisionAttributes = cancelWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setCancelWorkflowExecutionDecisionAttributes(
                CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes) {
            this.cancelWorkflowExecutionDecisionAttributes = cancelWorkflowExecutionDecisionAttributes;
        }

        public final ContinueAsNewWorkflowExecutionDecisionAttributes getContinueAsNewWorkflowExecutionDecisionAttributes() {
            return continueAsNewWorkflowExecutionDecisionAttributes;
        }

        @Override
        public final Builder continueAsNewWorkflowExecutionDecisionAttributes(
                ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes) {
            this.continueAsNewWorkflowExecutionDecisionAttributes = continueAsNewWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setContinueAsNewWorkflowExecutionDecisionAttributes(
                ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes) {
            this.continueAsNewWorkflowExecutionDecisionAttributes = continueAsNewWorkflowExecutionDecisionAttributes;
        }

        public final RecordMarkerDecisionAttributes getRecordMarkerDecisionAttributes() {
            return recordMarkerDecisionAttributes;
        }

        @Override
        public final Builder recordMarkerDecisionAttributes(RecordMarkerDecisionAttributes recordMarkerDecisionAttributes) {
            this.recordMarkerDecisionAttributes = recordMarkerDecisionAttributes;
            return this;
        }

        public final void setRecordMarkerDecisionAttributes(RecordMarkerDecisionAttributes recordMarkerDecisionAttributes) {
            this.recordMarkerDecisionAttributes = recordMarkerDecisionAttributes;
        }

        public final StartTimerDecisionAttributes getStartTimerDecisionAttributes() {
            return startTimerDecisionAttributes;
        }

        @Override
        public final Builder startTimerDecisionAttributes(StartTimerDecisionAttributes startTimerDecisionAttributes) {
            this.startTimerDecisionAttributes = startTimerDecisionAttributes;
            return this;
        }

        public final void setStartTimerDecisionAttributes(StartTimerDecisionAttributes startTimerDecisionAttributes) {
            this.startTimerDecisionAttributes = startTimerDecisionAttributes;
        }

        public final CancelTimerDecisionAttributes getCancelTimerDecisionAttributes() {
            return cancelTimerDecisionAttributes;
        }

        @Override
        public final Builder cancelTimerDecisionAttributes(CancelTimerDecisionAttributes cancelTimerDecisionAttributes) {
            this.cancelTimerDecisionAttributes = cancelTimerDecisionAttributes;
            return this;
        }

        public final void setCancelTimerDecisionAttributes(CancelTimerDecisionAttributes cancelTimerDecisionAttributes) {
            this.cancelTimerDecisionAttributes = cancelTimerDecisionAttributes;
        }

        public final SignalExternalWorkflowExecutionDecisionAttributes getSignalExternalWorkflowExecutionDecisionAttributes() {
            return signalExternalWorkflowExecutionDecisionAttributes;
        }

        @Override
        public final Builder signalExternalWorkflowExecutionDecisionAttributes(
                SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes) {
            this.signalExternalWorkflowExecutionDecisionAttributes = signalExternalWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setSignalExternalWorkflowExecutionDecisionAttributes(
                SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes) {
            this.signalExternalWorkflowExecutionDecisionAttributes = signalExternalWorkflowExecutionDecisionAttributes;
        }

        public final RequestCancelExternalWorkflowExecutionDecisionAttributes getRequestCancelExternalWorkflowExecutionDecisionAttributes() {
            return requestCancelExternalWorkflowExecutionDecisionAttributes;
        }

        @Override
        public final Builder requestCancelExternalWorkflowExecutionDecisionAttributes(
                RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes) {
            this.requestCancelExternalWorkflowExecutionDecisionAttributes = requestCancelExternalWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setRequestCancelExternalWorkflowExecutionDecisionAttributes(
                RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes) {
            this.requestCancelExternalWorkflowExecutionDecisionAttributes = requestCancelExternalWorkflowExecutionDecisionAttributes;
        }

        public final StartChildWorkflowExecutionDecisionAttributes getStartChildWorkflowExecutionDecisionAttributes() {
            return startChildWorkflowExecutionDecisionAttributes;
        }

        @Override
        public final Builder startChildWorkflowExecutionDecisionAttributes(
                StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes) {
            this.startChildWorkflowExecutionDecisionAttributes = startChildWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setStartChildWorkflowExecutionDecisionAttributes(
                StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes) {
            this.startChildWorkflowExecutionDecisionAttributes = startChildWorkflowExecutionDecisionAttributes;
        }

        public final ScheduleLambdaFunctionDecisionAttributes getScheduleLambdaFunctionDecisionAttributes() {
            return scheduleLambdaFunctionDecisionAttributes;
        }

        @Override
        public final Builder scheduleLambdaFunctionDecisionAttributes(
                ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes) {
            this.scheduleLambdaFunctionDecisionAttributes = scheduleLambdaFunctionDecisionAttributes;
            return this;
        }

        public final void setScheduleLambdaFunctionDecisionAttributes(
                ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes) {
            this.scheduleLambdaFunctionDecisionAttributes = scheduleLambdaFunctionDecisionAttributes;
        }

        @Override
        public Decision build() {
            return new Decision(this);
        }
    }
}
