/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to return the requested sending authorization policies for an identity. Sending authorization is
 * an Amazon SES feature that enables you to authorize other senders to use your identities. For information, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetIdentityPoliciesRequest extends SesRequest implements
        ToCopyableBuilder<GetIdentityPoliciesRequest.Builder, GetIdentityPoliciesRequest> {
    private static final SdkField<String> IDENTITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Identity").getter(getter(GetIdentityPoliciesRequest::identity)).setter(setter(Builder::identity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Identity").build()).build();

    private static final SdkField<List<String>> POLICY_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PolicyNames")
            .getter(getter(GetIdentityPoliciesRequest::policyNames))
            .setter(setter(Builder::policyNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTITY_FIELD,
            POLICY_NAMES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String identity;

    private final List<String> policyNames;

    private GetIdentityPoliciesRequest(BuilderImpl builder) {
        super(builder);
        this.identity = builder.identity;
        this.policyNames = builder.policyNames;
    }

    /**
     * <p>
     * The identity for which the policies are retrieved. You can specify an identity by using its name or by using its
     * Amazon Resource Name (ARN). Examples: <code>user@example.com</code>, <code>example.com</code>,
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
     * </p>
     * <p>
     * To successfully call this operation, you must own the identity.
     * </p>
     * 
     * @return The identity for which the policies are retrieved. You can specify an identity by using its name or by
     *         using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>, <code>example.com</code>,
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.</p>
     *         <p>
     *         To successfully call this operation, you must own the identity.
     */
    public final String identity() {
        return identity;
    }

    /**
     * For responses, this returns true if the service returned a value for the PolicyNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPolicyNames() {
        return policyNames != null && !(policyNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time. If you do
     * not know the names of the policies that are attached to the identity, you can use
     * <code>ListIdentityPolicies</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPolicyNames} method.
     * </p>
     * 
     * @return A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time. If
     *         you do not know the names of the policies that are attached to the identity, you can use
     *         <code>ListIdentityPolicies</code>.
     */
    public final List<String> policyNames() {
        return policyNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(identity());
        hashCode = 31 * hashCode + Objects.hashCode(hasPolicyNames() ? policyNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetIdentityPoliciesRequest)) {
            return false;
        }
        GetIdentityPoliciesRequest other = (GetIdentityPoliciesRequest) obj;
        return Objects.equals(identity(), other.identity()) && hasPolicyNames() == other.hasPolicyNames()
                && Objects.equals(policyNames(), other.policyNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetIdentityPoliciesRequest").add("Identity", identity())
                .add("PolicyNames", hasPolicyNames() ? policyNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Identity":
            return Optional.ofNullable(clazz.cast(identity()));
        case "PolicyNames":
            return Optional.ofNullable(clazz.cast(policyNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Identity", IDENTITY_FIELD);
        map.put("PolicyNames", POLICY_NAMES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GetIdentityPoliciesRequest, T> g) {
        return obj -> g.apply((GetIdentityPoliciesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetIdentityPoliciesRequest> {
        /**
         * <p>
         * The identity for which the policies are retrieved. You can specify an identity by using its name or by using
         * its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>, <code>example.com</code>,
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
         * </p>
         * <p>
         * To successfully call this operation, you must own the identity.
         * </p>
         * 
         * @param identity
         *        The identity for which the policies are retrieved. You can specify an identity by using its name or by
         *        using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>,
         *        <code>example.com</code>, <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.</p>
         *        <p>
         *        To successfully call this operation, you must own the identity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identity(String identity);

        /**
         * <p>
         * A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time. If you
         * do not know the names of the policies that are attached to the identity, you can use
         * <code>ListIdentityPolicies</code>.
         * </p>
         * 
         * @param policyNames
         *        A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time.
         *        If you do not know the names of the policies that are attached to the identity, you can use
         *        <code>ListIdentityPolicies</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyNames(Collection<String> policyNames);

        /**
         * <p>
         * A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time. If you
         * do not know the names of the policies that are attached to the identity, you can use
         * <code>ListIdentityPolicies</code>.
         * </p>
         * 
         * @param policyNames
         *        A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time.
         *        If you do not know the names of the policies that are attached to the identity, you can use
         *        <code>ListIdentityPolicies</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyNames(String... policyNames);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String identity;

        private List<String> policyNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetIdentityPoliciesRequest model) {
            super(model);
            identity(model.identity);
            policyNames(model.policyNames);
        }

        public final String getIdentity() {
            return identity;
        }

        public final void setIdentity(String identity) {
            this.identity = identity;
        }

        @Override
        public final Builder identity(String identity) {
            this.identity = identity;
            return this;
        }

        public final Collection<String> getPolicyNames() {
            if (policyNames instanceof SdkAutoConstructList) {
                return null;
            }
            return policyNames;
        }

        public final void setPolicyNames(Collection<String> policyNames) {
            this.policyNames = PolicyNameListCopier.copy(policyNames);
        }

        @Override
        public final Builder policyNames(Collection<String> policyNames) {
            this.policyNames = PolicyNameListCopier.copy(policyNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policyNames(String... policyNames) {
            policyNames(Arrays.asList(policyNames));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetIdentityPoliciesRequest build() {
            return new GetIdentityPoliciesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
