/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemakerfeaturestoreruntime;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.internal.SageMakerFeatureStoreRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.AccessForbiddenException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.BatchGetRecordRequest;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.BatchGetRecordResponse;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.DeleteRecordRequest;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.DeleteRecordResponse;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.GetRecordRequest;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.GetRecordResponse;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.InternalFailureException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.PutRecordRequest;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.PutRecordResponse;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.SageMakerFeatureStoreRuntimeException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.ServiceUnavailableException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.ValidationErrorException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.transform.BatchGetRecordRequestMarshaller;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.transform.DeleteRecordRequestMarshaller;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.transform.GetRecordRequestMarshaller;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.transform.PutRecordRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SageMakerFeatureStoreRuntimeAsyncClient}.
 *
 * @see SageMakerFeatureStoreRuntimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSageMakerFeatureStoreRuntimeAsyncClient implements SageMakerFeatureStoreRuntimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSageMakerFeatureStoreRuntimeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSageMakerFeatureStoreRuntimeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Retrieves a batch of <code>Records</code> from a <code>FeatureGroup</code>.
     * </p>
     *
     * @param batchGetRecordRequest
     * @return A Java Future containing the result of the BatchGetRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationErrorException There was an error validating your request.</li>
     *         <li>InternalFailureException An internal failure occurred. Try your request again. If the problem
     *         persists, contact Amazon Web Services customer support.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>AccessForbiddenException You do not have permission to perform an action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerFeatureStoreRuntimeException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerFeatureStoreRuntimeAsyncClient.BatchGetRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-featurestore-runtime-2020-07-01/BatchGetRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetRecordResponse> batchGetRecord(BatchGetRecordRequest batchGetRecordRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetRecordRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker FeatureStore Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetRecordResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetRecordRequest, BatchGetRecordResponse>()
                            .withOperationName("BatchGetRecord").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetRecordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetRecordRequest));
            CompletableFuture<BatchGetRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a <code>Record</code> from a <code>FeatureGroup</code> in the <code>OnlineStore</code>. Feature Store
     * supports both <code>SoftDelete</code> and <code>HardDelete</code>. For <code>SoftDelete</code> (default), feature
     * columns are set to <code>null</code> and the record is no longer retrievable by <code>GetRecord</code> or
     * <code>BatchGetRecord</code>. For <code>HardDelete</code>, the complete <code>Record</code> is removed from the
     * <code>OnlineStore</code>. In both cases, Feature Store appends the deleted record marker to the
     * <code>OfflineStore</code>. The deleted record marker is a record with the same <code>RecordIdentifer</code> as
     * the original, but with <code>is_deleted</code> value set to <code>True</code>, <code>EventTime</code> set to the
     * delete input <code>EventTime</code>, and other feature values set to <code>null</code>.
     * </p>
     * <p>
     * Note that the <code>EventTime</code> specified in <code>DeleteRecord</code> should be set later than the
     * <code>EventTime</code> of the existing record in the <code>OnlineStore</code> for that
     * <code>RecordIdentifer</code>. If it is not, the deletion does not occur:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>SoftDelete</code>, the existing (not deleted) record remains in the <code>OnlineStore</code>, though
     * the delete record marker is still written to the <code>OfflineStore</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HardDelete</code> returns <code>EventTime</code>: <code>400 ValidationException</code> to indicate that the
     * delete operation failed. No delete record marker is written to the <code>OfflineStore</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When a record is deleted from the <code>OnlineStore</code>, the deleted record marker is appended to the
     * <code>OfflineStore</code>. If you have the Iceberg table format enabled for your <code>OfflineStore</code>, you
     * can remove all history of a record from the <code>OfflineStore</code> using Amazon Athena or Apache Spark. For
     * information on how to hard delete a record from the <code>OfflineStore</code> with the Iceberg table format
     * enabled, see <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/feature-store-delete-records-offline-store.html#feature-store-delete-records-offline-store"
     * >Delete records from the offline store</a>.
     * </p>
     *
     * @param deleteRecordRequest
     * @return A Java Future containing the result of the DeleteRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationErrorException There was an error validating your request.</li>
     *         <li>InternalFailureException An internal failure occurred. Try your request again. If the problem
     *         persists, contact Amazon Web Services customer support.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>AccessForbiddenException You do not have permission to perform an action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerFeatureStoreRuntimeException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerFeatureStoreRuntimeAsyncClient.DeleteRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-featurestore-runtime-2020-07-01/DeleteRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRecordResponse> deleteRecord(DeleteRecordRequest deleteRecordRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRecordRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker FeatureStore Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRecordRequest, DeleteRecordResponse>()
                            .withOperationName("DeleteRecord").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRecordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRecordRequest));
            CompletableFuture<DeleteRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use for <code>OnlineStore</code> serving from a <code>FeatureStore</code>. Only the latest records stored in the
     * <code>OnlineStore</code> can be retrieved. If no Record with <code>RecordIdentifierValue</code> is found, then an
     * empty result is returned.
     * </p>
     *
     * @param getRecordRequest
     * @return A Java Future containing the result of the GetRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationErrorException There was an error validating your request.</li>
     *         <li>ResourceNotFoundException A resource that is required to perform an action was not found.</li>
     *         <li>InternalFailureException An internal failure occurred. Try your request again. If the problem
     *         persists, contact Amazon Web Services customer support.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>AccessForbiddenException You do not have permission to perform an action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerFeatureStoreRuntimeException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerFeatureStoreRuntimeAsyncClient.GetRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-featurestore-runtime-2020-07-01/GetRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecordResponse> getRecord(GetRecordRequest getRecordRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecordRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker FeatureStore Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecordRequest, GetRecordResponse>().withOperationName("GetRecord")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRecordRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRecordRequest));
            CompletableFuture<GetRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>PutRecord</code> API is used to ingest a list of <code>Records</code> into your feature group.
     * </p>
     * <p>
     * If a new record’s <code>EventTime</code> is greater, the new record is written to both the
     * <code>OnlineStore</code> and <code>OfflineStore</code>. Otherwise, the record is a historic record and it is
     * written only to the <code>OfflineStore</code>.
     * </p>
     * <p>
     * You can specify the ingestion to be applied to the <code>OnlineStore</code>, <code>OfflineStore</code>, or both
     * by using the <code>TargetStores</code> request parameter.
     * </p>
     * <p>
     * You can set the ingested record to expire at a given time to live (TTL) duration after the record’s event time,
     * <code>ExpiresAt</code> = <code>EventTime</code> + <code>TtlDuration</code>, by specifying the
     * <code>TtlDuration</code> parameter. A record level <code>TtlDuration</code> is set when specifying the
     * <code>TtlDuration</code> parameter using the <code>PutRecord</code> API call. If the input
     * <code>TtlDuration</code> is <code>null</code> or unspecified, <code>TtlDuration</code> is set to the default
     * feature group level <code>TtlDuration</code>. A record level <code>TtlDuration</code> supersedes the group level
     * <code>TtlDuration</code>.
     * </p>
     *
     * @param putRecordRequest
     * @return A Java Future containing the result of the PutRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationErrorException There was an error validating your request.</li>
     *         <li>InternalFailureException An internal failure occurred. Try your request again. If the problem
     *         persists, contact Amazon Web Services customer support.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>AccessForbiddenException You do not have permission to perform an action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerFeatureStoreRuntimeException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerFeatureStoreRuntimeAsyncClient.PutRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-featurestore-runtime-2020-07-01/PutRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRecordResponse> putRecord(PutRecordRequest putRecordRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRecordRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker FeatureStore Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRecordRequest, PutRecordResponse>().withOperationName("PutRecord")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutRecordRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(putRecordRequest));
            CompletableFuture<PutRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SageMakerFeatureStoreRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return new SageMakerFeatureStoreRuntimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SageMakerFeatureStoreRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationError")
                                .exceptionBuilderSupplier(ValidationErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailable")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessForbidden")
                                .exceptionBuilderSupplier(AccessForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailure")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SageMakerFeatureStoreRuntimeServiceClientConfigurationBuilder serviceConfigBuilder = new SageMakerFeatureStoreRuntimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
