/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rolesanywhere;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.rolesanywhere.internal.RolesAnywhereServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.rolesanywhere.model.AccessDeniedException;
import software.amazon.awssdk.services.rolesanywhere.model.CreateProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.CreateProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.CreateTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.CreateTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DisableCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DisableCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DisableProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DisableProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DisableTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DisableTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.EnableCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.EnableCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.EnableProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.EnableProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.EnableTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.EnableTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.GetCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.GetCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.GetProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.GetProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.GetSubjectRequest;
import software.amazon.awssdk.services.rolesanywhere.model.GetSubjectResponse;
import software.amazon.awssdk.services.rolesanywhere.model.GetTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.GetTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ImportCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ImportCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListCrlsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListCrlsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListProfilesRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListProfilesResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListSubjectsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListSubjectsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListTrustAnchorsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListTrustAnchorsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.PutNotificationSettingsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.PutNotificationSettingsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ResetNotificationSettingsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ResetNotificationSettingsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rolesanywhere.model.RolesAnywhereException;
import software.amazon.awssdk.services.rolesanywhere.model.TagResourceRequest;
import software.amazon.awssdk.services.rolesanywhere.model.TagResourceResponse;
import software.amazon.awssdk.services.rolesanywhere.model.TooManyTagsException;
import software.amazon.awssdk.services.rolesanywhere.model.UntagResourceRequest;
import software.amazon.awssdk.services.rolesanywhere.model.UntagResourceResponse;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ValidationException;
import software.amazon.awssdk.services.rolesanywhere.transform.CreateProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.CreateTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DeleteCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DeleteProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DeleteTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DisableCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DisableProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DisableTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.EnableCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.EnableProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.EnableTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.GetCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.GetProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.GetSubjectRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.GetTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ImportCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListCrlsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListProfilesRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListSubjectsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListTrustAnchorsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.PutNotificationSettingsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ResetNotificationSettingsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.UpdateCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.UpdateProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.UpdateTrustAnchorRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RolesAnywhereAsyncClient}.
 *
 * @see RolesAnywhereAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRolesAnywhereAsyncClient implements RolesAnywhereAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRolesAnywhereAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRolesAnywhereAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a <i>profile</i>, a list of the roles that Roles Anywhere service is trusted to assume. You use profiles
     * to intersect permissions with IAM managed policies.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:CreateProfile</code>.
     * </p>
     *
     * @param createProfileRequest
     * @return A Java Future containing the result of the CreateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.CreateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/CreateProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProfileResponse> createProfile(CreateProfileRequest createProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProfileRequest, CreateProfileResponse>()
                            .withOperationName("CreateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createProfileRequest));
            CompletableFuture<CreateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a trust anchor to establish trust between IAM Roles Anywhere and your certificate authority (CA). You can
     * define a trust anchor as a reference to an Private Certificate Authority (Private CA) or by uploading a CA
     * certificate. Your Amazon Web Services workloads can authenticate with the trust anchor using certificates issued
     * by the CA in exchange for temporary Amazon Web Services credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:CreateTrustAnchor</code>.
     * </p>
     *
     * @param createTrustAnchorRequest
     * @return A Java Future containing the result of the CreateTrustAnchor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.CreateTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/CreateTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTrustAnchorResponse> createTrustAnchor(CreateTrustAnchorRequest createTrustAnchorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTrustAnchor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTrustAnchorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTrustAnchorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTrustAnchorRequest, CreateTrustAnchorResponse>()
                            .withOperationName("CreateTrustAnchor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTrustAnchorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTrustAnchorRequest));
            CompletableFuture<CreateTrustAnchorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a certificate revocation list (CRL).
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DeleteCrl</code>.
     * </p>
     *
     * @param deleteCrlRequest
     * @return A Java Future containing the result of the DeleteCrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.DeleteCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DeleteCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCrlResponse> deleteCrl(DeleteCrlRequest deleteCrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteCrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCrlRequest, DeleteCrlResponse>().withOperationName("DeleteCrl")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCrlRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteCrlRequest));
            CompletableFuture<DeleteCrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a profile.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DeleteProfile</code>.
     * </p>
     *
     * @param deleteProfileRequest
     * @return A Java Future containing the result of the DeleteProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.DeleteProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DeleteProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProfileResponse> deleteProfile(DeleteProfileRequest deleteProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProfileRequest, DeleteProfileResponse>()
                            .withOperationName("DeleteProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteProfileRequest));
            CompletableFuture<DeleteProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a trust anchor.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DeleteTrustAnchor</code>.
     * </p>
     *
     * @param deleteTrustAnchorRequest
     * @return A Java Future containing the result of the DeleteTrustAnchor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.DeleteTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DeleteTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTrustAnchorResponse> deleteTrustAnchor(DeleteTrustAnchorRequest deleteTrustAnchorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTrustAnchor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTrustAnchorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTrustAnchorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTrustAnchorRequest, DeleteTrustAnchorResponse>()
                            .withOperationName("DeleteTrustAnchor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTrustAnchorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTrustAnchorRequest));
            CompletableFuture<DeleteTrustAnchorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables a certificate revocation list (CRL).
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DisableCrl</code>.
     * </p>
     *
     * @param disableCrlRequest
     * @return A Java Future containing the result of the DisableCrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.DisableCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DisableCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisableCrlResponse> disableCrl(DisableCrlRequest disableCrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableCrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DisableCrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableCrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableCrlRequest, DisableCrlResponse>().withOperationName("DisableCrl")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisableCrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableCrlRequest));
            CompletableFuture<DisableCrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables a profile. When disabled, temporary credential requests with this profile fail.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DisableProfile</code>.
     * </p>
     *
     * @param disableProfileRequest
     * @return A Java Future containing the result of the DisableProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.DisableProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DisableProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableProfileResponse> disableProfile(DisableProfileRequest disableProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableProfileRequest, DisableProfileResponse>()
                            .withOperationName("DisableProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisableProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableProfileRequest));
            CompletableFuture<DisableProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables a trust anchor. When disabled, temporary credential requests specifying this trust anchor are
     * unauthorized.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DisableTrustAnchor</code>.
     * </p>
     *
     * @param disableTrustAnchorRequest
     * @return A Java Future containing the result of the DisableTrustAnchor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.DisableTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DisableTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableTrustAnchorResponse> disableTrustAnchor(DisableTrustAnchorRequest disableTrustAnchorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableTrustAnchor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableTrustAnchorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableTrustAnchorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableTrustAnchorRequest, DisableTrustAnchorResponse>()
                            .withOperationName("DisableTrustAnchor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisableTrustAnchorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableTrustAnchorRequest));
            CompletableFuture<DisableTrustAnchorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables a certificate revocation list (CRL). When enabled, certificates stored in the CRL are unauthorized to
     * receive session credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:EnableCrl</code>.
     * </p>
     *
     * @param enableCrlRequest
     * @return A Java Future containing the result of the EnableCrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.EnableCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/EnableCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EnableCrlResponse> enableCrl(EnableCrlRequest enableCrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableCrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableCrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableCrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableCrlRequest, EnableCrlResponse>().withOperationName("EnableCrl")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EnableCrlRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableCrlRequest));
            CompletableFuture<EnableCrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables temporary credential requests for a profile.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:EnableProfile</code>.
     * </p>
     *
     * @param enableProfileRequest
     * @return A Java Future containing the result of the EnableProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.EnableProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/EnableProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableProfileResponse> enableProfile(EnableProfileRequest enableProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableProfileRequest, EnableProfileResponse>()
                            .withOperationName("EnableProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EnableProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableProfileRequest));
            CompletableFuture<EnableProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables a trust anchor. When enabled, certificates in the trust anchor chain are authorized for trust validation.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:EnableTrustAnchor</code>.
     * </p>
     *
     * @param enableTrustAnchorRequest
     * @return A Java Future containing the result of the EnableTrustAnchor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.EnableTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/EnableTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableTrustAnchorResponse> enableTrustAnchor(EnableTrustAnchorRequest enableTrustAnchorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableTrustAnchor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableTrustAnchorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableTrustAnchorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableTrustAnchorRequest, EnableTrustAnchorResponse>()
                            .withOperationName("EnableTrustAnchor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EnableTrustAnchorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableTrustAnchorRequest));
            CompletableFuture<EnableTrustAnchorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a certificate revocation list (CRL).
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:GetCrl</code>.
     * </p>
     *
     * @param getCrlRequest
     * @return A Java Future containing the result of the GetCrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.GetCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/GetCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCrlResponse> getCrl(GetCrlRequest getCrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCrlRequest, GetCrlResponse>().withOperationName("GetCrl")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetCrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCrlRequest));
            CompletableFuture<GetCrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a profile.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:GetProfile</code>.
     * </p>
     *
     * @param getProfileRequest
     * @return A Java Future containing the result of the GetProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.GetProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/GetProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProfileResponse> getProfile(GetProfileRequest getProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProfileRequest, GetProfileResponse>().withOperationName("GetProfile")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProfileRequest));
            CompletableFuture<GetProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a <i>subject</i>, which associates a certificate identity with authentication attempts. The subject stores
     * auditing information such as the status of the last authentication attempt, the certificate data used in the
     * attempt, and the last time the associated identity attempted authentication.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:GetSubject</code>.
     * </p>
     *
     * @param getSubjectRequest
     * @return A Java Future containing the result of the GetSubject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.GetSubject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/GetSubject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSubjectResponse> getSubject(GetSubjectRequest getSubjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSubjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSubjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSubject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSubjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSubjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSubjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSubjectRequest, GetSubjectResponse>().withOperationName("GetSubject")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSubjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSubjectRequest));
            CompletableFuture<GetSubjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a trust anchor.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:GetTrustAnchor</code>.
     * </p>
     *
     * @param getTrustAnchorRequest
     * @return A Java Future containing the result of the GetTrustAnchor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.GetTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/GetTrustAnchor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTrustAnchorResponse> getTrustAnchor(GetTrustAnchorRequest getTrustAnchorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTrustAnchorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTrustAnchor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTrustAnchorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTrustAnchorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTrustAnchorRequest, GetTrustAnchorResponse>()
                            .withOperationName("GetTrustAnchor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTrustAnchorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTrustAnchorRequest));
            CompletableFuture<GetTrustAnchorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports the certificate revocation list (CRL). A CRL is a list of certificates that have been revoked by the
     * issuing certificate Authority (CA). IAM Roles Anywhere validates against the CRL before issuing credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ImportCrl</code>.
     * </p>
     *
     * @param importCrlRequest
     * @return A Java Future containing the result of the ImportCrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.ImportCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ImportCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ImportCrlResponse> importCrl(ImportCrlRequest importCrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportCrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ImportCrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportCrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportCrlRequest, ImportCrlResponse>().withOperationName("ImportCrl")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ImportCrlRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(importCrlRequest));
            CompletableFuture<ImportCrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all certificate revocation lists (CRL) in the authenticated account and Amazon Web Services Region.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListCrls</code>.
     * </p>
     *
     * @param listCrlsRequest
     * @return A Java Future containing the result of the ListCrls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.ListCrls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListCrls" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListCrlsResponse> listCrls(ListCrlsRequest listCrlsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCrlsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCrlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCrls");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCrlsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListCrlsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCrlsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCrlsRequest, ListCrlsResponse>().withOperationName("ListCrls")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCrlsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCrlsRequest));
            CompletableFuture<ListCrlsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all profiles in the authenticated account and Amazon Web Services Region.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListProfiles</code>.
     * </p>
     *
     * @param listProfilesRequest
     * @return A Java Future containing the result of the ListProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.ListProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfilesResponse> listProfiles(ListProfilesRequest listProfilesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfilesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfilesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfilesRequest, ListProfilesResponse>()
                            .withOperationName("ListProfiles").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProfilesRequest));
            CompletableFuture<ListProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the subjects in the authenticated account and Amazon Web Services Region.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListSubjects</code>.
     * </p>
     *
     * @param listSubjectsRequest
     * @return A Java Future containing the result of the ListSubjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.ListSubjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListSubjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSubjectsResponse> listSubjects(ListSubjectsRequest listSubjectsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSubjectsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSubjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSubjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSubjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSubjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSubjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSubjectsRequest, ListSubjectsResponse>()
                            .withOperationName("ListSubjects").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSubjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSubjectsRequest));
            CompletableFuture<ListSubjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags attached to the resource.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListTagsForResource</code>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the trust anchors in the authenticated account and Amazon Web Services Region.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListTrustAnchors</code>.
     * </p>
     *
     * @param listTrustAnchorsRequest
     * @return A Java Future containing the result of the ListTrustAnchors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.ListTrustAnchors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListTrustAnchors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTrustAnchorsResponse> listTrustAnchors(ListTrustAnchorsRequest listTrustAnchorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTrustAnchorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrustAnchorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrustAnchors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTrustAnchorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTrustAnchorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTrustAnchorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTrustAnchorsRequest, ListTrustAnchorsResponse>()
                            .withOperationName("ListTrustAnchors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTrustAnchorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTrustAnchorsRequest));
            CompletableFuture<ListTrustAnchorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches a list of <i>notification settings</i> to a trust anchor.
     * </p>
     * <p>
     * A notification setting includes information such as event name, threshold, status of the notification setting,
     * and the channel to notify.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:PutNotificationSettings</code>.
     * </p>
     *
     * @param putNotificationSettingsRequest
     * @return A Java Future containing the result of the PutNotificationSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.PutNotificationSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/PutNotificationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutNotificationSettingsResponse> putNotificationSettings(
            PutNotificationSettingsRequest putNotificationSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putNotificationSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putNotificationSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutNotificationSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutNotificationSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutNotificationSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutNotificationSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutNotificationSettingsRequest, PutNotificationSettingsResponse>()
                            .withOperationName("PutNotificationSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutNotificationSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putNotificationSettingsRequest));
            CompletableFuture<PutNotificationSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resets the <i>custom notification setting</i> to IAM Roles Anywhere default setting.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ResetNotificationSettings</code>.
     * </p>
     *
     * @param resetNotificationSettingsRequest
     * @return A Java Future containing the result of the ResetNotificationSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.ResetNotificationSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ResetNotificationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ResetNotificationSettingsResponse> resetNotificationSettings(
            ResetNotificationSettingsRequest resetNotificationSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetNotificationSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetNotificationSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetNotificationSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResetNotificationSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResetNotificationSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResetNotificationSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResetNotificationSettingsRequest, ResetNotificationSettingsResponse>()
                            .withOperationName("ResetNotificationSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ResetNotificationSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(resetNotificationSettingsRequest));
            CompletableFuture<ResetNotificationSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches tags to a resource.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:TagResource</code>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>TooManyTagsException Too many tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the resource.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:UntagResource</code>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the certificate revocation list (CRL). A CRL is a list of certificates that have been revoked by the
     * issuing certificate authority (CA). IAM Roles Anywhere validates against the CRL before issuing credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:UpdateCrl</code>.
     * </p>
     *
     * @param updateCrlRequest
     * @return A Java Future containing the result of the UpdateCrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.UpdateCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/UpdateCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCrlResponse> updateCrl(UpdateCrlRequest updateCrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateCrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCrlRequest, UpdateCrlResponse>().withOperationName("UpdateCrl")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateCrlRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateCrlRequest));
            CompletableFuture<UpdateCrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a <i>profile</i>, a list of the roles that IAM Roles Anywhere service is trusted to assume. You use
     * profiles to intersect permissions with IAM managed policies.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:UpdateProfile</code>.
     * </p>
     *
     * @param updateProfileRequest
     * @return A Java Future containing the result of the UpdateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.UpdateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/UpdateProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProfileResponse> updateProfile(UpdateProfileRequest updateProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProfileRequest, UpdateProfileResponse>()
                            .withOperationName("UpdateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateProfileRequest));
            CompletableFuture<UpdateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a trust anchor. You establish trust between IAM Roles Anywhere and your certificate authority (CA) by
     * configuring a trust anchor. You can define a trust anchor as a reference to an Private Certificate Authority
     * (Private CA) or by uploading a CA certificate. Your Amazon Web Services workloads can authenticate with the trust
     * anchor using certificates issued by the CA in exchange for temporary Amazon Web Services credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:UpdateTrustAnchor</code>.
     * </p>
     *
     * @param updateTrustAnchorRequest
     * @return A Java Future containing the result of the UpdateTrustAnchor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RolesAnywhereException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RolesAnywhereAsyncClient.UpdateTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/UpdateTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTrustAnchorResponse> updateTrustAnchor(UpdateTrustAnchorRequest updateTrustAnchorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTrustAnchor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTrustAnchorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTrustAnchorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTrustAnchorRequest, UpdateTrustAnchorResponse>()
                            .withOperationName("UpdateTrustAnchor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateTrustAnchorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTrustAnchorRequest));
            CompletableFuture<UpdateTrustAnchorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final RolesAnywhereServiceClientConfiguration serviceClientConfiguration() {
        return new RolesAnywhereServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RolesAnywhereException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        RolesAnywhereServiceClientConfigurationBuilder serviceConfigBuilder = new RolesAnywhereServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
