/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resiliencehub.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines a recommendation template created with the <a>CreateRecommendationTemplate</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecommendationTemplate implements SdkPojo, Serializable,
        ToCopyableBuilder<RecommendationTemplate.Builder, RecommendationTemplate> {
    private static final SdkField<String> APP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("appArn")
            .getter(getter(RecommendationTemplate::appArn)).setter(setter(Builder::appArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appArn").build()).build();

    private static final SdkField<String> ASSESSMENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("assessmentArn").getter(getter(RecommendationTemplate::assessmentArn))
            .setter(setter(Builder::assessmentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("assessmentArn").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(RecommendationTemplate::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("format")
            .getter(getter(RecommendationTemplate::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("format").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(RecommendationTemplate::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(RecommendationTemplate::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Boolean> NEEDS_REPLACEMENTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("needsReplacements").getter(getter(RecommendationTemplate::needsReplacements))
            .setter(setter(Builder::needsReplacements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("needsReplacements").build()).build();

    private static final SdkField<List<String>> RECOMMENDATION_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("recommendationIds")
            .getter(getter(RecommendationTemplate::recommendationIds))
            .setter(setter(Builder::recommendationIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RECOMMENDATION_TEMPLATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("recommendationTemplateArn").getter(getter(RecommendationTemplate::recommendationTemplateArn))
            .setter(setter(Builder::recommendationTemplateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationTemplateArn").build())
            .build();

    private static final SdkField<List<String>> RECOMMENDATION_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("recommendationTypes")
            .getter(getter(RecommendationTemplate::recommendationTypesAsStrings))
            .setter(setter(Builder::recommendationTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(RecommendationTemplate::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(RecommendationTemplate::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(RecommendationTemplate::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<S3Location> TEMPLATES_LOCATION_FIELD = SdkField.<S3Location> builder(MarshallingType.SDK_POJO)
            .memberName("templatesLocation").getter(getter(RecommendationTemplate::templatesLocation))
            .setter(setter(Builder::templatesLocation)).constructor(S3Location::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("templatesLocation").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APP_ARN_FIELD,
            ASSESSMENT_ARN_FIELD, END_TIME_FIELD, FORMAT_FIELD, MESSAGE_FIELD, NAME_FIELD, NEEDS_REPLACEMENTS_FIELD,
            RECOMMENDATION_IDS_FIELD, RECOMMENDATION_TEMPLATE_ARN_FIELD, RECOMMENDATION_TYPES_FIELD, START_TIME_FIELD,
            STATUS_FIELD, TAGS_FIELD, TEMPLATES_LOCATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String appArn;

    private final String assessmentArn;

    private final Instant endTime;

    private final String format;

    private final String message;

    private final String name;

    private final Boolean needsReplacements;

    private final List<String> recommendationIds;

    private final String recommendationTemplateArn;

    private final List<String> recommendationTypes;

    private final Instant startTime;

    private final String status;

    private final Map<String, String> tags;

    private final S3Location templatesLocation;

    private RecommendationTemplate(BuilderImpl builder) {
        this.appArn = builder.appArn;
        this.assessmentArn = builder.assessmentArn;
        this.endTime = builder.endTime;
        this.format = builder.format;
        this.message = builder.message;
        this.name = builder.name;
        this.needsReplacements = builder.needsReplacements;
        this.recommendationIds = builder.recommendationIds;
        this.recommendationTemplateArn = builder.recommendationTemplateArn;
        this.recommendationTypes = builder.recommendationTypes;
        this.startTime = builder.startTime;
        this.status = builder.status;
        this.tags = builder.tags;
        this.templatesLocation = builder.templatesLocation;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the application. The format for this ARN is: arn:<code>partition</code>:dcps:
     * <code>region</code>:<code>account</code>:app/<code>app-id</code>. For more information about ARNs, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource Names
     * (ARNs)</a> in the <i>AWS General Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the application. The format for this ARN is: arn:<code>partition</code>
     *         :dcps:<code>region</code>:<code>account</code>:app/<code>app-id</code>. For more information about ARNs,
     *         see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource
     *         Names (ARNs)</a> in the <i>AWS General Reference</i>.
     */
    public final String appArn() {
        return appArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the assessment. The format for this ARN is: arn:<code>partition</code>:dcps:
     * <code>region</code>:<code>account</code>:app-assessment/<code>app-id</code>. For more information about ARNs, see
     * <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource Names
     * (ARNs)</a> in the <i>AWS General Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the assessment. The format for this ARN is: arn:<code>partition</code>
     *         :dcps:<code>region</code>:<code>account</code>:app-assessment/<code>app-id</code>. For more information
     *         about ARNs, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
     *         Amazon Resource Names (ARNs)</a> in the <i>AWS General Reference</i>.
     */
    public final String assessmentArn() {
        return assessmentArn;
    }

    /**
     * <p>
     * The end time for the action.
     * </p>
     * 
     * @return The end time for the action.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The format of the recommendation template.
     * </p>
     * <dl>
     * <dt>CfnJson</dt>
     * <dd>
     * <p>
     * The template is CloudFormation JSON.
     * </p>
     * </dd>
     * <dt>CfnYaml</dt>
     * <dd>
     * <p>
     * The template is CloudFormation YAML.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link TemplateFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the recommendation template.</p>
     *         <dl>
     *         <dt>CfnJson</dt>
     *         <dd>
     *         <p>
     *         The template is CloudFormation JSON.
     *         </p>
     *         </dd>
     *         <dt>CfnYaml</dt>
     *         <dd>
     *         <p>
     *         The template is CloudFormation YAML.
     *         </p>
     *         </dd>
     * @see TemplateFormat
     */
    public final TemplateFormat format() {
        return TemplateFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the recommendation template.
     * </p>
     * <dl>
     * <dt>CfnJson</dt>
     * <dd>
     * <p>
     * The template is CloudFormation JSON.
     * </p>
     * </dd>
     * <dt>CfnYaml</dt>
     * <dd>
     * <p>
     * The template is CloudFormation YAML.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link TemplateFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the recommendation template.</p>
     *         <dl>
     *         <dt>CfnJson</dt>
     *         <dd>
     *         <p>
     *         The template is CloudFormation JSON.
     *         </p>
     *         </dd>
     *         <dt>CfnYaml</dt>
     *         <dd>
     *         <p>
     *         The template is CloudFormation YAML.
     *         </p>
     *         </dd>
     * @see TemplateFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The message for the recommendation template.
     * </p>
     * 
     * @return The message for the recommendation template.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * The name for the recommendation template.
     * </p>
     * 
     * @return The name for the recommendation template.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Indicates if replacements are needed.
     * </p>
     * 
     * @return Indicates if replacements are needed.
     */
    public final Boolean needsReplacements() {
        return needsReplacements;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendationIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendationIds() {
        return recommendationIds != null && !(recommendationIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Identifiers for the recommendations used in the recommendation template.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationIds} method.
     * </p>
     * 
     * @return Identifiers for the recommendations used in the recommendation template.
     */
    public final List<String> recommendationIds() {
        return recommendationIds;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the recommendation template.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the recommendation template.
     */
    public final String recommendationTemplateArn() {
        return recommendationTemplateArn;
    }

    /**
     * <p>
     * An array of strings that specify the recommendation template type or types.
     * </p>
     * <dl>
     * <dt>Alarm</dt>
     * <dd>
     * <p>
     * The template is an <a>AlarmRecommendation</a> template.
     * </p>
     * </dd>
     * <dt>Sop</dt>
     * <dd>
     * <p>
     * The template is a <a>SopRecommendation</a> template.
     * </p>
     * </dd>
     * <dt>Test</dt>
     * <dd>
     * <p>
     * The template is a <a>TestRecommendation</a> template.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationTypes} method.
     * </p>
     * 
     * @return An array of strings that specify the recommendation template type or types.</p>
     *         <dl>
     *         <dt>Alarm</dt>
     *         <dd>
     *         <p>
     *         The template is an <a>AlarmRecommendation</a> template.
     *         </p>
     *         </dd>
     *         <dt>Sop</dt>
     *         <dd>
     *         <p>
     *         The template is a <a>SopRecommendation</a> template.
     *         </p>
     *         </dd>
     *         <dt>Test</dt>
     *         <dd>
     *         <p>
     *         The template is a <a>TestRecommendation</a> template.
     *         </p>
     *         </dd>
     */
    public final List<RenderRecommendationType> recommendationTypes() {
        return RenderRecommendationTypeListCopier.copyStringToEnum(recommendationTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendationTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendationTypes() {
        return recommendationTypes != null && !(recommendationTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of strings that specify the recommendation template type or types.
     * </p>
     * <dl>
     * <dt>Alarm</dt>
     * <dd>
     * <p>
     * The template is an <a>AlarmRecommendation</a> template.
     * </p>
     * </dd>
     * <dt>Sop</dt>
     * <dd>
     * <p>
     * The template is a <a>SopRecommendation</a> template.
     * </p>
     * </dd>
     * <dt>Test</dt>
     * <dd>
     * <p>
     * The template is a <a>TestRecommendation</a> template.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationTypes} method.
     * </p>
     * 
     * @return An array of strings that specify the recommendation template type or types.</p>
     *         <dl>
     *         <dt>Alarm</dt>
     *         <dd>
     *         <p>
     *         The template is an <a>AlarmRecommendation</a> template.
     *         </p>
     *         </dd>
     *         <dt>Sop</dt>
     *         <dd>
     *         <p>
     *         The template is a <a>SopRecommendation</a> template.
     *         </p>
     *         </dd>
     *         <dt>Test</dt>
     *         <dd>
     *         <p>
     *         The template is a <a>TestRecommendation</a> template.
     *         </p>
     *         </dd>
     */
    public final List<String> recommendationTypesAsStrings() {
        return recommendationTypes;
    }

    /**
     * <p>
     * The start time for the action.
     * </p>
     * 
     * @return The start time for the action.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The status of the action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link RecommendationTemplateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the action.
     * @see RecommendationTemplateStatus
     */
    public final RecommendationTemplateStatus status() {
        return RecommendationTemplateStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link RecommendationTemplateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the action.
     * @see RecommendationTemplateStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags assigned to the resource. A tag is a label that you assign to an Amazon Web Services resource. Each tag
     * consists of a key/value pair.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the resource. A tag is a label that you assign to an Amazon Web Services resource.
     *         Each tag consists of a key/value pair.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The file location of the template.
     * </p>
     * 
     * @return The file location of the template.
     */
    public final S3Location templatesLocation() {
        return templatesLocation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(appArn());
        hashCode = 31 * hashCode + Objects.hashCode(assessmentArn());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(needsReplacements());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendationIds() ? recommendationIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recommendationTemplateArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendationTypes() ? recommendationTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(templatesLocation());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecommendationTemplate)) {
            return false;
        }
        RecommendationTemplate other = (RecommendationTemplate) obj;
        return Objects.equals(appArn(), other.appArn()) && Objects.equals(assessmentArn(), other.assessmentArn())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(message(), other.message()) && Objects.equals(name(), other.name())
                && Objects.equals(needsReplacements(), other.needsReplacements())
                && hasRecommendationIds() == other.hasRecommendationIds()
                && Objects.equals(recommendationIds(), other.recommendationIds())
                && Objects.equals(recommendationTemplateArn(), other.recommendationTemplateArn())
                && hasRecommendationTypes() == other.hasRecommendationTypes()
                && Objects.equals(recommendationTypesAsStrings(), other.recommendationTypesAsStrings())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(statusAsString(), other.statusAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(templatesLocation(), other.templatesLocation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecommendationTemplate").add("AppArn", appArn()).add("AssessmentArn", assessmentArn())
                .add("EndTime", endTime()).add("Format", formatAsString()).add("Message", message()).add("Name", name())
                .add("NeedsReplacements", needsReplacements())
                .add("RecommendationIds", hasRecommendationIds() ? recommendationIds() : null)
                .add("RecommendationTemplateArn", recommendationTemplateArn())
                .add("RecommendationTypes", hasRecommendationTypes() ? recommendationTypesAsStrings() : null)
                .add("StartTime", startTime()).add("Status", statusAsString())
                .add("Tags", tags() == null ? null : "*** Sensitive Data Redacted ***")
                .add("TemplatesLocation", templatesLocation()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "appArn":
            return Optional.ofNullable(clazz.cast(appArn()));
        case "assessmentArn":
            return Optional.ofNullable(clazz.cast(assessmentArn()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "needsReplacements":
            return Optional.ofNullable(clazz.cast(needsReplacements()));
        case "recommendationIds":
            return Optional.ofNullable(clazz.cast(recommendationIds()));
        case "recommendationTemplateArn":
            return Optional.ofNullable(clazz.cast(recommendationTemplateArn()));
        case "recommendationTypes":
            return Optional.ofNullable(clazz.cast(recommendationTypesAsStrings()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "templatesLocation":
            return Optional.ofNullable(clazz.cast(templatesLocation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecommendationTemplate, T> g) {
        return obj -> g.apply((RecommendationTemplate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecommendationTemplate> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the application. The format for this ARN is: arn:<code>partition</code>
         * :dcps:<code>region</code>:<code>account</code>:app/<code>app-id</code>. For more information about ARNs, see
         * <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource Names
         * (ARNs)</a> in the <i>AWS General Reference</i>.
         * </p>
         * 
         * @param appArn
         *        The Amazon Resource Name (ARN) of the application. The format for this ARN is: arn:
         *        <code>partition</code>:dcps:<code>region</code>:<code>account</code>:app/<code>app-id</code>. For more
         *        information about ARNs, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource
         *        Names (ARNs)</a> in the <i>AWS General Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appArn(String appArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the assessment. The format for this ARN is: arn:<code>partition</code>
         * :dcps:<code>region</code>:<code>account</code>:app-assessment/<code>app-id</code>. For more information about
         * ARNs, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon
         * Resource Names (ARNs)</a> in the <i>AWS General Reference</i>.
         * </p>
         * 
         * @param assessmentArn
         *        The Amazon Resource Name (ARN) of the assessment. The format for this ARN is: arn:
         *        <code>partition</code>:dcps:<code>region</code>:<code>account</code>:app-assessment/
         *        <code>app-id</code>. For more information about ARNs, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource
         *        Names (ARNs)</a> in the <i>AWS General Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assessmentArn(String assessmentArn);

        /**
         * <p>
         * The end time for the action.
         * </p>
         * 
         * @param endTime
         *        The end time for the action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The format of the recommendation template.
         * </p>
         * <dl>
         * <dt>CfnJson</dt>
         * <dd>
         * <p>
         * The template is CloudFormation JSON.
         * </p>
         * </dd>
         * <dt>CfnYaml</dt>
         * <dd>
         * <p>
         * The template is CloudFormation YAML.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param format
         *        The format of the recommendation template.</p>
         *        <dl>
         *        <dt>CfnJson</dt>
         *        <dd>
         *        <p>
         *        The template is CloudFormation JSON.
         *        </p>
         *        </dd>
         *        <dt>CfnYaml</dt>
         *        <dd>
         *        <p>
         *        The template is CloudFormation YAML.
         *        </p>
         *        </dd>
         * @see TemplateFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TemplateFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the recommendation template.
         * </p>
         * <dl>
         * <dt>CfnJson</dt>
         * <dd>
         * <p>
         * The template is CloudFormation JSON.
         * </p>
         * </dd>
         * <dt>CfnYaml</dt>
         * <dd>
         * <p>
         * The template is CloudFormation YAML.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param format
         *        The format of the recommendation template.</p>
         *        <dl>
         *        <dt>CfnJson</dt>
         *        <dd>
         *        <p>
         *        The template is CloudFormation JSON.
         *        </p>
         *        </dd>
         *        <dt>CfnYaml</dt>
         *        <dd>
         *        <p>
         *        The template is CloudFormation YAML.
         *        </p>
         *        </dd>
         * @see TemplateFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TemplateFormat
         */
        Builder format(TemplateFormat format);

        /**
         * <p>
         * The message for the recommendation template.
         * </p>
         * 
         * @param message
         *        The message for the recommendation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The name for the recommendation template.
         * </p>
         * 
         * @param name
         *        The name for the recommendation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Indicates if replacements are needed.
         * </p>
         * 
         * @param needsReplacements
         *        Indicates if replacements are needed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder needsReplacements(Boolean needsReplacements);

        /**
         * <p>
         * Identifiers for the recommendations used in the recommendation template.
         * </p>
         * 
         * @param recommendationIds
         *        Identifiers for the recommendations used in the recommendation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationIds(Collection<String> recommendationIds);

        /**
         * <p>
         * Identifiers for the recommendations used in the recommendation template.
         * </p>
         * 
         * @param recommendationIds
         *        Identifiers for the recommendations used in the recommendation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationIds(String... recommendationIds);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the recommendation template.
         * </p>
         * 
         * @param recommendationTemplateArn
         *        The Amazon Resource Name (ARN) for the recommendation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationTemplateArn(String recommendationTemplateArn);

        /**
         * <p>
         * An array of strings that specify the recommendation template type or types.
         * </p>
         * <dl>
         * <dt>Alarm</dt>
         * <dd>
         * <p>
         * The template is an <a>AlarmRecommendation</a> template.
         * </p>
         * </dd>
         * <dt>Sop</dt>
         * <dd>
         * <p>
         * The template is a <a>SopRecommendation</a> template.
         * </p>
         * </dd>
         * <dt>Test</dt>
         * <dd>
         * <p>
         * The template is a <a>TestRecommendation</a> template.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param recommendationTypes
         *        An array of strings that specify the recommendation template type or types.</p>
         *        <dl>
         *        <dt>Alarm</dt>
         *        <dd>
         *        <p>
         *        The template is an <a>AlarmRecommendation</a> template.
         *        </p>
         *        </dd>
         *        <dt>Sop</dt>
         *        <dd>
         *        <p>
         *        The template is a <a>SopRecommendation</a> template.
         *        </p>
         *        </dd>
         *        <dt>Test</dt>
         *        <dd>
         *        <p>
         *        The template is a <a>TestRecommendation</a> template.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationTypesWithStrings(Collection<String> recommendationTypes);

        /**
         * <p>
         * An array of strings that specify the recommendation template type or types.
         * </p>
         * <dl>
         * <dt>Alarm</dt>
         * <dd>
         * <p>
         * The template is an <a>AlarmRecommendation</a> template.
         * </p>
         * </dd>
         * <dt>Sop</dt>
         * <dd>
         * <p>
         * The template is a <a>SopRecommendation</a> template.
         * </p>
         * </dd>
         * <dt>Test</dt>
         * <dd>
         * <p>
         * The template is a <a>TestRecommendation</a> template.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param recommendationTypes
         *        An array of strings that specify the recommendation template type or types.</p>
         *        <dl>
         *        <dt>Alarm</dt>
         *        <dd>
         *        <p>
         *        The template is an <a>AlarmRecommendation</a> template.
         *        </p>
         *        </dd>
         *        <dt>Sop</dt>
         *        <dd>
         *        <p>
         *        The template is a <a>SopRecommendation</a> template.
         *        </p>
         *        </dd>
         *        <dt>Test</dt>
         *        <dd>
         *        <p>
         *        The template is a <a>TestRecommendation</a> template.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationTypesWithStrings(String... recommendationTypes);

        /**
         * <p>
         * An array of strings that specify the recommendation template type or types.
         * </p>
         * <dl>
         * <dt>Alarm</dt>
         * <dd>
         * <p>
         * The template is an <a>AlarmRecommendation</a> template.
         * </p>
         * </dd>
         * <dt>Sop</dt>
         * <dd>
         * <p>
         * The template is a <a>SopRecommendation</a> template.
         * </p>
         * </dd>
         * <dt>Test</dt>
         * <dd>
         * <p>
         * The template is a <a>TestRecommendation</a> template.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param recommendationTypes
         *        An array of strings that specify the recommendation template type or types.</p>
         *        <dl>
         *        <dt>Alarm</dt>
         *        <dd>
         *        <p>
         *        The template is an <a>AlarmRecommendation</a> template.
         *        </p>
         *        </dd>
         *        <dt>Sop</dt>
         *        <dd>
         *        <p>
         *        The template is a <a>SopRecommendation</a> template.
         *        </p>
         *        </dd>
         *        <dt>Test</dt>
         *        <dd>
         *        <p>
         *        The template is a <a>TestRecommendation</a> template.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationTypes(Collection<RenderRecommendationType> recommendationTypes);

        /**
         * <p>
         * An array of strings that specify the recommendation template type or types.
         * </p>
         * <dl>
         * <dt>Alarm</dt>
         * <dd>
         * <p>
         * The template is an <a>AlarmRecommendation</a> template.
         * </p>
         * </dd>
         * <dt>Sop</dt>
         * <dd>
         * <p>
         * The template is a <a>SopRecommendation</a> template.
         * </p>
         * </dd>
         * <dt>Test</dt>
         * <dd>
         * <p>
         * The template is a <a>TestRecommendation</a> template.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param recommendationTypes
         *        An array of strings that specify the recommendation template type or types.</p>
         *        <dl>
         *        <dt>Alarm</dt>
         *        <dd>
         *        <p>
         *        The template is an <a>AlarmRecommendation</a> template.
         *        </p>
         *        </dd>
         *        <dt>Sop</dt>
         *        <dd>
         *        <p>
         *        The template is a <a>SopRecommendation</a> template.
         *        </p>
         *        </dd>
         *        <dt>Test</dt>
         *        <dd>
         *        <p>
         *        The template is a <a>TestRecommendation</a> template.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationTypes(RenderRecommendationType... recommendationTypes);

        /**
         * <p>
         * The start time for the action.
         * </p>
         * 
         * @param startTime
         *        The start time for the action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The status of the action.
         * </p>
         * 
         * @param status
         *        The status of the action.
         * @see RecommendationTemplateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationTemplateStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the action.
         * </p>
         * 
         * @param status
         *        The status of the action.
         * @see RecommendationTemplateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationTemplateStatus
         */
        Builder status(RecommendationTemplateStatus status);

        /**
         * <p>
         * The tags assigned to the resource. A tag is a label that you assign to an Amazon Web Services resource. Each
         * tag consists of a key/value pair.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the resource. A tag is a label that you assign to an Amazon Web Services
         *        resource. Each tag consists of a key/value pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The file location of the template.
         * </p>
         * 
         * @param templatesLocation
         *        The file location of the template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templatesLocation(S3Location templatesLocation);

        /**
         * <p>
         * The file location of the template.
         * </p>
         * This is a convenience that creates an instance of the {@link S3Location.Builder} avoiding the need to create
         * one manually via {@link S3Location#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Location.Builder#build()} is called immediately and its result
         * is passed to {@link #templatesLocation(S3Location)}.
         * 
         * @param templatesLocation
         *        a consumer that will call methods on {@link S3Location.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #templatesLocation(S3Location)
         */
        default Builder templatesLocation(Consumer<S3Location.Builder> templatesLocation) {
            return templatesLocation(S3Location.builder().applyMutation(templatesLocation).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String appArn;

        private String assessmentArn;

        private Instant endTime;

        private String format;

        private String message;

        private String name;

        private Boolean needsReplacements;

        private List<String> recommendationIds = DefaultSdkAutoConstructList.getInstance();

        private String recommendationTemplateArn;

        private List<String> recommendationTypes = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private String status;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private S3Location templatesLocation;

        private BuilderImpl() {
        }

        private BuilderImpl(RecommendationTemplate model) {
            appArn(model.appArn);
            assessmentArn(model.assessmentArn);
            endTime(model.endTime);
            format(model.format);
            message(model.message);
            name(model.name);
            needsReplacements(model.needsReplacements);
            recommendationIds(model.recommendationIds);
            recommendationTemplateArn(model.recommendationTemplateArn);
            recommendationTypesWithStrings(model.recommendationTypes);
            startTime(model.startTime);
            status(model.status);
            tags(model.tags);
            templatesLocation(model.templatesLocation);
        }

        public final String getAppArn() {
            return appArn;
        }

        public final void setAppArn(String appArn) {
            this.appArn = appArn;
        }

        @Override
        @Transient
        public final Builder appArn(String appArn) {
            this.appArn = appArn;
            return this;
        }

        public final String getAssessmentArn() {
            return assessmentArn;
        }

        public final void setAssessmentArn(String assessmentArn) {
            this.assessmentArn = assessmentArn;
        }

        @Override
        @Transient
        public final Builder assessmentArn(String assessmentArn) {
            this.assessmentArn = assessmentArn;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        @Transient
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        @Transient
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        @Transient
        public final Builder format(TemplateFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        @Transient
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Boolean getNeedsReplacements() {
            return needsReplacements;
        }

        public final void setNeedsReplacements(Boolean needsReplacements) {
            this.needsReplacements = needsReplacements;
        }

        @Override
        @Transient
        public final Builder needsReplacements(Boolean needsReplacements) {
            this.needsReplacements = needsReplacements;
            return this;
        }

        public final Collection<String> getRecommendationIds() {
            if (recommendationIds instanceof SdkAutoConstructList) {
                return null;
            }
            return recommendationIds;
        }

        public final void setRecommendationIds(Collection<String> recommendationIds) {
            this.recommendationIds = RecommendationIdListCopier.copy(recommendationIds);
        }

        @Override
        @Transient
        public final Builder recommendationIds(Collection<String> recommendationIds) {
            this.recommendationIds = RecommendationIdListCopier.copy(recommendationIds);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder recommendationIds(String... recommendationIds) {
            recommendationIds(Arrays.asList(recommendationIds));
            return this;
        }

        public final String getRecommendationTemplateArn() {
            return recommendationTemplateArn;
        }

        public final void setRecommendationTemplateArn(String recommendationTemplateArn) {
            this.recommendationTemplateArn = recommendationTemplateArn;
        }

        @Override
        @Transient
        public final Builder recommendationTemplateArn(String recommendationTemplateArn) {
            this.recommendationTemplateArn = recommendationTemplateArn;
            return this;
        }

        public final Collection<String> getRecommendationTypes() {
            if (recommendationTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return recommendationTypes;
        }

        public final void setRecommendationTypes(Collection<String> recommendationTypes) {
            this.recommendationTypes = RenderRecommendationTypeListCopier.copy(recommendationTypes);
        }

        @Override
        @Transient
        public final Builder recommendationTypesWithStrings(Collection<String> recommendationTypes) {
            this.recommendationTypes = RenderRecommendationTypeListCopier.copy(recommendationTypes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder recommendationTypesWithStrings(String... recommendationTypes) {
            recommendationTypesWithStrings(Arrays.asList(recommendationTypes));
            return this;
        }

        @Override
        @Transient
        public final Builder recommendationTypes(Collection<RenderRecommendationType> recommendationTypes) {
            this.recommendationTypes = RenderRecommendationTypeListCopier.copyEnumToString(recommendationTypes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder recommendationTypes(RenderRecommendationType... recommendationTypes) {
            recommendationTypes(Arrays.asList(recommendationTypes));
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        @Transient
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(RecommendationTemplateStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        @Transient
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final S3Location.Builder getTemplatesLocation() {
            return templatesLocation != null ? templatesLocation.toBuilder() : null;
        }

        public final void setTemplatesLocation(S3Location.BuilderImpl templatesLocation) {
            this.templatesLocation = templatesLocation != null ? templatesLocation.build() : null;
        }

        @Override
        @Transient
        public final Builder templatesLocation(S3Location templatesLocation) {
            this.templatesLocation = templatesLocation;
            return this;
        }

        @Override
        public RecommendationTemplate build() {
            return new RecommendationTemplate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
