/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the input image either as bytes or an S3 object.
 * </p>
 * <p>
 * You pass image bytes to an Amazon Rekognition API operation by using the <code>Bytes</code> property. For example,
 * you would use the <code>Bytes</code> property to pass an image loaded from a local file system. Image bytes passed by
 * using the <code>Bytes</code> property must be base64-encoded. Your code may not need to encode image bytes if you are
 * using an AWS SDK to call Amazon Rekognition API operations.
 * </p>
 * <p>
 * For more information, see Analyzing an Image Loaded from a Local File System in the Amazon Rekognition Developer
 * Guide.
 * </p>
 * <p>
 * You pass images stored in an S3 bucket to an Amazon Rekognition API operation by using the <code>S3Object</code>
 * property. Images stored in an S3 bucket do not need to be base64-encoded.
 * </p>
 * <p>
 * The region for the S3 bucket containing the S3 object must match the region you use for Amazon Rekognition
 * operations.
 * </p>
 * <p>
 * If you use the AWS CLI to call Amazon Rekognition operations, passing image bytes using the Bytes property is not
 * supported. You must first upload the image to an Amazon S3 bucket and then call the operation using the S3Object
 * property.
 * </p>
 * <p>
 * For Amazon Rekognition to process an S3 object, the user must have permission to access the S3 object. For more
 * information, see How Amazon Rekognition works with IAM in the Amazon Rekognition Developer Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Image implements SdkPojo, Serializable, ToCopyableBuilder<Image.Builder, Image> {
    private static final SdkField<SdkBytes> BYTES_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("Bytes").getter(getter(Image::bytes)).setter(setter(Builder::bytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bytes").build()).build();

    private static final SdkField<S3Object> S3_OBJECT_FIELD = SdkField.<S3Object> builder(MarshallingType.SDK_POJO)
            .memberName("S3Object").getter(getter(Image::s3Object)).setter(setter(Builder::s3Object))
            .constructor(S3Object::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Object").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BYTES_FIELD, S3_OBJECT_FIELD));

    private static final long serialVersionUID = 1L;

    private final SdkBytes bytes;

    private final S3Object s3Object;

    private Image(BuilderImpl builder) {
        this.bytes = builder.bytes;
        this.s3Object = builder.s3Object;
    }

    /**
     * <p>
     * Blob of image bytes up to 5 MBs. Note that the maximum image size you can pass to <code>DetectCustomLabels</code>
     * is 4MB.
     * </p>
     * 
     * @return Blob of image bytes up to 5 MBs. Note that the maximum image size you can pass to
     *         <code>DetectCustomLabels</code> is 4MB.
     */
    public final SdkBytes bytes() {
        return bytes;
    }

    /**
     * <p>
     * Identifies an S3 object as the image source.
     * </p>
     * 
     * @return Identifies an S3 object as the image source.
     */
    public final S3Object s3Object() {
        return s3Object;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bytes());
        hashCode = 31 * hashCode + Objects.hashCode(s3Object());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Image)) {
            return false;
        }
        Image other = (Image) obj;
        return Objects.equals(bytes(), other.bytes()) && Objects.equals(s3Object(), other.s3Object());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Image").add("Bytes", bytes()).add("S3Object", s3Object()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bytes":
            return Optional.ofNullable(clazz.cast(bytes()));
        case "S3Object":
            return Optional.ofNullable(clazz.cast(s3Object()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Image, T> g) {
        return obj -> g.apply((Image) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Image> {
        /**
         * <p>
         * Blob of image bytes up to 5 MBs. Note that the maximum image size you can pass to
         * <code>DetectCustomLabels</code> is 4MB.
         * </p>
         * 
         * @param bytes
         *        Blob of image bytes up to 5 MBs. Note that the maximum image size you can pass to
         *        <code>DetectCustomLabels</code> is 4MB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytes(SdkBytes bytes);

        /**
         * <p>
         * Identifies an S3 object as the image source.
         * </p>
         * 
         * @param s3Object
         *        Identifies an S3 object as the image source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Object(S3Object s3Object);

        /**
         * <p>
         * Identifies an S3 object as the image source.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Object.Builder} avoiding the need to
         * create one manually via {@link S3Object#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3Object.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Object(S3Object)}.
         * 
         * @param s3Object
         *        a consumer that will call methods on {@link S3Object.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Object(S3Object)
         */
        default Builder s3Object(Consumer<S3Object.Builder> s3Object) {
            return s3Object(S3Object.builder().applyMutation(s3Object).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private SdkBytes bytes;

        private S3Object s3Object;

        private BuilderImpl() {
        }

        private BuilderImpl(Image model) {
            bytes(model.bytes);
            s3Object(model.s3Object);
        }

        public final ByteBuffer getBytes() {
            return bytes == null ? null : bytes.asByteBuffer();
        }

        public final void setBytes(ByteBuffer bytes) {
            bytes(bytes == null ? null : SdkBytes.fromByteBuffer(bytes));
        }

        @Override
        public final Builder bytes(SdkBytes bytes) {
            this.bytes = bytes;
            return this;
        }

        public final S3Object.Builder getS3Object() {
            return s3Object != null ? s3Object.toBuilder() : null;
        }

        public final void setS3Object(S3Object.BuilderImpl s3Object) {
            this.s3Object = s3Object != null ? s3Object.build() : null;
        }

        @Override
        public final Builder s3Object(S3Object s3Object) {
            this.s3Object = s3Object;
            return this;
        }

        @Override
        public Image build() {
            return new Image(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
