/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.StandardMemberCopier;
import software.amazon.awssdk.services.rekognition.transform.ImageMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the input image either as bytes or an S3 object.
 * </p>
 * <p>
 * You pass image bytes to a Rekognition API operation by using the <code>Bytes</code> property. For example, you would
 * use the <code>Bytes</code> property to pass an image loaded from a local file system. Image bytes passed by using the
 * <code>Bytes</code> property must be base64-encoded. Your code may not need to encode image bytes if you are using an
 * AWS SDK to call Rekognition API operations. For more information, see <a>example4</a>.
 * </p>
 * <p>
 * You pass images stored in an S3 bucket to a Rekognition API operation by using the <code>S3Object</code> property.
 * Images stored in an S3 bucket do not need to be base64-encoded.
 * </p>
 * <p>
 * The region for the S3 bucket containing the S3 object must match the region you use for Amazon Rekognition
 * operations.
 * </p>
 * <p>
 * If you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes using the Bytes property is not
 * supported. You must first upload the image to an Amazon S3 bucket and then call the operation using the S3Object
 * property.
 * </p>
 * <p>
 * For Amazon Rekognition to process an S3 object, the user must have permission to access the S3 object. For more
 * information, see <a>manage-access-resource-policies</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Image implements StructuredPojo, ToCopyableBuilder<Image.Builder, Image> {
    private final ByteBuffer bytes;

    private final S3Object s3Object;

    private Image(BuilderImpl builder) {
        this.bytes = builder.bytes;
        this.s3Object = builder.s3Object;
    }

    /**
     * <p>
     * Blob of image bytes up to 5 MBs.
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return Blob of image bytes up to 5 MBs.
     */
    public ByteBuffer bytes() {
        return bytes == null ? null : bytes.asReadOnlyBuffer();
    }

    /**
     * <p>
     * Identifies an S3 object as the image source.
     * </p>
     * 
     * @return Identifies an S3 object as the image source.
     */
    public S3Object s3Object() {
        return s3Object;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bytes());
        hashCode = 31 * hashCode + Objects.hashCode(s3Object());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Image)) {
            return false;
        }
        Image other = (Image) obj;
        return Objects.equals(bytes(), other.bytes()) && Objects.equals(s3Object(), other.s3Object());
    }

    @Override
    public String toString() {
        return ToString.builder("Image").add("Bytes", bytes()).add("S3Object", s3Object()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bytes":
            return Optional.of(clazz.cast(bytes()));
        case "S3Object":
            return Optional.of(clazz.cast(s3Object()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ImageMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Image> {
        /**
         * <p>
         * Blob of image bytes up to 5 MBs.
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param bytes
         *        Blob of image bytes up to 5 MBs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytes(ByteBuffer bytes);

        /**
         * <p>
         * Identifies an S3 object as the image source.
         * </p>
         * 
         * @param s3Object
         *        Identifies an S3 object as the image source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Object(S3Object s3Object);

        /**
         * <p>
         * Identifies an S3 object as the image source.
         * </p>
         * This is a convenience that creates an instance of the {@link S3Object.Builder} avoiding the need to create
         * one manually via {@link S3Object#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Object.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Object(S3Object)}.
         * 
         * @param s3Object
         *        a consumer that will call methods on {@link S3Object.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Object(S3Object)
         */
        default Builder s3Object(Consumer<S3Object.Builder> s3Object) {
            return s3Object(S3Object.builder().apply(s3Object).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private ByteBuffer bytes;

        private S3Object s3Object;

        private BuilderImpl() {
        }

        private BuilderImpl(Image model) {
            bytes(model.bytes);
            s3Object(model.s3Object);
        }

        public final ByteBuffer getBytes() {
            return bytes;
        }

        @Override
        public final Builder bytes(ByteBuffer bytes) {
            this.bytes = StandardMemberCopier.copy(bytes);
            return this;
        }

        public final void setBytes(ByteBuffer bytes) {
            this.bytes = StandardMemberCopier.copy(bytes);
        }

        public final S3Object.Builder getS3Object() {
            return s3Object != null ? s3Object.toBuilder() : null;
        }

        @Override
        public final Builder s3Object(S3Object s3Object) {
            this.s3Object = s3Object;
            return this;
        }

        public final void setS3Object(S3Object.BuilderImpl s3Object) {
            this.s3Object = s3Object != null ? s3Object.build() : null;
        }

        @Override
        public Image build() {
            return new Image(this);
        }
    }
}
