/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The snapshot copy grant that grants Amazon Redshift permission to encrypt copied snapshots with the specified
 * customer master key (CMK) from Amazon Web Services KMS in the destination region.
 * </p>
 * <p>
 * For more information about managing snapshot copy grants, go to <a
 * href="https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html">Amazon Redshift Database
 * Encryption</a> in the <i>Amazon Redshift Cluster Management Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SnapshotCopyGrant implements SdkPojo, Serializable,
        ToCopyableBuilder<SnapshotCopyGrant.Builder, SnapshotCopyGrant> {
    private static final SdkField<String> SNAPSHOT_COPY_GRANT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnapshotCopyGrantName").getter(getter(SnapshotCopyGrant::snapshotCopyGrantName))
            .setter(setter(Builder::snapshotCopyGrantName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotCopyGrantName").build())
            .build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(SnapshotCopyGrant::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(SnapshotCopyGrant::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Tag")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Tag").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            SNAPSHOT_COPY_GRANT_NAME_FIELD, KMS_KEY_ID_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String snapshotCopyGrantName;

    private final String kmsKeyId;

    private final List<Tag> tags;

    private SnapshotCopyGrant(BuilderImpl builder) {
        this.snapshotCopyGrantName = builder.snapshotCopyGrantName;
        this.kmsKeyId = builder.kmsKeyId;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the snapshot copy grant.
     * </p>
     * 
     * @return The name of the snapshot copy grant.
     */
    public final String snapshotCopyGrantName() {
        return snapshotCopyGrantName;
    }

    /**
     * <p>
     * The unique identifier of the customer master key (CMK) in Amazon Web Services KMS to which Amazon Redshift is
     * granted permission.
     * </p>
     * 
     * @return The unique identifier of the customer master key (CMK) in Amazon Web Services KMS to which Amazon
     *         Redshift is granted permission.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tag instances.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tag instances.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(snapshotCopyGrantName());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SnapshotCopyGrant)) {
            return false;
        }
        SnapshotCopyGrant other = (SnapshotCopyGrant) obj;
        return Objects.equals(snapshotCopyGrantName(), other.snapshotCopyGrantName())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SnapshotCopyGrant").add("SnapshotCopyGrantName", snapshotCopyGrantName())
                .add("KmsKeyId", kmsKeyId()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SnapshotCopyGrantName":
            return Optional.ofNullable(clazz.cast(snapshotCopyGrantName()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SnapshotCopyGrant, T> g) {
        return obj -> g.apply((SnapshotCopyGrant) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SnapshotCopyGrant> {
        /**
         * <p>
         * The name of the snapshot copy grant.
         * </p>
         * 
         * @param snapshotCopyGrantName
         *        The name of the snapshot copy grant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotCopyGrantName(String snapshotCopyGrantName);

        /**
         * <p>
         * The unique identifier of the customer master key (CMK) in Amazon Web Services KMS to which Amazon Redshift is
         * granted permission.
         * </p>
         * 
         * @param kmsKeyId
         *        The unique identifier of the customer master key (CMK) in Amazon Web Services KMS to which Amazon
         *        Redshift is granted permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * A list of tag instances.
         * </p>
         * 
         * @param tags
         *        A list of tag instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tag instances.
         * </p>
         * 
         * @param tags
         *        A list of tag instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tag instances.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String snapshotCopyGrantName;

        private String kmsKeyId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SnapshotCopyGrant model) {
            snapshotCopyGrantName(model.snapshotCopyGrantName);
            kmsKeyId(model.kmsKeyId);
            tags(model.tags);
        }

        public final String getSnapshotCopyGrantName() {
            return snapshotCopyGrantName;
        }

        public final void setSnapshotCopyGrantName(String snapshotCopyGrantName) {
            this.snapshotCopyGrantName = snapshotCopyGrantName;
        }

        @Override
        @Transient
        public final Builder snapshotCopyGrantName(String snapshotCopyGrantName) {
            this.snapshotCopyGrantName = snapshotCopyGrantName;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        @Transient
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public SnapshotCopyGrant build() {
            return new SnapshotCopyGrant(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
