/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a modify cluster parameter group operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ModifyClusterParameterGroupRequest extends RedshiftRequest implements
        ToCopyableBuilder<ModifyClusterParameterGroupRequest.Builder, ModifyClusterParameterGroupRequest> {
    private static final SdkField<String> PARAMETER_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParameterGroupName").getter(getter(ModifyClusterParameterGroupRequest::parameterGroupName))
            .setter(setter(Builder::parameterGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterGroupName").build())
            .build();

    private static final SdkField<List<Parameter>> PARAMETERS_FIELD = SdkField
            .<List<Parameter>> builder(MarshallingType.LIST)
            .memberName("Parameters")
            .getter(getter(ModifyClusterParameterGroupRequest::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Parameter")
                            .memberFieldInfo(
                                    SdkField.<Parameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Parameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Parameter").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PARAMETER_GROUP_NAME_FIELD,
            PARAMETERS_FIELD));

    private final String parameterGroupName;

    private final List<Parameter> parameters;

    private ModifyClusterParameterGroupRequest(BuilderImpl builder) {
        super(builder);
        this.parameterGroupName = builder.parameterGroupName;
        this.parameters = builder.parameters;
    }

    /**
     * <p>
     * The name of the parameter group to be modified.
     * </p>
     * 
     * @return The name of the parameter group to be modified.
     */
    public final String parameterGroupName() {
        return parameterGroupName;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of parameters to be modified. A maximum of 20 parameters can be modified in a single request.
     * </p>
     * <p>
     * For each parameter to be modified, you must supply at least the parameter name and parameter value; other
     * name-value pairs of the parameter are optional.
     * </p>
     * <p>
     * For the workload management (WLM) configuration, you must supply all the name-value pairs in the
     * wlm_json_configuration parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of parameters to be modified. A maximum of 20 parameters can be modified in a single
     *         request.</p>
     *         <p>
     *         For each parameter to be modified, you must supply at least the parameter name and parameter value; other
     *         name-value pairs of the parameter are optional.
     *         </p>
     *         <p>
     *         For the workload management (WLM) configuration, you must supply all the name-value pairs in the
     *         wlm_json_configuration parameter.
     */
    public final List<Parameter> parameters() {
        return parameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(parameterGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModifyClusterParameterGroupRequest)) {
            return false;
        }
        ModifyClusterParameterGroupRequest other = (ModifyClusterParameterGroupRequest) obj;
        return Objects.equals(parameterGroupName(), other.parameterGroupName()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ModifyClusterParameterGroupRequest").add("ParameterGroupName", parameterGroupName())
                .add("Parameters", hasParameters() ? parameters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ParameterGroupName":
            return Optional.ofNullable(clazz.cast(parameterGroupName()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ModifyClusterParameterGroupRequest, T> g) {
        return obj -> g.apply((ModifyClusterParameterGroupRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RedshiftRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, ModifyClusterParameterGroupRequest> {
        /**
         * <p>
         * The name of the parameter group to be modified.
         * </p>
         * 
         * @param parameterGroupName
         *        The name of the parameter group to be modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterGroupName(String parameterGroupName);

        /**
         * <p>
         * An array of parameters to be modified. A maximum of 20 parameters can be modified in a single request.
         * </p>
         * <p>
         * For each parameter to be modified, you must supply at least the parameter name and parameter value; other
         * name-value pairs of the parameter are optional.
         * </p>
         * <p>
         * For the workload management (WLM) configuration, you must supply all the name-value pairs in the
         * wlm_json_configuration parameter.
         * </p>
         * 
         * @param parameters
         *        An array of parameters to be modified. A maximum of 20 parameters can be modified in a single
         *        request.</p>
         *        <p>
         *        For each parameter to be modified, you must supply at least the parameter name and parameter value;
         *        other name-value pairs of the parameter are optional.
         *        </p>
         *        <p>
         *        For the workload management (WLM) configuration, you must supply all the name-value pairs in the
         *        wlm_json_configuration parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * An array of parameters to be modified. A maximum of 20 parameters can be modified in a single request.
         * </p>
         * <p>
         * For each parameter to be modified, you must supply at least the parameter name and parameter value; other
         * name-value pairs of the parameter are optional.
         * </p>
         * <p>
         * For the workload management (WLM) configuration, you must supply all the name-value pairs in the
         * wlm_json_configuration parameter.
         * </p>
         * 
         * @param parameters
         *        An array of parameters to be modified. A maximum of 20 parameters can be modified in a single
         *        request.</p>
         *        <p>
         *        For each parameter to be modified, you must supply at least the parameter name and parameter value;
         *        other name-value pairs of the parameter are optional.
         *        </p>
         *        <p>
         *        For the workload management (WLM) configuration, you must supply all the name-value pairs in the
         *        wlm_json_configuration parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * An array of parameters to be modified. A maximum of 20 parameters can be modified in a single request.
         * </p>
         * <p>
         * For each parameter to be modified, you must supply at least the parameter name and parameter value; other
         * name-value pairs of the parameter are optional.
         * </p>
         * <p>
         * For the workload management (WLM) configuration, you must supply all the name-value pairs in the
         * wlm_json_configuration parameter.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Parameter>.Builder} avoiding the need to
         * create one manually via {@link List<Parameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Parameter>.Builder#build()} is called immediately and its
         * result is passed to {@link #parameters(List<Parameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<Parameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<Parameter>)
         */
        Builder parameters(Consumer<Parameter.Builder>... parameters);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends RedshiftRequest.BuilderImpl implements Builder {
        private String parameterGroupName;

        private List<Parameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ModifyClusterParameterGroupRequest model) {
            super(model);
            parameterGroupName(model.parameterGroupName);
            parameters(model.parameters);
        }

        public final String getParameterGroupName() {
            return parameterGroupName;
        }

        @Override
        public final Builder parameterGroupName(String parameterGroupName) {
            this.parameterGroupName = parameterGroupName;
            return this;
        }

        public final void setParameterGroupName(String parameterGroupName) {
            this.parameterGroupName = parameterGroupName;
        }

        public final Collection<Parameter.Builder> getParameters() {
            if (parameters instanceof SdkAutoConstructList) {
                return null;
            }
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersListCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<Parameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> Parameter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersListCopier.copyFromBuilder(parameters);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ModifyClusterParameterGroupRequest build() {
            return new ModifyClusterParameterGroupRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
