/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the default cluster parameters for a parameter group family.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DefaultClusterParameters implements SdkPojo, Serializable,
        ToCopyableBuilder<DefaultClusterParameters.Builder, DefaultClusterParameters> {
    private static final SdkField<String> PARAMETER_GROUP_FAMILY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DefaultClusterParameters::parameterGroupFamily)).setter(setter(Builder::parameterGroupFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterGroupFamily").build())
            .build();

    private static final SdkField<String> MARKER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DefaultClusterParameters::marker)).setter(setter(Builder::marker))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Marker").build()).build();

    private static final SdkField<List<Parameter>> PARAMETERS_FIELD = SdkField
            .<List<Parameter>> builder(MarshallingType.LIST)
            .getter(getter(DefaultClusterParameters::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Parameter")
                            .memberFieldInfo(
                                    SdkField.<Parameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Parameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Parameter").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PARAMETER_GROUP_FAMILY_FIELD,
            MARKER_FIELD, PARAMETERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String parameterGroupFamily;

    private final String marker;

    private final List<Parameter> parameters;

    private DefaultClusterParameters(BuilderImpl builder) {
        this.parameterGroupFamily = builder.parameterGroupFamily;
        this.marker = builder.marker;
        this.parameters = builder.parameters;
    }

    /**
     * <p>
     * The name of the cluster parameter group family to which the engine default parameters apply.
     * </p>
     * 
     * @return The name of the cluster parameter group family to which the engine default parameters apply.
     */
    public String parameterGroupFamily() {
        return parameterGroupFamily;
    }

    /**
     * <p>
     * A value that indicates the starting point for the next set of response records in a subsequent request. If a
     * value is returned in a response, you can retrieve the next set of records by providing this returned marker value
     * in the <code>Marker</code> parameter and retrying the command. If the <code>Marker</code> field is empty, all
     * response records have been retrieved for the request.
     * </p>
     * 
     * @return A value that indicates the starting point for the next set of response records in a subsequent request.
     *         If a value is returned in a response, you can retrieve the next set of records by providing this returned
     *         marker value in the <code>Marker</code> parameter and retrying the command. If the <code>Marker</code>
     *         field is empty, all response records have been retrieved for the request.
     */
    public String marker() {
        return marker;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of cluster default parameters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of cluster default parameters.
     */
    public List<Parameter> parameters() {
        return parameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(parameterGroupFamily());
        hashCode = 31 * hashCode + Objects.hashCode(marker());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DefaultClusterParameters)) {
            return false;
        }
        DefaultClusterParameters other = (DefaultClusterParameters) obj;
        return Objects.equals(parameterGroupFamily(), other.parameterGroupFamily()) && Objects.equals(marker(), other.marker())
                && Objects.equals(parameters(), other.parameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DefaultClusterParameters").add("ParameterGroupFamily", parameterGroupFamily())
                .add("Marker", marker()).add("Parameters", parameters()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ParameterGroupFamily":
            return Optional.ofNullable(clazz.cast(parameterGroupFamily()));
        case "Marker":
            return Optional.ofNullable(clazz.cast(marker()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DefaultClusterParameters, T> g) {
        return obj -> g.apply((DefaultClusterParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DefaultClusterParameters> {
        /**
         * <p>
         * The name of the cluster parameter group family to which the engine default parameters apply.
         * </p>
         * 
         * @param parameterGroupFamily
         *        The name of the cluster parameter group family to which the engine default parameters apply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterGroupFamily(String parameterGroupFamily);

        /**
         * <p>
         * A value that indicates the starting point for the next set of response records in a subsequent request. If a
         * value is returned in a response, you can retrieve the next set of records by providing this returned marker
         * value in the <code>Marker</code> parameter and retrying the command. If the <code>Marker</code> field is
         * empty, all response records have been retrieved for the request.
         * </p>
         * 
         * @param marker
         *        A value that indicates the starting point for the next set of response records in a subsequent
         *        request. If a value is returned in a response, you can retrieve the next set of records by providing
         *        this returned marker value in the <code>Marker</code> parameter and retrying the command. If the
         *        <code>Marker</code> field is empty, all response records have been retrieved for the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder marker(String marker);

        /**
         * <p>
         * The list of cluster default parameters.
         * </p>
         * 
         * @param parameters
         *        The list of cluster default parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * The list of cluster default parameters.
         * </p>
         * 
         * @param parameters
         *        The list of cluster default parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * The list of cluster default parameters.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Parameter>.Builder} avoiding the need to
         * create one manually via {@link List<Parameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Parameter>.Builder#build()} is called immediately and its
         * result is passed to {@link #parameters(List<Parameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<Parameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<Parameter>)
         */
        Builder parameters(Consumer<Parameter.Builder>... parameters);
    }

    static final class BuilderImpl implements Builder {
        private String parameterGroupFamily;

        private String marker;

        private List<Parameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DefaultClusterParameters model) {
            parameterGroupFamily(model.parameterGroupFamily);
            marker(model.marker);
            parameters(model.parameters);
        }

        public final String getParameterGroupFamily() {
            return parameterGroupFamily;
        }

        @Override
        public final Builder parameterGroupFamily(String parameterGroupFamily) {
            this.parameterGroupFamily = parameterGroupFamily;
            return this;
        }

        public final void setParameterGroupFamily(String parameterGroupFamily) {
            this.parameterGroupFamily = parameterGroupFamily;
        }

        public final String getMarker() {
            return marker;
        }

        @Override
        public final Builder marker(String marker) {
            this.marker = marker;
            return this;
        }

        public final void setMarker(String marker) {
            this.marker = marker;
        }

        public final Collection<Parameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersListCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<Parameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> Parameter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersListCopier.copyFromBuilder(parameters);
        }

        @Override
        public DefaultClusterParameters build() {
            return new DefaultClusterParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
