/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a <code>ClusterDbRevision</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClusterDbRevision implements SdkPojo, Serializable,
        ToCopyableBuilder<ClusterDbRevision.Builder, ClusterDbRevision> {
    private static final SdkField<String> CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterDbRevision::clusterIdentifier)).setter(setter(Builder::clusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterIdentifier").build()).build();

    private static final SdkField<String> CURRENT_DATABASE_REVISION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterDbRevision::currentDatabaseRevision)).setter(setter(Builder::currentDatabaseRevision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentDatabaseRevision").build())
            .build();

    private static final SdkField<Instant> DATABASE_REVISION_RELEASE_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ClusterDbRevision::databaseRevisionReleaseDate))
            .setter(setter(Builder::databaseRevisionReleaseDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseRevisionReleaseDate")
                    .build()).build();

    private static final SdkField<List<RevisionTarget>> REVISION_TARGETS_FIELD = SdkField
            .<List<RevisionTarget>> builder(MarshallingType.LIST)
            .getter(getter(ClusterDbRevision::revisionTargets))
            .setter(setter(Builder::revisionTargets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RevisionTargets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("RevisionTarget")
                            .memberFieldInfo(
                                    SdkField.<RevisionTarget> builder(MarshallingType.SDK_POJO)
                                            .constructor(RevisionTarget::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("RevisionTarget").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_IDENTIFIER_FIELD,
            CURRENT_DATABASE_REVISION_FIELD, DATABASE_REVISION_RELEASE_DATE_FIELD, REVISION_TARGETS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String clusterIdentifier;

    private final String currentDatabaseRevision;

    private final Instant databaseRevisionReleaseDate;

    private final List<RevisionTarget> revisionTargets;

    private ClusterDbRevision(BuilderImpl builder) {
        this.clusterIdentifier = builder.clusterIdentifier;
        this.currentDatabaseRevision = builder.currentDatabaseRevision;
        this.databaseRevisionReleaseDate = builder.databaseRevisionReleaseDate;
        this.revisionTargets = builder.revisionTargets;
    }

    /**
     * <p>
     * The unique identifier of the cluster.
     * </p>
     * 
     * @return The unique identifier of the cluster.
     */
    public String clusterIdentifier() {
        return clusterIdentifier;
    }

    /**
     * <p>
     * A string representing the current cluster version.
     * </p>
     * 
     * @return A string representing the current cluster version.
     */
    public String currentDatabaseRevision() {
        return currentDatabaseRevision;
    }

    /**
     * <p>
     * The date on which the database revision was released.
     * </p>
     * 
     * @return The date on which the database revision was released.
     */
    public Instant databaseRevisionReleaseDate() {
        return databaseRevisionReleaseDate;
    }

    /**
     * <p>
     * A list of <code>RevisionTarget</code> objects, where each object describes the database revision that a cluster
     * can be updated to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of <code>RevisionTarget</code> objects, where each object describes the database revision that a
     *         cluster can be updated to.
     */
    public List<RevisionTarget> revisionTargets() {
        return revisionTargets;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(currentDatabaseRevision());
        hashCode = 31 * hashCode + Objects.hashCode(databaseRevisionReleaseDate());
        hashCode = 31 * hashCode + Objects.hashCode(revisionTargets());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClusterDbRevision)) {
            return false;
        }
        ClusterDbRevision other = (ClusterDbRevision) obj;
        return Objects.equals(clusterIdentifier(), other.clusterIdentifier())
                && Objects.equals(currentDatabaseRevision(), other.currentDatabaseRevision())
                && Objects.equals(databaseRevisionReleaseDate(), other.databaseRevisionReleaseDate())
                && Objects.equals(revisionTargets(), other.revisionTargets());
    }

    @Override
    public String toString() {
        return ToString.builder("ClusterDbRevision").add("ClusterIdentifier", clusterIdentifier())
                .add("CurrentDatabaseRevision", currentDatabaseRevision())
                .add("DatabaseRevisionReleaseDate", databaseRevisionReleaseDate()).add("RevisionTargets", revisionTargets())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClusterIdentifier":
            return Optional.ofNullable(clazz.cast(clusterIdentifier()));
        case "CurrentDatabaseRevision":
            return Optional.ofNullable(clazz.cast(currentDatabaseRevision()));
        case "DatabaseRevisionReleaseDate":
            return Optional.ofNullable(clazz.cast(databaseRevisionReleaseDate()));
        case "RevisionTargets":
            return Optional.ofNullable(clazz.cast(revisionTargets()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ClusterDbRevision, T> g) {
        return obj -> g.apply((ClusterDbRevision) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClusterDbRevision> {
        /**
         * <p>
         * The unique identifier of the cluster.
         * </p>
         * 
         * @param clusterIdentifier
         *        The unique identifier of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterIdentifier(String clusterIdentifier);

        /**
         * <p>
         * A string representing the current cluster version.
         * </p>
         * 
         * @param currentDatabaseRevision
         *        A string representing the current cluster version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentDatabaseRevision(String currentDatabaseRevision);

        /**
         * <p>
         * The date on which the database revision was released.
         * </p>
         * 
         * @param databaseRevisionReleaseDate
         *        The date on which the database revision was released.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseRevisionReleaseDate(Instant databaseRevisionReleaseDate);

        /**
         * <p>
         * A list of <code>RevisionTarget</code> objects, where each object describes the database revision that a
         * cluster can be updated to.
         * </p>
         * 
         * @param revisionTargets
         *        A list of <code>RevisionTarget</code> objects, where each object describes the database revision that
         *        a cluster can be updated to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisionTargets(Collection<RevisionTarget> revisionTargets);

        /**
         * <p>
         * A list of <code>RevisionTarget</code> objects, where each object describes the database revision that a
         * cluster can be updated to.
         * </p>
         * 
         * @param revisionTargets
         *        A list of <code>RevisionTarget</code> objects, where each object describes the database revision that
         *        a cluster can be updated to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisionTargets(RevisionTarget... revisionTargets);

        /**
         * <p>
         * A list of <code>RevisionTarget</code> objects, where each object describes the database revision that a
         * cluster can be updated to.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RevisionTarget>.Builder} avoiding the need
         * to create one manually via {@link List<RevisionTarget>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RevisionTarget>.Builder#build()} is called immediately and
         * its result is passed to {@link #revisionTargets(List<RevisionTarget>)}.
         * 
         * @param revisionTargets
         *        a consumer that will call methods on {@link List<RevisionTarget>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #revisionTargets(List<RevisionTarget>)
         */
        Builder revisionTargets(Consumer<RevisionTarget.Builder>... revisionTargets);
    }

    static final class BuilderImpl implements Builder {
        private String clusterIdentifier;

        private String currentDatabaseRevision;

        private Instant databaseRevisionReleaseDate;

        private List<RevisionTarget> revisionTargets = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ClusterDbRevision model) {
            clusterIdentifier(model.clusterIdentifier);
            currentDatabaseRevision(model.currentDatabaseRevision);
            databaseRevisionReleaseDate(model.databaseRevisionReleaseDate);
            revisionTargets(model.revisionTargets);
        }

        public final String getClusterIdentifier() {
            return clusterIdentifier;
        }

        @Override
        public final Builder clusterIdentifier(String clusterIdentifier) {
            this.clusterIdentifier = clusterIdentifier;
            return this;
        }

        public final void setClusterIdentifier(String clusterIdentifier) {
            this.clusterIdentifier = clusterIdentifier;
        }

        public final String getCurrentDatabaseRevision() {
            return currentDatabaseRevision;
        }

        @Override
        public final Builder currentDatabaseRevision(String currentDatabaseRevision) {
            this.currentDatabaseRevision = currentDatabaseRevision;
            return this;
        }

        public final void setCurrentDatabaseRevision(String currentDatabaseRevision) {
            this.currentDatabaseRevision = currentDatabaseRevision;
        }

        public final Instant getDatabaseRevisionReleaseDate() {
            return databaseRevisionReleaseDate;
        }

        @Override
        public final Builder databaseRevisionReleaseDate(Instant databaseRevisionReleaseDate) {
            this.databaseRevisionReleaseDate = databaseRevisionReleaseDate;
            return this;
        }

        public final void setDatabaseRevisionReleaseDate(Instant databaseRevisionReleaseDate) {
            this.databaseRevisionReleaseDate = databaseRevisionReleaseDate;
        }

        public final Collection<RevisionTarget.Builder> getRevisionTargets() {
            return revisionTargets != null ? revisionTargets.stream().map(RevisionTarget::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder revisionTargets(Collection<RevisionTarget> revisionTargets) {
            this.revisionTargets = RevisionTargetsListCopier.copy(revisionTargets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder revisionTargets(RevisionTarget... revisionTargets) {
            revisionTargets(Arrays.asList(revisionTargets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder revisionTargets(Consumer<RevisionTarget.Builder>... revisionTargets) {
            revisionTargets(Stream.of(revisionTargets).map(c -> RevisionTarget.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRevisionTargets(Collection<RevisionTarget.BuilderImpl> revisionTargets) {
            this.revisionTargets = RevisionTargetsListCopier.copyFromBuilder(revisionTargets);
        }

        @Override
        public ClusterDbRevision build() {
            return new ClusterDbRevision(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
