/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qconnect.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration details for the web data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WebCrawlerConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<WebCrawlerConfiguration.Builder, WebCrawlerConfiguration> {
    private static final SdkField<WebCrawlerLimits> CRAWLER_LIMITS_FIELD = SdkField
            .<WebCrawlerLimits> builder(MarshallingType.SDK_POJO).memberName("crawlerLimits")
            .getter(getter(WebCrawlerConfiguration::crawlerLimits)).setter(setter(Builder::crawlerLimits))
            .constructor(WebCrawlerLimits::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("crawlerLimits").build()).build();

    private static final SdkField<List<String>> EXCLUSION_FILTERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("exclusionFilters")
            .getter(getter(WebCrawlerConfiguration::exclusionFilters))
            .setter(setter(Builder::exclusionFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exclusionFilters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INCLUSION_FILTERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("inclusionFilters")
            .getter(getter(WebCrawlerConfiguration::inclusionFilters))
            .setter(setter(Builder::inclusionFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inclusionFilters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("scope")
            .getter(getter(WebCrawlerConfiguration::scopeAsString)).setter(setter(Builder::scope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scope").build()).build();

    private static final SdkField<UrlConfiguration> URL_CONFIGURATION_FIELD = SdkField
            .<UrlConfiguration> builder(MarshallingType.SDK_POJO).memberName("urlConfiguration")
            .getter(getter(WebCrawlerConfiguration::urlConfiguration)).setter(setter(Builder::urlConfiguration))
            .constructor(UrlConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("urlConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CRAWLER_LIMITS_FIELD,
            EXCLUSION_FILTERS_FIELD, INCLUSION_FILTERS_FIELD, SCOPE_FIELD, URL_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final WebCrawlerLimits crawlerLimits;

    private final List<String> exclusionFilters;

    private final List<String> inclusionFilters;

    private final String scope;

    private final UrlConfiguration urlConfiguration;

    private WebCrawlerConfiguration(BuilderImpl builder) {
        this.crawlerLimits = builder.crawlerLimits;
        this.exclusionFilters = builder.exclusionFilters;
        this.inclusionFilters = builder.inclusionFilters;
        this.scope = builder.scope;
        this.urlConfiguration = builder.urlConfiguration;
    }

    /**
     * <p>
     * The configuration of crawl limits for the web URLs.
     * </p>
     * 
     * @return The configuration of crawl limits for the web URLs.
     */
    public final WebCrawlerLimits crawlerLimits() {
        return crawlerLimits;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExclusionFilters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExclusionFilters() {
        return exclusionFilters != null && !(exclusionFilters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of one or more exclusion regular expression patterns to exclude certain URLs. If you specify an inclusion
     * and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the web content of
     * the URL isn’t crawled.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusionFilters} method.
     * </p>
     * 
     * @return A list of one or more exclusion regular expression patterns to exclude certain URLs. If you specify an
     *         inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and
     *         the web content of the URL isn’t crawled.
     */
    public final List<String> exclusionFilters() {
        return exclusionFilters;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionFilters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionFilters() {
        return inclusionFilters != null && !(inclusionFilters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of one or more inclusion regular expression patterns to include certain URLs. If you specify an inclusion
     * and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the web content of
     * the URL isn’t crawled.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionFilters} method.
     * </p>
     * 
     * @return A list of one or more inclusion regular expression patterns to include certain URLs. If you specify an
     *         inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and
     *         the web content of the URL isn’t crawled.
     */
    public final List<String> inclusionFilters() {
        return inclusionFilters;
    }

    /**
     * <p>
     * The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the same host
     * or primary domain. For example, only web pages that contain the seed URL
     * <code>https://docs.aws.amazon.com/bedrock/latest/userguide/</code> and no other domains. You can choose to
     * include sub domains in addition to the host or primary domain. For example, web pages that contain
     * <code>aws.amazon.com</code> can also include sub domain <code>docs.aws.amazon.com</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scope} will return
     * {@link WebScopeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scopeAsString}.
     * </p>
     * 
     * @return The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the
     *         same host or primary domain. For example, only web pages that contain the seed URL
     *         <code>https://docs.aws.amazon.com/bedrock/latest/userguide/</code> and no other domains. You can choose
     *         to include sub domains in addition to the host or primary domain. For example, web pages that contain
     *         <code>aws.amazon.com</code> can also include sub domain <code>docs.aws.amazon.com</code>.
     * @see WebScopeType
     */
    public final WebScopeType scope() {
        return WebScopeType.fromValue(scope);
    }

    /**
     * <p>
     * The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the same host
     * or primary domain. For example, only web pages that contain the seed URL
     * <code>https://docs.aws.amazon.com/bedrock/latest/userguide/</code> and no other domains. You can choose to
     * include sub domains in addition to the host or primary domain. For example, web pages that contain
     * <code>aws.amazon.com</code> can also include sub domain <code>docs.aws.amazon.com</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scope} will return
     * {@link WebScopeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scopeAsString}.
     * </p>
     * 
     * @return The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the
     *         same host or primary domain. For example, only web pages that contain the seed URL
     *         <code>https://docs.aws.amazon.com/bedrock/latest/userguide/</code> and no other domains. You can choose
     *         to include sub domains in addition to the host or primary domain. For example, web pages that contain
     *         <code>aws.amazon.com</code> can also include sub domain <code>docs.aws.amazon.com</code>.
     * @see WebScopeType
     */
    public final String scopeAsString() {
        return scope;
    }

    /**
     * <p>
     * The configuration of the URL/URLs for the web content that you want to crawl. You should be authorized to crawl
     * the URLs.
     * </p>
     * 
     * @return The configuration of the URL/URLs for the web content that you want to crawl. You should be authorized to
     *         crawl the URLs.
     */
    public final UrlConfiguration urlConfiguration() {
        return urlConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(crawlerLimits());
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusionFilters() ? exclusionFilters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionFilters() ? inclusionFilters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(scopeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(urlConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WebCrawlerConfiguration)) {
            return false;
        }
        WebCrawlerConfiguration other = (WebCrawlerConfiguration) obj;
        return Objects.equals(crawlerLimits(), other.crawlerLimits()) && hasExclusionFilters() == other.hasExclusionFilters()
                && Objects.equals(exclusionFilters(), other.exclusionFilters())
                && hasInclusionFilters() == other.hasInclusionFilters()
                && Objects.equals(inclusionFilters(), other.inclusionFilters())
                && Objects.equals(scopeAsString(), other.scopeAsString())
                && Objects.equals(urlConfiguration(), other.urlConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WebCrawlerConfiguration").add("CrawlerLimits", crawlerLimits())
                .add("ExclusionFilters", exclusionFilters() == null ? null : "*** Sensitive Data Redacted ***")
                .add("InclusionFilters", inclusionFilters() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Scope", scopeAsString()).add("UrlConfiguration", urlConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "crawlerLimits":
            return Optional.ofNullable(clazz.cast(crawlerLimits()));
        case "exclusionFilters":
            return Optional.ofNullable(clazz.cast(exclusionFilters()));
        case "inclusionFilters":
            return Optional.ofNullable(clazz.cast(inclusionFilters()));
        case "scope":
            return Optional.ofNullable(clazz.cast(scopeAsString()));
        case "urlConfiguration":
            return Optional.ofNullable(clazz.cast(urlConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("crawlerLimits", CRAWLER_LIMITS_FIELD);
        map.put("exclusionFilters", EXCLUSION_FILTERS_FIELD);
        map.put("inclusionFilters", INCLUSION_FILTERS_FIELD);
        map.put("scope", SCOPE_FIELD);
        map.put("urlConfiguration", URL_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<WebCrawlerConfiguration, T> g) {
        return obj -> g.apply((WebCrawlerConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WebCrawlerConfiguration> {
        /**
         * <p>
         * The configuration of crawl limits for the web URLs.
         * </p>
         * 
         * @param crawlerLimits
         *        The configuration of crawl limits for the web URLs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlerLimits(WebCrawlerLimits crawlerLimits);

        /**
         * <p>
         * The configuration of crawl limits for the web URLs.
         * </p>
         * This is a convenience method that creates an instance of the {@link WebCrawlerLimits.Builder} avoiding the
         * need to create one manually via {@link WebCrawlerLimits#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WebCrawlerLimits.Builder#build()} is called immediately and its
         * result is passed to {@link #crawlerLimits(WebCrawlerLimits)}.
         * 
         * @param crawlerLimits
         *        a consumer that will call methods on {@link WebCrawlerLimits.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #crawlerLimits(WebCrawlerLimits)
         */
        default Builder crawlerLimits(Consumer<WebCrawlerLimits.Builder> crawlerLimits) {
            return crawlerLimits(WebCrawlerLimits.builder().applyMutation(crawlerLimits).build());
        }

        /**
         * <p>
         * A list of one or more exclusion regular expression patterns to exclude certain URLs. If you specify an
         * inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the
         * web content of the URL isn’t crawled.
         * </p>
         * 
         * @param exclusionFilters
         *        A list of one or more exclusion regular expression patterns to exclude certain URLs. If you specify an
         *        inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and
         *        the web content of the URL isn’t crawled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionFilters(Collection<String> exclusionFilters);

        /**
         * <p>
         * A list of one or more exclusion regular expression patterns to exclude certain URLs. If you specify an
         * inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the
         * web content of the URL isn’t crawled.
         * </p>
         * 
         * @param exclusionFilters
         *        A list of one or more exclusion regular expression patterns to exclude certain URLs. If you specify an
         *        inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and
         *        the web content of the URL isn’t crawled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionFilters(String... exclusionFilters);

        /**
         * <p>
         * A list of one or more inclusion regular expression patterns to include certain URLs. If you specify an
         * inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the
         * web content of the URL isn’t crawled.
         * </p>
         * 
         * @param inclusionFilters
         *        A list of one or more inclusion regular expression patterns to include certain URLs. If you specify an
         *        inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and
         *        the web content of the URL isn’t crawled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionFilters(Collection<String> inclusionFilters);

        /**
         * <p>
         * A list of one or more inclusion regular expression patterns to include certain URLs. If you specify an
         * inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the
         * web content of the URL isn’t crawled.
         * </p>
         * 
         * @param inclusionFilters
         *        A list of one or more inclusion regular expression patterns to include certain URLs. If you specify an
         *        inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and
         *        the web content of the URL isn’t crawled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionFilters(String... inclusionFilters);

        /**
         * <p>
         * The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the same
         * host or primary domain. For example, only web pages that contain the seed URL
         * <code>https://docs.aws.amazon.com/bedrock/latest/userguide/</code> and no other domains. You can choose to
         * include sub domains in addition to the host or primary domain. For example, web pages that contain
         * <code>aws.amazon.com</code> can also include sub domain <code>docs.aws.amazon.com</code>.
         * </p>
         * 
         * @param scope
         *        The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the
         *        same host or primary domain. For example, only web pages that contain the seed URL
         *        <code>https://docs.aws.amazon.com/bedrock/latest/userguide/</code> and no other domains. You can
         *        choose to include sub domains in addition to the host or primary domain. For example, web pages that
         *        contain <code>aws.amazon.com</code> can also include sub domain <code>docs.aws.amazon.com</code>.
         * @see WebScopeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebScopeType
         */
        Builder scope(String scope);

        /**
         * <p>
         * The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the same
         * host or primary domain. For example, only web pages that contain the seed URL
         * <code>https://docs.aws.amazon.com/bedrock/latest/userguide/</code> and no other domains. You can choose to
         * include sub domains in addition to the host or primary domain. For example, web pages that contain
         * <code>aws.amazon.com</code> can also include sub domain <code>docs.aws.amazon.com</code>.
         * </p>
         * 
         * @param scope
         *        The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the
         *        same host or primary domain. For example, only web pages that contain the seed URL
         *        <code>https://docs.aws.amazon.com/bedrock/latest/userguide/</code> and no other domains. You can
         *        choose to include sub domains in addition to the host or primary domain. For example, web pages that
         *        contain <code>aws.amazon.com</code> can also include sub domain <code>docs.aws.amazon.com</code>.
         * @see WebScopeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebScopeType
         */
        Builder scope(WebScopeType scope);

        /**
         * <p>
         * The configuration of the URL/URLs for the web content that you want to crawl. You should be authorized to
         * crawl the URLs.
         * </p>
         * 
         * @param urlConfiguration
         *        The configuration of the URL/URLs for the web content that you want to crawl. You should be authorized
         *        to crawl the URLs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urlConfiguration(UrlConfiguration urlConfiguration);

        /**
         * <p>
         * The configuration of the URL/URLs for the web content that you want to crawl. You should be authorized to
         * crawl the URLs.
         * </p>
         * This is a convenience method that creates an instance of the {@link UrlConfiguration.Builder} avoiding the
         * need to create one manually via {@link UrlConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UrlConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #urlConfiguration(UrlConfiguration)}.
         * 
         * @param urlConfiguration
         *        a consumer that will call methods on {@link UrlConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #urlConfiguration(UrlConfiguration)
         */
        default Builder urlConfiguration(Consumer<UrlConfiguration.Builder> urlConfiguration) {
            return urlConfiguration(UrlConfiguration.builder().applyMutation(urlConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private WebCrawlerLimits crawlerLimits;

        private List<String> exclusionFilters = DefaultSdkAutoConstructList.getInstance();

        private List<String> inclusionFilters = DefaultSdkAutoConstructList.getInstance();

        private String scope;

        private UrlConfiguration urlConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(WebCrawlerConfiguration model) {
            crawlerLimits(model.crawlerLimits);
            exclusionFilters(model.exclusionFilters);
            inclusionFilters(model.inclusionFilters);
            scope(model.scope);
            urlConfiguration(model.urlConfiguration);
        }

        public final WebCrawlerLimits.Builder getCrawlerLimits() {
            return crawlerLimits != null ? crawlerLimits.toBuilder() : null;
        }

        public final void setCrawlerLimits(WebCrawlerLimits.BuilderImpl crawlerLimits) {
            this.crawlerLimits = crawlerLimits != null ? crawlerLimits.build() : null;
        }

        @Override
        public final Builder crawlerLimits(WebCrawlerLimits crawlerLimits) {
            this.crawlerLimits = crawlerLimits;
            return this;
        }

        public final Collection<String> getExclusionFilters() {
            if (exclusionFilters instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusionFilters;
        }

        public final void setExclusionFilters(Collection<String> exclusionFilters) {
            this.exclusionFilters = UrlFilterListCopier.copy(exclusionFilters);
        }

        @Override
        public final Builder exclusionFilters(Collection<String> exclusionFilters) {
            this.exclusionFilters = UrlFilterListCopier.copy(exclusionFilters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusionFilters(String... exclusionFilters) {
            exclusionFilters(Arrays.asList(exclusionFilters));
            return this;
        }

        public final Collection<String> getInclusionFilters() {
            if (inclusionFilters instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionFilters;
        }

        public final void setInclusionFilters(Collection<String> inclusionFilters) {
            this.inclusionFilters = UrlFilterListCopier.copy(inclusionFilters);
        }

        @Override
        public final Builder inclusionFilters(Collection<String> inclusionFilters) {
            this.inclusionFilters = UrlFilterListCopier.copy(inclusionFilters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionFilters(String... inclusionFilters) {
            inclusionFilters(Arrays.asList(inclusionFilters));
            return this;
        }

        public final String getScope() {
            return scope;
        }

        public final void setScope(String scope) {
            this.scope = scope;
        }

        @Override
        public final Builder scope(String scope) {
            this.scope = scope;
            return this;
        }

        @Override
        public final Builder scope(WebScopeType scope) {
            this.scope(scope == null ? null : scope.toString());
            return this;
        }

        public final UrlConfiguration.Builder getUrlConfiguration() {
            return urlConfiguration != null ? urlConfiguration.toBuilder() : null;
        }

        public final void setUrlConfiguration(UrlConfiguration.BuilderImpl urlConfiguration) {
            this.urlConfiguration = urlConfiguration != null ? urlConfiguration.build() : null;
        }

        @Override
        public final Builder urlConfiguration(UrlConfiguration urlConfiguration) {
            this.urlConfiguration = urlConfiguration;
            return this;
        }

        @Override
        public WebCrawlerConfiguration build() {
            return new WebCrawlerConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
