/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.privatenetworks;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.privatenetworks.internal.PrivateNetworksServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.privatenetworks.model.AccessDeniedException;
import software.amazon.awssdk.services.privatenetworks.model.AcknowledgeOrderReceiptRequest;
import software.amazon.awssdk.services.privatenetworks.model.AcknowledgeOrderReceiptResponse;
import software.amazon.awssdk.services.privatenetworks.model.ActivateDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.ActivateDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.ActivateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.ActivateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.ConfigureAccessPointRequest;
import software.amazon.awssdk.services.privatenetworks.model.ConfigureAccessPointResponse;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeactivateDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeactivateDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetOrderRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetOrderResponse;
import software.amazon.awssdk.services.privatenetworks.model.InternalServerException;
import software.amazon.awssdk.services.privatenetworks.model.LimitExceededException;
import software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworksRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworksResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListOrdersRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListOrdersResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.PingRequest;
import software.amazon.awssdk.services.privatenetworks.model.PingResponse;
import software.amazon.awssdk.services.privatenetworks.model.PrivateNetworksException;
import software.amazon.awssdk.services.privatenetworks.model.ResourceNotFoundException;
import software.amazon.awssdk.services.privatenetworks.model.StartNetworkResourceUpdateRequest;
import software.amazon.awssdk.services.privatenetworks.model.StartNetworkResourceUpdateResponse;
import software.amazon.awssdk.services.privatenetworks.model.TagResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.TagResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.ThrottlingException;
import software.amazon.awssdk.services.privatenetworks.model.UntagResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.UntagResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSitePlanRequest;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSitePlanResponse;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.ValidationException;
import software.amazon.awssdk.services.privatenetworks.transform.AcknowledgeOrderReceiptRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ActivateDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ActivateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ConfigureAccessPointRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.CreateNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.CreateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeactivateDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeleteNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeleteNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetOrderRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListDeviceIdentifiersRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworkResourcesRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworkSitesRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworksRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListOrdersRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.PingRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.StartNetworkResourceUpdateRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UpdateNetworkSitePlanRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UpdateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PrivateNetworksAsyncClient}.
 *
 * @see PrivateNetworksAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPrivateNetworksAsyncClient implements PrivateNetworksAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPrivateNetworksAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPrivateNetworksAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Acknowledges that the specified network order was received.
     * </p>
     *
     * @param acknowledgeOrderReceiptRequest
     * @return A Java Future containing the result of the AcknowledgeOrderReceipt operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.AcknowledgeOrderReceipt
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/AcknowledgeOrderReceipt"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcknowledgeOrderReceiptResponse> acknowledgeOrderReceipt(
            AcknowledgeOrderReceiptRequest acknowledgeOrderReceiptRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acknowledgeOrderReceiptRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acknowledgeOrderReceiptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcknowledgeOrderReceipt");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcknowledgeOrderReceiptResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcknowledgeOrderReceiptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcknowledgeOrderReceiptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcknowledgeOrderReceiptRequest, AcknowledgeOrderReceiptResponse>()
                            .withOperationName("AcknowledgeOrderReceipt").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AcknowledgeOrderReceiptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acknowledgeOrderReceiptRequest));
            CompletableFuture<AcknowledgeOrderReceiptResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates the specified device identifier.
     * </p>
     *
     * @param activateDeviceIdentifierRequest
     * @return A Java Future containing the result of the ActivateDeviceIdentifier operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ActivateDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ActivateDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateDeviceIdentifierResponse> activateDeviceIdentifier(
            ActivateDeviceIdentifierRequest activateDeviceIdentifierRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(activateDeviceIdentifierRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateDeviceIdentifier");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateDeviceIdentifierResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateDeviceIdentifierResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateDeviceIdentifierRequest, ActivateDeviceIdentifierResponse>()
                            .withOperationName("ActivateDeviceIdentifier").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ActivateDeviceIdentifierRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(activateDeviceIdentifierRequest));
            CompletableFuture<ActivateDeviceIdentifierResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates the specified network site.
     * </p>
     *
     * @param activateNetworkSiteRequest
     * @return A Java Future containing the result of the ActivateNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ActivateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ActivateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateNetworkSiteResponse> activateNetworkSite(
            ActivateNetworkSiteRequest activateNetworkSiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(activateNetworkSiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateNetworkSiteRequest, ActivateNetworkSiteResponse>()
                            .withOperationName("ActivateNetworkSite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ActivateNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(activateNetworkSiteRequest));
            CompletableFuture<ActivateNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures the specified network resource.
     * </p>
     * <p>
     * Use this action to specify the geographic position of the hardware. You must provide Certified Professional
     * Installer (CPI) credentials in the request so that we can obtain spectrum grants. For more information, see <a
     * href="https://docs.aws.amazon.com/private-networks/latest/userguide/radio-units.html">Radio units</a> in the
     * <i>Amazon Web Services Private 5G User Guide</i>.
     * </p>
     *
     * @param configureAccessPointRequest
     * @return A Java Future containing the result of the ConfigureAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ConfigureAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ConfigureAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ConfigureAccessPointResponse> configureAccessPoint(
            ConfigureAccessPointRequest configureAccessPointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(configureAccessPointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, configureAccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfigureAccessPoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConfigureAccessPointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ConfigureAccessPointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConfigureAccessPointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfigureAccessPointRequest, ConfigureAccessPointResponse>()
                            .withOperationName("ConfigureAccessPoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ConfigureAccessPointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(configureAccessPointRequest));
            CompletableFuture<ConfigureAccessPointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a network.
     * </p>
     *
     * @param createNetworkRequest
     * @return A Java Future containing the result of the CreateNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>LimitExceededException The limit was exceeded.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.CreateNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/CreateNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNetworkResponse> createNetwork(CreateNetworkRequest createNetworkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createNetworkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNetworkRequest, CreateNetworkResponse>()
                            .withOperationName("CreateNetwork").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createNetworkRequest));
            CompletableFuture<CreateNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a network site.
     * </p>
     *
     * @param createNetworkSiteRequest
     * @return A Java Future containing the result of the CreateNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.CreateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/CreateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNetworkSiteResponse> createNetworkSite(CreateNetworkSiteRequest createNetworkSiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createNetworkSiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNetworkSiteRequest, CreateNetworkSiteResponse>()
                            .withOperationName("CreateNetworkSite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createNetworkSiteRequest));
            CompletableFuture<CreateNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deactivates the specified device identifier.
     * </p>
     *
     * @param deactivateDeviceIdentifierRequest
     * @return A Java Future containing the result of the DeactivateDeviceIdentifier operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.DeactivateDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeactivateDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivateDeviceIdentifierResponse> deactivateDeviceIdentifier(
            DeactivateDeviceIdentifierRequest deactivateDeviceIdentifierRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deactivateDeviceIdentifierRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateDeviceIdentifier");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivateDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivateDeviceIdentifierResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivateDeviceIdentifierResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivateDeviceIdentifierRequest, DeactivateDeviceIdentifierResponse>()
                            .withOperationName("DeactivateDeviceIdentifier").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeactivateDeviceIdentifierRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deactivateDeviceIdentifierRequest));
            CompletableFuture<DeactivateDeviceIdentifierResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified network. You must delete network sites before you delete the network. For more information,
     * see <a href="https://docs.aws.amazon.com/private-networks/latest/APIReference/API_DeleteNetworkSite.html">
     * DeleteNetworkSite</a> in the <i>API Reference for Amazon Web Services Private 5G</i>.
     * </p>
     *
     * @param deleteNetworkRequest
     * @return A Java Future containing the result of the DeleteNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.DeleteNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeleteNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNetworkResponse> deleteNetwork(DeleteNetworkRequest deleteNetworkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNetworkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNetworkRequest, DeleteNetworkResponse>()
                            .withOperationName("DeleteNetwork").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteNetworkRequest));
            CompletableFuture<DeleteNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified network site. Return the hardware after you delete the network site. You are responsible
     * for minimum charges. For more information, see <a
     * href="https://docs.aws.amazon.com/private-networks/latest/userguide/hardware-maintenance.html">Hardware
     * returns</a> in the <i>Amazon Web Services Private 5G User Guide</i>.
     * </p>
     *
     * @param deleteNetworkSiteRequest
     * @return A Java Future containing the result of the DeleteNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.DeleteNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeleteNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNetworkSiteResponse> deleteNetworkSite(DeleteNetworkSiteRequest deleteNetworkSiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNetworkSiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNetworkSiteRequest, DeleteNetworkSiteResponse>()
                            .withOperationName("DeleteNetworkSite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteNetworkSiteRequest));
            CompletableFuture<DeleteNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified device identifier.
     * </p>
     *
     * @param getDeviceIdentifierRequest
     * @return A Java Future containing the result of the GetDeviceIdentifier operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeviceIdentifierResponse> getDeviceIdentifier(
            GetDeviceIdentifierRequest getDeviceIdentifierRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeviceIdentifierRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeviceIdentifier");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDeviceIdentifierResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeviceIdentifierResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeviceIdentifierRequest, GetDeviceIdentifierResponse>()
                            .withOperationName("GetDeviceIdentifier").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDeviceIdentifierRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDeviceIdentifierRequest));
            CompletableFuture<GetDeviceIdentifierResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified network.
     * </p>
     *
     * @param getNetworkRequest
     * @return A Java Future containing the result of the GetNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNetworkResponse> getNetwork(GetNetworkRequest getNetworkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNetworkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNetworkRequest, GetNetworkResponse>().withOperationName("GetNetwork")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getNetworkRequest));
            CompletableFuture<GetNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified network resource.
     * </p>
     *
     * @param getNetworkResourceRequest
     * @return A Java Future containing the result of the GetNetworkResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetNetworkResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetworkResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNetworkResourceResponse> getNetworkResource(GetNetworkResourceRequest getNetworkResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNetworkResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetworkResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNetworkResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetNetworkResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNetworkResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNetworkResourceRequest, GetNetworkResourceResponse>()
                            .withOperationName("GetNetworkResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetNetworkResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getNetworkResourceRequest));
            CompletableFuture<GetNetworkResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified network site.
     * </p>
     *
     * @param getNetworkSiteRequest
     * @return A Java Future containing the result of the GetNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNetworkSiteResponse> getNetworkSite(GetNetworkSiteRequest getNetworkSiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNetworkSiteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNetworkSiteRequest, GetNetworkSiteResponse>()
                            .withOperationName("GetNetworkSite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getNetworkSiteRequest));
            CompletableFuture<GetNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified order.
     * </p>
     *
     * @param getOrderRequest
     * @return A Java Future containing the result of the GetOrder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetOrderResponse> getOrder(GetOrderRequest getOrderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOrderRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOrder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetOrderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOrderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOrderRequest, GetOrderResponse>().withOperationName("GetOrder")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetOrderRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getOrderRequest));
            CompletableFuture<GetOrderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists device identifiers. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order, the status of device identifiers, or the ARN of the traffic
     * group.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     *
     * @param listDeviceIdentifiersRequest
     * @return A Java Future containing the result of the ListDeviceIdentifiers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListDeviceIdentifiers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListDeviceIdentifiers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeviceIdentifiersResponse> listDeviceIdentifiers(
            ListDeviceIdentifiersRequest listDeviceIdentifiersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeviceIdentifiersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeviceIdentifiersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeviceIdentifiers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeviceIdentifiersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeviceIdentifiersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeviceIdentifiersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeviceIdentifiersRequest, ListDeviceIdentifiersResponse>()
                            .withOperationName("ListDeviceIdentifiers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDeviceIdentifiersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDeviceIdentifiersRequest));
            CompletableFuture<ListDeviceIdentifiersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists network resources. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order or the status of network resources.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     *
     * @param listNetworkResourcesRequest
     * @return A Java Future containing the result of the ListNetworkResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworkResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNetworkResourcesResponse> listNetworkResources(
            ListNetworkResourcesRequest listNetworkResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNetworkResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworkResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworkResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNetworkResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNetworkResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNetworkResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNetworkResourcesRequest, ListNetworkResourcesResponse>()
                            .withOperationName("ListNetworkResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListNetworkResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNetworkResourcesRequest));
            CompletableFuture<ListNetworkResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists network sites. Add filters to your request to return a more specific list of results. Use filters to match
     * the status of the network site.
     * </p>
     *
     * @param listNetworkSitesRequest
     * @return A Java Future containing the result of the ListNetworkSites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworkSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkSites"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNetworkSitesResponse> listNetworkSites(ListNetworkSitesRequest listNetworkSitesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNetworkSitesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworkSitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworkSites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNetworkSitesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNetworkSitesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNetworkSitesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNetworkSitesRequest, ListNetworkSitesResponse>()
                            .withOperationName("ListNetworkSites").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListNetworkSitesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNetworkSitesRequest));
            CompletableFuture<ListNetworkSitesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists networks. Add filters to your request to return a more specific list of results. Use filters to match the
     * status of the network.
     * </p>
     *
     * @param listNetworksRequest
     * @return A Java Future containing the result of the ListNetworks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNetworksResponse> listNetworks(ListNetworksRequest listNetworksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNetworksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNetworksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListNetworksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNetworksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNetworksRequest, ListNetworksResponse>()
                            .withOperationName("ListNetworks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListNetworksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNetworksRequest));
            CompletableFuture<ListNetworksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists orders. Add filters to your request to return a more specific list of results. Use filters to match the
     * Amazon Resource Name (ARN) of the network site or the status of the order.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     *
     * @param listOrdersRequest
     * @return A Java Future containing the result of the ListOrders operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListOrders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrdersResponse> listOrders(ListOrdersRequest listOrdersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOrdersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOrdersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrdersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListOrdersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrdersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrdersRequest, ListOrdersResponse>().withOperationName("ListOrders")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOrdersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOrdersRequest));
            CompletableFuture<ListOrdersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks the health of the service.
     * </p>
     *
     * @param pingRequest
     * @return A Java Future containing the result of the Ping operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.Ping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/Ping" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PingResponse> ping(PingRequest pingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(pingRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pingRequest.overrideConfiguration()
                .orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Ping");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PingRequest, PingResponse>().withOperationName("Ping")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new PingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(pingRequest));
            CompletableFuture<PingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this action to do the following tasks:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Update the duration and renewal status of the commitment period for a radio unit. The update goes into effect
     * immediately.
     * </p>
     * </li>
     * <li>
     * <p>
     * Request a replacement for a network resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * Request that you return a network resource.
     * </p>
     * </li>
     * </ul>
     * <p>
     * After you submit a request to replace or return a network resource, the status of the network resource changes to
     * <code>CREATING_SHIPPING_LABEL</code>. The shipping label is available when the status of the network resource is
     * <code>PENDING_RETURN</code>. After the network resource is successfully returned, its status changes to
     * <code>DELETED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/private-networks/latest/userguide/radio-units.html#return-radio-unit">Return a
     * radio unit</a>.
     * </p>
     *
     * @param startNetworkResourceUpdateRequest
     * @return A Java Future containing the result of the StartNetworkResourceUpdate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.StartNetworkResourceUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/StartNetworkResourceUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartNetworkResourceUpdateResponse> startNetworkResourceUpdate(
            StartNetworkResourceUpdateRequest startNetworkResourceUpdateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startNetworkResourceUpdateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startNetworkResourceUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartNetworkResourceUpdate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartNetworkResourceUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartNetworkResourceUpdateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartNetworkResourceUpdateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartNetworkResourceUpdateRequest, StartNetworkResourceUpdateResponse>()
                            .withOperationName("StartNetworkResourceUpdate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartNetworkResourceUpdateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startNetworkResourceUpdateRequest));
            CompletableFuture<StartNetworkResourceUpdateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified network site.
     * </p>
     *
     * @param updateNetworkSiteRequest
     * @return A Java Future containing the result of the UpdateNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.UpdateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UpdateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNetworkSiteResponse> updateNetworkSite(UpdateNetworkSiteRequest updateNetworkSiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateNetworkSiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNetworkSiteRequest, UpdateNetworkSiteResponse>()
                            .withOperationName("UpdateNetworkSite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateNetworkSiteRequest));
            CompletableFuture<UpdateNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified network site plan.
     * </p>
     *
     * @param updateNetworkSitePlanRequest
     * @return A Java Future containing the result of the UpdateNetworkSitePlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.UpdateNetworkSitePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UpdateNetworkSitePlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNetworkSitePlanResponse> updateNetworkSitePlan(
            UpdateNetworkSitePlanRequest updateNetworkSitePlanRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateNetworkSitePlanRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNetworkSitePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNetworkSitePlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNetworkSitePlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNetworkSitePlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNetworkSitePlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNetworkSitePlanRequest, UpdateNetworkSitePlanResponse>()
                            .withOperationName("UpdateNetworkSitePlan").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateNetworkSitePlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateNetworkSitePlanRequest));
            CompletableFuture<UpdateNetworkSitePlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PrivateNetworksServiceClientConfiguration serviceClientConfiguration() {
        return new PrivateNetworksServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PrivateNetworksException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        PrivateNetworksServiceClientConfigurationBuilder serviceConfigBuilder = new PrivateNetworksServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
