/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpointemail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * In Amazon Pinpoint, <i>events</i> include message sends, deliveries, opens, clicks, bounces, and complaints. <i>Event
 * destinations</i> are places that you can send information about these events to. For example, you can send event data
 * to Amazon SNS to receive notifications when you receive bounces or complaints, or you can use Amazon Kinesis Data
 * Firehose to stream data to Amazon S3 for long-term storage.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventDestination implements SdkPojo, Serializable,
        ToCopyableBuilder<EventDestination.Builder, EventDestination> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(EventDestination::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(EventDestination::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<List<String>> MATCHING_EVENT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MatchingEventTypes")
            .getter(getter(EventDestination::matchingEventTypesAsStrings))
            .setter(setter(Builder::matchingEventTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchingEventTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<KinesisFirehoseDestination> KINESIS_FIREHOSE_DESTINATION_FIELD = SdkField
            .<KinesisFirehoseDestination> builder(MarshallingType.SDK_POJO)
            .memberName("KinesisFirehoseDestination")
            .getter(getter(EventDestination::kinesisFirehoseDestination))
            .setter(setter(Builder::kinesisFirehoseDestination))
            .constructor(KinesisFirehoseDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KinesisFirehoseDestination").build())
            .build();

    private static final SdkField<CloudWatchDestination> CLOUD_WATCH_DESTINATION_FIELD = SdkField
            .<CloudWatchDestination> builder(MarshallingType.SDK_POJO).memberName("CloudWatchDestination")
            .getter(getter(EventDestination::cloudWatchDestination)).setter(setter(Builder::cloudWatchDestination))
            .constructor(CloudWatchDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchDestination").build())
            .build();

    private static final SdkField<SnsDestination> SNS_DESTINATION_FIELD = SdkField
            .<SnsDestination> builder(MarshallingType.SDK_POJO).memberName("SnsDestination")
            .getter(getter(EventDestination::snsDestination)).setter(setter(Builder::snsDestination))
            .constructor(SnsDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsDestination").build()).build();

    private static final SdkField<PinpointDestination> PINPOINT_DESTINATION_FIELD = SdkField
            .<PinpointDestination> builder(MarshallingType.SDK_POJO).memberName("PinpointDestination")
            .getter(getter(EventDestination::pinpointDestination)).setter(setter(Builder::pinpointDestination))
            .constructor(PinpointDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PinpointDestination").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ENABLED_FIELD,
            MATCHING_EVENT_TYPES_FIELD, KINESIS_FIREHOSE_DESTINATION_FIELD, CLOUD_WATCH_DESTINATION_FIELD, SNS_DESTINATION_FIELD,
            PINPOINT_DESTINATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Boolean enabled;

    private final List<String> matchingEventTypes;

    private final KinesisFirehoseDestination kinesisFirehoseDestination;

    private final CloudWatchDestination cloudWatchDestination;

    private final SnsDestination snsDestination;

    private final PinpointDestination pinpointDestination;

    private EventDestination(BuilderImpl builder) {
        this.name = builder.name;
        this.enabled = builder.enabled;
        this.matchingEventTypes = builder.matchingEventTypes;
        this.kinesisFirehoseDestination = builder.kinesisFirehoseDestination;
        this.cloudWatchDestination = builder.cloudWatchDestination;
        this.snsDestination = builder.snsDestination;
        this.pinpointDestination = builder.pinpointDestination;
    }

    /**
     * <p>
     * A name that identifies the event destination.
     * </p>
     * 
     * @return A name that identifies the event destination.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * If <code>true</code>, the event destination is enabled. When the event destination is enabled, the specified
     * event types are sent to the destinations in this <code>EventDestinationDefinition</code>.
     * </p>
     * <p>
     * If <code>false</code>, the event destination is disabled. When the event destination is disabled, events aren't
     * sent to the specified destinations.
     * </p>
     * 
     * @return If <code>true</code>, the event destination is enabled. When the event destination is enabled, the
     *         specified event types are sent to the destinations in this <code>EventDestinationDefinition</code>.</p>
     *         <p>
     *         If <code>false</code>, the event destination is disabled. When the event destination is disabled, events
     *         aren't sent to the specified destinations.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The types of events that Amazon Pinpoint sends to the specified event destinations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMatchingEventTypes} method.
     * </p>
     * 
     * @return The types of events that Amazon Pinpoint sends to the specified event destinations.
     */
    public final List<EventType> matchingEventTypes() {
        return EventTypesCopier.copyStringToEnum(matchingEventTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the MatchingEventTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMatchingEventTypes() {
        return matchingEventTypes != null && !(matchingEventTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The types of events that Amazon Pinpoint sends to the specified event destinations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMatchingEventTypes} method.
     * </p>
     * 
     * @return The types of events that Amazon Pinpoint sends to the specified event destinations.
     */
    public final List<String> matchingEventTypesAsStrings() {
        return matchingEventTypes;
    }

    /**
     * <p>
     * An object that defines an Amazon Kinesis Data Firehose destination for email events. You can use Amazon Kinesis
     * Data Firehose to stream data to other services, such as Amazon S3 and Amazon Redshift.
     * </p>
     * 
     * @return An object that defines an Amazon Kinesis Data Firehose destination for email events. You can use Amazon
     *         Kinesis Data Firehose to stream data to other services, such as Amazon S3 and Amazon Redshift.
     */
    public final KinesisFirehoseDestination kinesisFirehoseDestination() {
        return kinesisFirehoseDestination;
    }

    /**
     * <p>
     * An object that defines an Amazon CloudWatch destination for email events. You can use Amazon CloudWatch to
     * monitor and gain insights on your email sending metrics.
     * </p>
     * 
     * @return An object that defines an Amazon CloudWatch destination for email events. You can use Amazon CloudWatch
     *         to monitor and gain insights on your email sending metrics.
     */
    public final CloudWatchDestination cloudWatchDestination() {
        return cloudWatchDestination;
    }

    /**
     * <p>
     * An object that defines an Amazon SNS destination for email events. You can use Amazon SNS to send notification
     * when certain email events occur.
     * </p>
     * 
     * @return An object that defines an Amazon SNS destination for email events. You can use Amazon SNS to send
     *         notification when certain email events occur.
     */
    public final SnsDestination snsDestination() {
        return snsDestination;
    }

    /**
     * <p>
     * An object that defines a Amazon Pinpoint destination for email events. You can use Amazon Pinpoint events to
     * create attributes in Amazon Pinpoint projects. You can use these attributes to create segments for your
     * campaigns.
     * </p>
     * 
     * @return An object that defines a Amazon Pinpoint destination for email events. You can use Amazon Pinpoint events
     *         to create attributes in Amazon Pinpoint projects. You can use these attributes to create segments for
     *         your campaigns.
     */
    public final PinpointDestination pinpointDestination() {
        return pinpointDestination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasMatchingEventTypes() ? matchingEventTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kinesisFirehoseDestination());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchDestination());
        hashCode = 31 * hashCode + Objects.hashCode(snsDestination());
        hashCode = 31 * hashCode + Objects.hashCode(pinpointDestination());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventDestination)) {
            return false;
        }
        EventDestination other = (EventDestination) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(enabled(), other.enabled())
                && hasMatchingEventTypes() == other.hasMatchingEventTypes()
                && Objects.equals(matchingEventTypesAsStrings(), other.matchingEventTypesAsStrings())
                && Objects.equals(kinesisFirehoseDestination(), other.kinesisFirehoseDestination())
                && Objects.equals(cloudWatchDestination(), other.cloudWatchDestination())
                && Objects.equals(snsDestination(), other.snsDestination())
                && Objects.equals(pinpointDestination(), other.pinpointDestination());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EventDestination").add("Name", name()).add("Enabled", enabled())
                .add("MatchingEventTypes", hasMatchingEventTypes() ? matchingEventTypesAsStrings() : null)
                .add("KinesisFirehoseDestination", kinesisFirehoseDestination())
                .add("CloudWatchDestination", cloudWatchDestination()).add("SnsDestination", snsDestination())
                .add("PinpointDestination", pinpointDestination()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "MatchingEventTypes":
            return Optional.ofNullable(clazz.cast(matchingEventTypesAsStrings()));
        case "KinesisFirehoseDestination":
            return Optional.ofNullable(clazz.cast(kinesisFirehoseDestination()));
        case "CloudWatchDestination":
            return Optional.ofNullable(clazz.cast(cloudWatchDestination()));
        case "SnsDestination":
            return Optional.ofNullable(clazz.cast(snsDestination()));
        case "PinpointDestination":
            return Optional.ofNullable(clazz.cast(pinpointDestination()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventDestination, T> g) {
        return obj -> g.apply((EventDestination) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventDestination> {
        /**
         * <p>
         * A name that identifies the event destination.
         * </p>
         * 
         * @param name
         *        A name that identifies the event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If <code>true</code>, the event destination is enabled. When the event destination is enabled, the specified
         * event types are sent to the destinations in this <code>EventDestinationDefinition</code>.
         * </p>
         * <p>
         * If <code>false</code>, the event destination is disabled. When the event destination is disabled, events
         * aren't sent to the specified destinations.
         * </p>
         * 
         * @param enabled
         *        If <code>true</code>, the event destination is enabled. When the event destination is enabled, the
         *        specified event types are sent to the destinations in this <code>EventDestinationDefinition</code>
         *        .</p>
         *        <p>
         *        If <code>false</code>, the event destination is disabled. When the event destination is disabled,
         *        events aren't sent to the specified destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The types of events that Amazon Pinpoint sends to the specified event destinations.
         * </p>
         * 
         * @param matchingEventTypes
         *        The types of events that Amazon Pinpoint sends to the specified event destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypesWithStrings(Collection<String> matchingEventTypes);

        /**
         * <p>
         * The types of events that Amazon Pinpoint sends to the specified event destinations.
         * </p>
         * 
         * @param matchingEventTypes
         *        The types of events that Amazon Pinpoint sends to the specified event destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypesWithStrings(String... matchingEventTypes);

        /**
         * <p>
         * The types of events that Amazon Pinpoint sends to the specified event destinations.
         * </p>
         * 
         * @param matchingEventTypes
         *        The types of events that Amazon Pinpoint sends to the specified event destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypes(Collection<EventType> matchingEventTypes);

        /**
         * <p>
         * The types of events that Amazon Pinpoint sends to the specified event destinations.
         * </p>
         * 
         * @param matchingEventTypes
         *        The types of events that Amazon Pinpoint sends to the specified event destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypes(EventType... matchingEventTypes);

        /**
         * <p>
         * An object that defines an Amazon Kinesis Data Firehose destination for email events. You can use Amazon
         * Kinesis Data Firehose to stream data to other services, such as Amazon S3 and Amazon Redshift.
         * </p>
         * 
         * @param kinesisFirehoseDestination
         *        An object that defines an Amazon Kinesis Data Firehose destination for email events. You can use
         *        Amazon Kinesis Data Firehose to stream data to other services, such as Amazon S3 and Amazon Redshift.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination);

        /**
         * <p>
         * An object that defines an Amazon Kinesis Data Firehose destination for email events. You can use Amazon
         * Kinesis Data Firehose to stream data to other services, such as Amazon S3 and Amazon Redshift.
         * </p>
         * This is a convenience method that creates an instance of the {@link KinesisFirehoseDestination.Builder}
         * avoiding the need to create one manually via {@link KinesisFirehoseDestination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KinesisFirehoseDestination.Builder#build()} is called immediately
         * and its result is passed to {@link #kinesisFirehoseDestination(KinesisFirehoseDestination)}.
         * 
         * @param kinesisFirehoseDestination
         *        a consumer that will call methods on {@link KinesisFirehoseDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kinesisFirehoseDestination(KinesisFirehoseDestination)
         */
        default Builder kinesisFirehoseDestination(Consumer<KinesisFirehoseDestination.Builder> kinesisFirehoseDestination) {
            return kinesisFirehoseDestination(KinesisFirehoseDestination.builder().applyMutation(kinesisFirehoseDestination)
                    .build());
        }

        /**
         * <p>
         * An object that defines an Amazon CloudWatch destination for email events. You can use Amazon CloudWatch to
         * monitor and gain insights on your email sending metrics.
         * </p>
         * 
         * @param cloudWatchDestination
         *        An object that defines an Amazon CloudWatch destination for email events. You can use Amazon
         *        CloudWatch to monitor and gain insights on your email sending metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchDestination(CloudWatchDestination cloudWatchDestination);

        /**
         * <p>
         * An object that defines an Amazon CloudWatch destination for email events. You can use Amazon CloudWatch to
         * monitor and gain insights on your email sending metrics.
         * </p>
         * This is a convenience method that creates an instance of the {@link CloudWatchDestination.Builder} avoiding
         * the need to create one manually via {@link CloudWatchDestination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CloudWatchDestination.Builder#build()} is called immediately and
         * its result is passed to {@link #cloudWatchDestination(CloudWatchDestination)}.
         * 
         * @param cloudWatchDestination
         *        a consumer that will call methods on {@link CloudWatchDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchDestination(CloudWatchDestination)
         */
        default Builder cloudWatchDestination(Consumer<CloudWatchDestination.Builder> cloudWatchDestination) {
            return cloudWatchDestination(CloudWatchDestination.builder().applyMutation(cloudWatchDestination).build());
        }

        /**
         * <p>
         * An object that defines an Amazon SNS destination for email events. You can use Amazon SNS to send
         * notification when certain email events occur.
         * </p>
         * 
         * @param snsDestination
         *        An object that defines an Amazon SNS destination for email events. You can use Amazon SNS to send
         *        notification when certain email events occur.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsDestination(SnsDestination snsDestination);

        /**
         * <p>
         * An object that defines an Amazon SNS destination for email events. You can use Amazon SNS to send
         * notification when certain email events occur.
         * </p>
         * This is a convenience method that creates an instance of the {@link SnsDestination.Builder} avoiding the need
         * to create one manually via {@link SnsDestination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SnsDestination.Builder#build()} is called immediately and its
         * result is passed to {@link #snsDestination(SnsDestination)}.
         * 
         * @param snsDestination
         *        a consumer that will call methods on {@link SnsDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #snsDestination(SnsDestination)
         */
        default Builder snsDestination(Consumer<SnsDestination.Builder> snsDestination) {
            return snsDestination(SnsDestination.builder().applyMutation(snsDestination).build());
        }

        /**
         * <p>
         * An object that defines a Amazon Pinpoint destination for email events. You can use Amazon Pinpoint events to
         * create attributes in Amazon Pinpoint projects. You can use these attributes to create segments for your
         * campaigns.
         * </p>
         * 
         * @param pinpointDestination
         *        An object that defines a Amazon Pinpoint destination for email events. You can use Amazon Pinpoint
         *        events to create attributes in Amazon Pinpoint projects. You can use these attributes to create
         *        segments for your campaigns.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pinpointDestination(PinpointDestination pinpointDestination);

        /**
         * <p>
         * An object that defines a Amazon Pinpoint destination for email events. You can use Amazon Pinpoint events to
         * create attributes in Amazon Pinpoint projects. You can use these attributes to create segments for your
         * campaigns.
         * </p>
         * This is a convenience method that creates an instance of the {@link PinpointDestination.Builder} avoiding the
         * need to create one manually via {@link PinpointDestination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PinpointDestination.Builder#build()} is called immediately and
         * its result is passed to {@link #pinpointDestination(PinpointDestination)}.
         * 
         * @param pinpointDestination
         *        a consumer that will call methods on {@link PinpointDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pinpointDestination(PinpointDestination)
         */
        default Builder pinpointDestination(Consumer<PinpointDestination.Builder> pinpointDestination) {
            return pinpointDestination(PinpointDestination.builder().applyMutation(pinpointDestination).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Boolean enabled;

        private List<String> matchingEventTypes = DefaultSdkAutoConstructList.getInstance();

        private KinesisFirehoseDestination kinesisFirehoseDestination;

        private CloudWatchDestination cloudWatchDestination;

        private SnsDestination snsDestination;

        private PinpointDestination pinpointDestination;

        private BuilderImpl() {
        }

        private BuilderImpl(EventDestination model) {
            name(model.name);
            enabled(model.enabled);
            matchingEventTypesWithStrings(model.matchingEventTypes);
            kinesisFirehoseDestination(model.kinesisFirehoseDestination);
            cloudWatchDestination(model.cloudWatchDestination);
            snsDestination(model.snsDestination);
            pinpointDestination(model.pinpointDestination);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Collection<String> getMatchingEventTypes() {
            if (matchingEventTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return matchingEventTypes;
        }

        public final void setMatchingEventTypes(Collection<String> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copy(matchingEventTypes);
        }

        @Override
        public final Builder matchingEventTypesWithStrings(Collection<String> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copy(matchingEventTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchingEventTypesWithStrings(String... matchingEventTypes) {
            matchingEventTypesWithStrings(Arrays.asList(matchingEventTypes));
            return this;
        }

        @Override
        public final Builder matchingEventTypes(Collection<EventType> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copyEnumToString(matchingEventTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchingEventTypes(EventType... matchingEventTypes) {
            matchingEventTypes(Arrays.asList(matchingEventTypes));
            return this;
        }

        public final KinesisFirehoseDestination.Builder getKinesisFirehoseDestination() {
            return kinesisFirehoseDestination != null ? kinesisFirehoseDestination.toBuilder() : null;
        }

        public final void setKinesisFirehoseDestination(KinesisFirehoseDestination.BuilderImpl kinesisFirehoseDestination) {
            this.kinesisFirehoseDestination = kinesisFirehoseDestination != null ? kinesisFirehoseDestination.build() : null;
        }

        @Override
        public final Builder kinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination) {
            this.kinesisFirehoseDestination = kinesisFirehoseDestination;
            return this;
        }

        public final CloudWatchDestination.Builder getCloudWatchDestination() {
            return cloudWatchDestination != null ? cloudWatchDestination.toBuilder() : null;
        }

        public final void setCloudWatchDestination(CloudWatchDestination.BuilderImpl cloudWatchDestination) {
            this.cloudWatchDestination = cloudWatchDestination != null ? cloudWatchDestination.build() : null;
        }

        @Override
        public final Builder cloudWatchDestination(CloudWatchDestination cloudWatchDestination) {
            this.cloudWatchDestination = cloudWatchDestination;
            return this;
        }

        public final SnsDestination.Builder getSnsDestination() {
            return snsDestination != null ? snsDestination.toBuilder() : null;
        }

        public final void setSnsDestination(SnsDestination.BuilderImpl snsDestination) {
            this.snsDestination = snsDestination != null ? snsDestination.build() : null;
        }

        @Override
        public final Builder snsDestination(SnsDestination snsDestination) {
            this.snsDestination = snsDestination;
            return this;
        }

        public final PinpointDestination.Builder getPinpointDestination() {
            return pinpointDestination != null ? pinpointDestination.toBuilder() : null;
        }

        public final void setPinpointDestination(PinpointDestination.BuilderImpl pinpointDestination) {
            this.pinpointDestination = pinpointDestination != null ? pinpointDestination.build() : null;
        }

        @Override
        public final Builder pinpointDestination(PinpointDestination pinpointDestination) {
            this.pinpointDestination = pinpointDestination;
            return this;
        }

        @Override
        public EventDestination build() {
            return new EventDestination(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
