/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pcaconnectorscep;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.pcaconnectorscep.internal.PcaConnectorScepServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pcaconnectorscep.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.pcaconnectorscep.model.AccessDeniedException;
import software.amazon.awssdk.services.pcaconnectorscep.model.BadRequestException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ConflictException;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateChallengeRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateChallengeResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteChallengeRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteChallengeResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengeMetadataRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengeMetadataResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengePasswordRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengePasswordResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.InternalServerException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.PcaConnectorScepException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ResourceNotFoundException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.pcaconnectorscep.model.TagResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.TagResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ThrottlingException;
import software.amazon.awssdk.services.pcaconnectorscep.model.UntagResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.UntagResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ValidationException;
import software.amazon.awssdk.services.pcaconnectorscep.transform.CreateChallengeRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.CreateConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.DeleteChallengeRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.DeleteConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.GetChallengeMetadataRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.GetChallengePasswordRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.GetConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.ListChallengeMetadataRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.ListConnectorsRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PcaConnectorScepClient}.
 *
 * @see PcaConnectorScepClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPcaConnectorScepClient implements PcaConnectorScepClient {
    private static final Logger log = Logger.loggerFor(DefaultPcaConnectorScepClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPcaConnectorScepClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Pca_Connector_Scep" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * For general-purpose connectors. Creates a <i>challenge password</i> for the specified connector. The SCEP
     * protocol uses a challenge password to authenticate a request before issuing a certificate from a certificate
     * authority (CA). Your SCEP clients include the challenge password as part of their certificate request to
     * Connector for SCEP. To retrieve the connector Amazon Resource Names (ARNs) for the connectors in your account,
     * call <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_ListConnectors.html"
     * >ListConnectors</a>.
     * </p>
     * <p>
     * To create additional challenge passwords for the connector, call <code>CreateChallenge</code> again. We recommend
     * frequently rotating your challenge passwords.
     * </p>
     *
     * @param createChallengeRequest
     * @return Result of the CreateChallenge operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.CreateChallenge
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/CreateChallenge"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateChallengeResponse createChallenge(CreateChallengeRequest createChallengeRequest)
            throws ResourceNotFoundException, BadRequestException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateChallengeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateChallengeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChallengeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChallengeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChallenge");

            return clientHandler.execute(new ClientExecutionParams<CreateChallengeRequest, CreateChallengeResponse>()
                    .withOperationName("CreateChallenge").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createChallengeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateChallengeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a SCEP connector. A SCEP connector links Amazon Web Services Private Certificate Authority to your
     * SCEP-compatible devices and mobile device management (MDM) systems. Before you create a connector, you must
     * complete a set of prerequisites, including creation of a private certificate authority (CA) to use with this
     * connector. For more information, see <a href=
     * "https://docs.aws.amazon.com/privateca/latest/userguide/scep-connector.htmlconnector-for-scep-prerequisites.html"
     * >Connector for SCEP prerequisites</a>.
     * </p>
     *
     * @param createConnectorRequest
     * @return Result of the CreateConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.CreateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/CreateConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateConnectorResponse createConnector(CreateConnectorRequest createConnectorRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateConnectorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnector");

            return clientHandler.execute(new ClientExecutionParams<CreateConnectorRequest, CreateConnectorResponse>()
                    .withOperationName("CreateConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param deleteChallengeRequest
     * @return Result of the DeleteChallenge operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.DeleteChallenge
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/DeleteChallenge"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteChallengeResponse deleteChallenge(DeleteChallengeRequest deleteChallengeRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteChallengeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteChallengeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChallengeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChallengeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChallenge");

            return clientHandler.execute(new ClientExecutionParams<DeleteChallengeRequest, DeleteChallengeResponse>()
                    .withOperationName("DeleteChallenge").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteChallengeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteChallengeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Connector.html"
     * >Connector</a>. This operation also deletes any challenges associated with the connector.
     * </p>
     *
     * @param deleteConnectorRequest
     * @return Result of the DeleteConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.DeleteConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/DeleteConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteConnectorResponse deleteConnector(DeleteConnectorRequest deleteConnectorRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteConnectorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnector");

            return clientHandler.execute(new ClientExecutionParams<DeleteConnectorRequest, DeleteConnectorResponse>()
                    .withOperationName("DeleteConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the metadata for the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param getChallengeMetadataRequest
     * @return Result of the GetChallengeMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetChallengeMetadataResponse getChallengeMetadata(GetChallengeMetadataRequest getChallengeMetadataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChallengeMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetChallengeMetadataResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChallengeMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChallengeMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChallengeMetadata");

            return clientHandler.execute(new ClientExecutionParams<GetChallengeMetadataRequest, GetChallengeMetadataResponse>()
                    .withOperationName("GetChallengeMetadata").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getChallengeMetadataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetChallengeMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the challenge password for the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param getChallengePasswordRequest
     * @return Result of the GetChallengePassword operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetChallengePassword
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetChallengePassword"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetChallengePasswordResponse getChallengePassword(GetChallengePasswordRequest getChallengePasswordRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChallengePasswordResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetChallengePasswordResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChallengePasswordRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChallengePasswordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChallengePassword");

            return clientHandler.execute(new ClientExecutionParams<GetChallengePasswordRequest, GetChallengePasswordResponse>()
                    .withOperationName("GetChallengePassword").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getChallengePasswordRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetChallengePasswordRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details about the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Connector.html"
     * >Connector</a>. Calling this action returns important details about the connector, such as the public SCEP URL
     * where your clients can request certificates.
     * </p>
     *
     * @param getConnectorRequest
     * @return Result of the GetConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetConnectorResponse getConnector(GetConnectorRequest getConnectorRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConnectorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConnectorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnector");

            return clientHandler.execute(new ClientExecutionParams<GetConnectorRequest, GetConnectorResponse>()
                    .withOperationName("GetConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the challenge metadata for the specified ARN.
     * </p>
     *
     * @param listChallengeMetadataRequest
     * @return Result of the ListChallengeMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListChallengeMetadataResponse listChallengeMetadata(ListChallengeMetadataRequest listChallengeMetadataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChallengeMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListChallengeMetadataResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChallengeMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChallengeMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChallengeMetadata");

            return clientHandler.execute(new ClientExecutionParams<ListChallengeMetadataRequest, ListChallengeMetadataResponse>()
                    .withOperationName("ListChallengeMetadata").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listChallengeMetadataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChallengeMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the connectors belonging to your Amazon Web Services account.
     * </p>
     *
     * @param listConnectorsRequest
     * @return Result of the ListConnectors operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListConnectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListConnectorsResponse listConnectors(ListConnectorsRequest listConnectorsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListConnectorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectors");

            return clientHandler.execute(new ClientExecutionParams<ListConnectorsRequest, ListConnectorsResponse>()
                    .withOperationName("ListConnectors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listConnectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConnectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the tags associated with the specified resource. Tags are key-value pairs that you can use to
     * categorize and manage your resources, for purposes like billing. For example, you might set the tag key to
     * "customer" and the value to the customer name or ID. You can specify one or more tags to add to each Amazon Web
     * Services resource, up to 50 tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to your resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from your resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        PcaConnectorScepServiceClientConfigurationBuilder serviceConfigBuilder = new PcaConnectorScepServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PcaConnectorScepException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    @Override
    public final PcaConnectorScepServiceClientConfiguration serviceClientConfiguration() {
        return new PcaConnectorScepServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
