/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptographydata;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.paymentcryptographydata.internal.PaymentCryptographyDataServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.paymentcryptographydata.model.AccessDeniedException;
import software.amazon.awssdk.services.paymentcryptographydata.model.DecryptDataRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.DecryptDataResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.EncryptDataRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.EncryptDataResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.GenerateCardValidationDataRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.GenerateCardValidationDataResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.GenerateMacRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.GenerateMacResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.GeneratePinDataRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.GeneratePinDataResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.InternalServerException;
import software.amazon.awssdk.services.paymentcryptographydata.model.PaymentCryptographyDataException;
import software.amazon.awssdk.services.paymentcryptographydata.model.ReEncryptDataRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.ReEncryptDataResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.ResourceNotFoundException;
import software.amazon.awssdk.services.paymentcryptographydata.model.ThrottlingException;
import software.amazon.awssdk.services.paymentcryptographydata.model.TranslatePinDataRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.TranslatePinDataResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.ValidationException;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerificationFailedException;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerifyAuthRequestCryptogramRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerifyAuthRequestCryptogramResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerifyCardValidationDataRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerifyCardValidationDataResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerifyMacRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerifyMacResponse;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerifyPinDataRequest;
import software.amazon.awssdk.services.paymentcryptographydata.model.VerifyPinDataResponse;
import software.amazon.awssdk.services.paymentcryptographydata.transform.DecryptDataRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.EncryptDataRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.GenerateCardValidationDataRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.GenerateMacRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.GeneratePinDataRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.ReEncryptDataRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.TranslatePinDataRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.VerifyAuthRequestCryptogramRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.VerifyCardValidationDataRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.VerifyMacRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptographydata.transform.VerifyPinDataRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PaymentCryptographyDataAsyncClient}.
 *
 * @see PaymentCryptographyDataAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPaymentCryptographyDataAsyncClient implements PaymentCryptographyDataAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPaymentCryptographyDataAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final PaymentCryptographyDataServiceClientConfiguration serviceClientConfiguration;

    protected DefaultPaymentCryptographyDataAsyncClient(
            PaymentCryptographyDataServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Decrypts ciphertext data to plaintext using symmetric, asymmetric, or DUKPT data encryption key. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/decrypt-data.html">Decrypt data</a> in
     * the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * You can use an encryption key generated within Amazon Web Services Payment Cryptography, or you can import your
     * own encryption key by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>. For
     * this operation, the key must have <code>KeyModesOfUse</code> set to <code>Decrypt</code>. In asymmetric
     * decryption, Amazon Web Services Payment Cryptography decrypts the ciphertext using the private component of the
     * asymmetric encryption key pair. For data encryption outside of Amazon Web Services Payment Cryptography, you can
     * export the public component of the asymmetric key pair by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetPublicKeyCertificate.html"
     * >GetPublicCertificate</a>.
     * </p>
     * <p>
     * For symmetric and DUKPT decryption, Amazon Web Services Payment Cryptography supports <code>TDES</code> and
     * <code>AES</code> algorithms. For asymmetric decryption, Amazon Web Services Payment Cryptography supports
     * <code>RSA</code>. When you use DUKPT, for <code>TDES</code> algorithm, the ciphertext data length must be a
     * multiple of 16 bytes. For <code>AES</code> algorithm, the ciphertext data length must be a multiple of 32 bytes.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>EncryptData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetPublicKeyCertificate.html">
     * GetPublicCertificate</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param decryptDataRequest
     * @return A Java Future containing the result of the DecryptData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.DecryptData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/DecryptData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DecryptDataResponse> decryptData(DecryptDataRequest decryptDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(decryptDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, decryptDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DecryptData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DecryptDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DecryptDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DecryptDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DecryptDataRequest, DecryptDataResponse>()
                            .withOperationName("DecryptData").withMarshaller(new DecryptDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(decryptDataRequest));
            CompletableFuture<DecryptDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Encrypts plaintext data to ciphertext using symmetric, asymmetric, or DUKPT data encryption key. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/encrypt-data.html">Encrypt data</a> in
     * the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * You can generate an encryption key within Amazon Web Services Payment Cryptography by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a>. You
     * can import your own encryption key by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>. For
     * this operation, the key must have <code>KeyModesOfUse</code> set to <code>Encrypt</code>. In asymmetric
     * encryption, plaintext is encrypted using public component. You can import the public component of an asymmetric
     * key pair created outside Amazon Web Services Payment Cryptography by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>).
     * </p>
     * <p>
     * for symmetric and DUKPT encryption, Amazon Web Services Payment Cryptography supports <code>TDES</code> and
     * <code>AES</code> algorithms. For asymmetric encryption, Amazon Web Services Payment Cryptography supports
     * <code>RSA</code>. To encrypt using DUKPT, you must already have a DUKPT key in your account with
     * <code>KeyModesOfUse</code> set to <code>DeriveKey</code>, or you can generate a new DUKPT key by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a>.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DecryptData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetPublicKeyCertificate.html">
     * GetPublicCertificate</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ReEncryptData</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param encryptDataRequest
     * @return A Java Future containing the result of the EncryptData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.EncryptData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/EncryptData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EncryptDataResponse> encryptData(EncryptDataRequest encryptDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(encryptDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, encryptDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EncryptData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EncryptDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EncryptDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EncryptDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EncryptDataRequest, EncryptDataResponse>()
                            .withOperationName("EncryptData").withMarshaller(new EncryptDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(encryptDataRequest));
            CompletableFuture<EncryptDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates card-related validation data using algorithms such as Card Verification Values (CVV/CVV2), Dynamic Card
     * Verification Values (dCVV/dCVV2), or Card Security Codes (CSC). For more information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/generate-card-data.html">Generate card
     * data</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * This operation generates a CVV or CSC value that is printed on a payment credit or debit card during card
     * production. The CVV or CSC, PAN (Primary Account Number) and expiration date of the card are required to check
     * its validity during transaction processing. To begin this operation, a CVK (Card Verification Key) encryption key
     * is required. You can use <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a> or
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>
     * to establish a CVK within Amazon Web Services Payment Cryptography. The <code>KeyModesOfUse</code> should be set
     * to <code>Generate</code> and <code>Verify</code> for a CVK encryption key.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>VerifyCardValidationData</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param generateCardValidationDataRequest
     * @return A Java Future containing the result of the GenerateCardValidationData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.GenerateCardValidationData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/GenerateCardValidationData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GenerateCardValidationDataResponse> generateCardValidationData(
            GenerateCardValidationDataRequest generateCardValidationDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(generateCardValidationDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateCardValidationDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateCardValidationData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GenerateCardValidationDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GenerateCardValidationDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GenerateCardValidationDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GenerateCardValidationDataRequest, GenerateCardValidationDataResponse>()
                            .withOperationName("GenerateCardValidationData")
                            .withMarshaller(new GenerateCardValidationDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(generateCardValidationDataRequest));
            CompletableFuture<GenerateCardValidationDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates a Message Authentication Code (MAC) cryptogram within Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * You can use this operation when keys won't be shared but mutual data is present on both ends for validation. In
     * this case, known data values are used to generate a MAC on both ends for comparision without sending or receiving
     * data in ciphertext or plaintext. You can use this operation to generate a DUPKT, HMAC or EMV MAC by setting
     * generation attributes and algorithm to the associated values. The MAC generation encryption key must have valid
     * values for <code>KeyUsage</code> such as <code>TR31_M7_HMAC_KEY</code> for HMAC generation, and they key must
     * have <code>KeyModesOfUse</code> set to <code>Generate</code> and <code>Verify</code>.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>VerifyMac</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param generateMacRequest
     * @return A Java Future containing the result of the GenerateMac operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.GenerateMac
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/GenerateMac"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GenerateMacResponse> generateMac(GenerateMacRequest generateMacRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(generateMacRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateMacRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateMac");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GenerateMacResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GenerateMacResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GenerateMacResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GenerateMacRequest, GenerateMacResponse>()
                            .withOperationName("GenerateMac").withMarshaller(new GenerateMacRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(generateMacRequest));
            CompletableFuture<GenerateMacResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates pin-related data such as PIN, PIN Verification Value (PVV), PIN Block, and PIN Offset during new card
     * issuance or reissuance. For more information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/generate-pin-data.html">Generate PIN
     * data</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * PIN data is never transmitted in clear to or from Amazon Web Services Payment Cryptography. This operation
     * generates PIN, PVV, or PIN Offset and then encrypts it using Pin Encryption Key (PEK) to create an
     * <code>EncryptedPinBlock</code> for transmission from Amazon Web Services Payment Cryptography. This operation
     * uses a separate Pin Verification Key (PVK) for VISA PVV generation.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GenerateCardValidationData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TranslatePinData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>VerifyPinData</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param generatePinDataRequest
     * @return A Java Future containing the result of the GeneratePinData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.GeneratePinData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/GeneratePinData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GeneratePinDataResponse> generatePinData(GeneratePinDataRequest generatePinDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(generatePinDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generatePinDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GeneratePinData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GeneratePinDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GeneratePinDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GeneratePinDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GeneratePinDataRequest, GeneratePinDataResponse>()
                            .withOperationName("GeneratePinData")
                            .withMarshaller(new GeneratePinDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(generatePinDataRequest));
            CompletableFuture<GeneratePinDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Re-encrypt ciphertext using DUKPT, Symmetric and Asymmetric Data Encryption Keys.
     * </p>
     * <p>
     * You can either generate an encryption key within Amazon Web Services Payment Cryptography by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a> or
     * import your own encryption key by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>. The
     * <code>KeyArn</code> for use with this operation must be in a compatible key state with <code>KeyModesOfUse</code>
     * set to <code>Encrypt</code>. In asymmetric encryption, ciphertext is encrypted using public component (imported
     * by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>) of
     * the asymmetric key pair created outside of Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * For symmetric and DUKPT encryption, Amazon Web Services Payment Cryptography supports <code>TDES</code> and
     * <code>AES</code> algorithms. For asymmetric encryption, Amazon Web Services Payment Cryptography supports
     * <code>RSA</code>. To encrypt using DUKPT, a DUKPT key must already exist within your account with
     * <code>KeyModesOfUse</code> set to <code>DeriveKey</code> or a new DUKPT can be generated by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a>.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DecryptData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>EncryptData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetPublicKeyCertificate.html">
     * GetPublicCertificate</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param reEncryptDataRequest
     * @return A Java Future containing the result of the ReEncryptData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.ReEncryptData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/ReEncryptData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ReEncryptDataResponse> reEncryptData(ReEncryptDataRequest reEncryptDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(reEncryptDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reEncryptDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReEncryptData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReEncryptDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ReEncryptDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReEncryptDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReEncryptDataRequest, ReEncryptDataResponse>()
                            .withOperationName("ReEncryptData")
                            .withMarshaller(new ReEncryptDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(reEncryptDataRequest));
            CompletableFuture<ReEncryptDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Translates encrypted PIN block from and to ISO 9564 formats 0,1,3,4. For more information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/translate-pin-data.html">Translate PIN
     * data</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * PIN block translation involves changing the encrytion of PIN block from one encryption key to another encryption
     * key and changing PIN block format from one to another without PIN block data leaving Amazon Web Services Payment
     * Cryptography. The encryption key transformation can be from PEK (Pin Encryption Key) to BDK (Base Derivation Key)
     * for DUKPT or from BDK for DUKPT to PEK. Amazon Web Services Payment Cryptography supports <code>TDES</code> and
     * <code>AES</code> key derivation type for DUKPT tranlations. You can use this operation for P2PE (Point to Point
     * Encryption) use cases where the encryption keys should change but the processing system either does not need to,
     * or is not permitted to, decrypt the data.
     * </p>
     * <p>
     * The allowed combinations of PIN block format translations are guided by PCI. It is important to note that not all
     * encrypted PIN block formats (example, format 1) require PAN (Primary Account Number) as input. And as such, PIN
     * block format that requires PAN (example, formats 0,3,4) cannot be translated to a format (format 1) that does not
     * require a PAN for generation.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <note>
     * <p>
     * At this time, Amazon Web Services Payment Cryptography does not support translations to PIN format 4.
     * </p>
     * </note>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GeneratePinData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>VerifyPinData</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param translatePinDataRequest
     * @return A Java Future containing the result of the TranslatePinData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.TranslatePinData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/TranslatePinData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TranslatePinDataResponse> translatePinData(TranslatePinDataRequest translatePinDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(translatePinDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, translatePinDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TranslatePinData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TranslatePinDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TranslatePinDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TranslatePinDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TranslatePinDataRequest, TranslatePinDataResponse>()
                            .withOperationName("TranslatePinData")
                            .withMarshaller(new TranslatePinDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(translatePinDataRequest));
            CompletableFuture<TranslatePinDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Verifies Authorization Request Cryptogram (ARQC) for a EMV chip payment card authorization. For more information,
     * see <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/userguide/data-operations.verifyauthrequestcryptogram.html"
     * >Verify auth request cryptogram</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * ARQC generation is done outside of Amazon Web Services Payment Cryptography and is typically generated on a point
     * of sale terminal for an EMV chip card to obtain payment authorization during transaction time. For ARQC
     * verification, you must first import the ARQC generated outside of Amazon Web Services Payment Cryptography by
     * calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>.
     * This operation uses the imported ARQC and an major encryption key (DUKPT) created by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a> to
     * either provide a boolean ARQC verification result or provide an APRC (Authorization Response Cryptogram) response
     * using Method 1 or Method 2. The <code>ARPC_METHOD_1</code> uses <code>AuthResponseCode</code> to generate ARPC
     * and <code>ARPC_METHOD_2</code> uses <code>CardStatusUpdate</code> to generate ARPC.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>VerifyCardValidationData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>VerifyPinData</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param verifyAuthRequestCryptogramRequest
     * @return A Java Future containing the result of the VerifyAuthRequestCryptogram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>VerificationFailedException This request failed verification.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.VerifyAuthRequestCryptogram
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/VerifyAuthRequestCryptogram"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<VerifyAuthRequestCryptogramResponse> verifyAuthRequestCryptogram(
            VerifyAuthRequestCryptogramRequest verifyAuthRequestCryptogramRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(verifyAuthRequestCryptogramRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, verifyAuthRequestCryptogramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VerifyAuthRequestCryptogram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<VerifyAuthRequestCryptogramResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, VerifyAuthRequestCryptogramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<VerifyAuthRequestCryptogramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<VerifyAuthRequestCryptogramRequest, VerifyAuthRequestCryptogramResponse>()
                            .withOperationName("VerifyAuthRequestCryptogram")
                            .withMarshaller(new VerifyAuthRequestCryptogramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(verifyAuthRequestCryptogramRequest));
            CompletableFuture<VerifyAuthRequestCryptogramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Verifies card-related validation data using algorithms such as Card Verification Values (CVV/CVV2), Dynamic Card
     * Verification Values (dCVV/dCVV2) and Card Security Codes (CSC). For more information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/verify-card-data.html">Verify card
     * data</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * This operation validates the CVV or CSC codes that is printed on a payment credit or debit card during card
     * payment transaction. The input values are typically provided as part of an inbound transaction to an issuer or
     * supporting platform partner. Amazon Web Services Payment Cryptography uses CVV or CSC, PAN (Primary Account
     * Number) and expiration date of the card to check its validity during transaction processing. In this operation,
     * the CVK (Card Verification Key) encryption key for use with card data verification is same as the one in used for
     * <a>GenerateCardValidationData</a>.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GenerateCardValidationData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>VerifyAuthRequestCryptogram</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>VerifyPinData</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param verifyCardValidationDataRequest
     * @return A Java Future containing the result of the VerifyCardValidationData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>VerificationFailedException This request failed verification.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.VerifyCardValidationData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/VerifyCardValidationData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<VerifyCardValidationDataResponse> verifyCardValidationData(
            VerifyCardValidationDataRequest verifyCardValidationDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(verifyCardValidationDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, verifyCardValidationDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VerifyCardValidationData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<VerifyCardValidationDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, VerifyCardValidationDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<VerifyCardValidationDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<VerifyCardValidationDataRequest, VerifyCardValidationDataResponse>()
                            .withOperationName("VerifyCardValidationData")
                            .withMarshaller(new VerifyCardValidationDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(verifyCardValidationDataRequest));
            CompletableFuture<VerifyCardValidationDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Verifies a Message Authentication Code (MAC).
     * </p>
     * <p>
     * You can use this operation when keys won't be shared but mutual data is present on both ends for validation. In
     * this case, known data values are used to generate a MAC on both ends for verification without sending or
     * receiving data in ciphertext or plaintext. You can use this operation to verify a DUPKT, HMAC or EMV MAC by
     * setting generation attributes and algorithm to the associated values. Use the same encryption key for MAC
     * verification as you use for <a>GenerateMac</a>.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GenerateMac</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param verifyMacRequest
     * @return A Java Future containing the result of the VerifyMac operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>VerificationFailedException This request failed verification.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.VerifyMac
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/VerifyMac"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<VerifyMacResponse> verifyMac(VerifyMacRequest verifyMacRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(verifyMacRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, verifyMacRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VerifyMac");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<VerifyMacResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    VerifyMacResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<VerifyMacResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<VerifyMacRequest, VerifyMacResponse>().withOperationName("VerifyMac")
                            .withMarshaller(new VerifyMacRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(verifyMacRequest));
            CompletableFuture<VerifyMacResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Verifies pin-related data such as PIN and PIN Offset using algorithms including VISA PVV and IBM3624. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/verify-pin-data.html">Verify PIN data</a>
     * in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * This operation verifies PIN data for user payment card. A card holder PIN data is never transmitted in clear to
     * or from Amazon Web Services Payment Cryptography. This operation uses PIN Verification Key (PVK) for PIN or PIN
     * Offset generation and then encrypts it using PIN Encryption Key (PEK) to create an <code>EncryptedPinBlock</code>
     * for transmission from Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * For information about valid keys for this operation, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> and <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/crypto-ops-validkeys-ops.html">Key types
     * for specific data operations</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GeneratePinData</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TranslatePinData</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param verifyPinDataRequest
     * @return A Java Future containing the result of the VerifyPinData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>VerificationFailedException This request failed verification.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyDataAsyncClient.VerifyPinData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-data-2022-02-03/VerifyPinData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<VerifyPinDataResponse> verifyPinData(VerifyPinDataRequest verifyPinDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(verifyPinDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, verifyPinDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VerifyPinData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<VerifyPinDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    VerifyPinDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<VerifyPinDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<VerifyPinDataRequest, VerifyPinDataResponse>()
                            .withOperationName("VerifyPinData")
                            .withMarshaller(new VerifyPinDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(verifyPinDataRequest));
            CompletableFuture<VerifyPinDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PaymentCryptographyDataServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PaymentCryptographyDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("VerificationFailedException")
                                .exceptionBuilderSupplier(VerificationFailedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        PaymentCryptographyDataServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = PaymentCryptographyDataServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
